/* Vt.c -- Voltage as function of time in fetchex data
 *
 * Vstream(FILE *, char *, bufsiz) -- identify the open stream from
 *	which to read voltages
 * Vend() -- finish processing and clean up
 * Vinfo() -- return information about record
 * V(TIME) -- return voltage at indicated time (macro)
 * fV(TIME) -- return voltage at indicated time (function)
 *
 * The following globals are defined to make the macro V work; avoid
 * conflict:
 *
 * VOLTAGE	*V_infbuf;
 * int	V_ibnv;
 * TIME	V_ibt0;
 * TIME	V_lastt;
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "config.h"
#include "Vt.h"

#ifdef	__STDC__
VOIDST	ealloc(size_t);
VOIDST	erealloc(VOIDST, size_t);
#else	/* __STDC__ */
VOIDST	ealloc();
VOIDST	erealloc();
#endif	/* __STDC__ */

typedef struct FILTYP {			/* format information		*/
    char	*magic;			/* magic number			*/
    int		maglen;			/* magic number length		*/
    char	*name;			/* name of this format		*/
    long	hdrsize;		/* length of header		*/
    int		sb;			/* true if need to swab bytes	*/
    int		valsize;		/* one value in bytes		*/
    void	(*hdrinfo)(char *, V_INFO *); /* extract info from header */
}		FILTYP;

static	void	fexhdr(char *, V_INFO *);
static	void	abhdr(char *, V_INFO *);
static	void	ab2hdr(char *, V_INFO *);
#define	AB_MAGLEN	28
#define	AB2_MAGLEN	27
static FILTYP	filtyps[] = {
    {"accbin format #2(header=1k)", AB2_MAGLEN, "accbin format #2", 1000, FALSE, 2, ab2hdr},
    {"accbin format #1(header=.5k)", AB_MAGLEN, "accbin format #1", 500, FALSE, 2, abhdr},
    {"\0\0\040\101", 4, "FETCHEX", 1024, TRUE, 2, fexhdr}
};
#define	NTYPES	((sizeof(filtyps)/sizeof(FILTYP)))
#define	MAGMAX	64			/* max magic number length	*/

#define	MIN(a, b)	(((a) < (b)) ? (a) : (b))
#define	MAX(a, b)	(((a) > (b)) ? (a) : (b))
#define	MINBUF	1024
#define	ELEN	80			/* max error message length	*/

static	int	vread(VOLTAGE *, int);
static	int	readhdr(void);

static	char	*ifn = "Standard Input";/* input file name		*/
static	FILE	*infile = stdin;	/* input stream			*/
static	int	seekable = FALSE;	/* TRUE if fseek will work	*/
static	int	ibsiz = 0;		/* buffer size			*/
static	char	*V_header = NULL;	/* store header here		*/
static	char	*format;		/* name of file format		*/
static	TIME	hdrsize;		/* size of whole header		*/
					/* extract header info		*/
static	void	(*hdrinfo)(char *, V_INFO *);
static	int	sb;			/* whether to swap bytes	*/
static	int	valsize;		/* bytes/value in file		*/
VOLTAGE	*V_infbuf = NULL;		/* input buffer			*/
int	V_ibnv = 0;			/* # vals actually in buffer	*/
TIME	V_ibt0 = 0;			/* time of first value		*/
TIME	V_lastt = 0;			/* last time read		*/
static	char	emsg[ELEN] = "";	/* error message buffer		*/
char	*V_errmsg = emsg;		/* message stored here on error	*/

/* Vstream -- identify open stream from which to read voltages
 *
 *  FILE	*stream;
 *  char	*filename;
 *  unsigned	bufsiz;
 *  if (EOF == Vstream(stream, filename, bufsiz)) error...
 */
int
Vstream(s, fn, bs)
FILE	*s;
char	*fn;
unsigned bs;
{
    int			ret;

    Vend();
    ifn = fn;
    infile = s;
    seekable = (0 == fseek(s, 0L, 0));
    ibsiz = bs;
    if (ibsiz < MINBUF) ibsiz = MINBUF;
    V_infbuf = (VOLTAGE *) ealloc(ibsiz * sizeof(VOLTAGE));
    if (EOF == readhdr()) return(EOF);
    V_ibt0 = V_ibnv = 0;
    return(0);
}

/* Vend -- clean up							*/
void
Vend()
{
    if (NULL != V_infbuf) free(V_infbuf);
    V_infbuf = NULL;
    if (NULL != V_header) free(V_header);
    V_header = NULL;
}

/* Vinfo -- return file info
 *
 * V_INFO	*infop;
 * infop = Vinfo();
 *
 * Vinfo returns a pointer to a V_INFO structure containing
 * information about the currently open file.  The structure is in an
 * internal static area and should not be modified by the caller.
 */
V_INFO *
Vinfo()
{
    static V_INFO	info;

    if ((NULL == hdrinfo) || (NULL == V_header)) return(NULL);
    (*hdrinfo)(V_header, &info);
    return(&info);
}

/* fV -- return value at time t, handling buffers if necessary		*/
VOLTAGE
fV(
TIME		t
) {
    register int	n;
    TIME		nt0;
    TIME		tread;
    int			nread;
    VOLTAGE		*where;

    if ((t >= V_ibt0) && (t < V_ibt0+V_ibnv))
        return(V_infbuf[t-V_ibt0]);	/* already there		*/
    if (t < 0) return(VEOF);
    if ((t < V_ibt0) && !seekable) {
	sprintf(emsg, "can't go backward on %s", ifn);
	V_errmsg = emsg;
	return(VEOF);
    }
    nt0 = t - (ibsiz-1)/2;		/* center t in new buffer	*/
    if (nt0 < 0) nt0 = 0;
    tread = nt0;
    nread = ibsiz;
    where = V_infbuf;
					/* first part already there	*/
    if ((tread >= V_ibt0) && (tread < V_ibt0+V_ibnv)) {
	n = V_ibt0 + V_ibnv - tread;
	bcopy(
	    &V_infbuf[tread-V_ibt0],
	    where,
	    n * sizeof(VOLTAGE)
	);
	V_ibt0 = tread;
	V_ibnv = n;
	tread += n;
	nread -= n;
	where += n;
	if (
	    seekable &&
	    EOF == fseek(infile, (long) (hdrsize + tread*valsize), 0)
	) {
	    sprintf(emsg, "unable to find point %ld on %s", (long) tread, ifn);
	    V_errmsg = emsg;
	    return(VEOF);
	}
	n = vread(where, nread);
	V_ibnv += n;
    }
					/* last part already there	 */
    else if ((tread < V_ibt0) && (tread + nread > V_ibt0)) {
	n = tread + nread - V_ibt0;
	bcopy(
	    V_infbuf,
	    where + V_ibt0 - tread,
	    n * sizeof(VOLTAGE)
	);
	nread -= n;
	if (EOF == fseek(infile, (long) (hdrsize + tread*valsize), 0)) {
	    sprintf(emsg, "unable to find point %ld on %s", (long) tread, ifn);
	    V_errmsg = emsg;
	    return(VEOF);
	}
	n = vread(where, nread);
	V_ibt0 = tread;
	V_ibnv = (n == nread) ? ibsiz : n;
    }
    else if (seekable) {		/* no overlap, but seekable	*/
	if (EOF == fseek(infile, (long) (hdrsize + tread*valsize), 0)) {
	    sprintf(emsg, "unable to find point %ld on %s", (long) tread, ifn);
	    V_errmsg = emsg;
	    return(VEOF);
	}
	V_ibnv = vread(where, nread);
	V_ibt0 = tread;
    }
    else {				/* no overlap, non-seekable	*/
	while(V_ibnv <= nt0-V_ibt0) {	/* seek forward if necessary	*/
	    V_ibt0 += V_ibnv;
	    V_ibnv = vread(V_infbuf, ibsiz);
	    if (0 == V_ibnv) return(VEOF);
	}
	if (nt0 > V_ibt0) {		/* shift any useful old values	*/
	    V_ibnv -= nt0-V_ibt0;
	    bcopy(
		&V_infbuf[nt0-V_ibt0],
		&V_infbuf[0],
		V_ibnv * sizeof(VOLTAGE)
	    );
	    V_ibt0 = nt0;
	}
	if (ibsiz > V_ibnv) {
	    V_ibnv += vread(&V_infbuf[V_ibnv], ibsiz-V_ibnv);
	}
    }
    return(
	((t >= V_ibt0) && (t < V_ibt0+V_ibnv)) ? V_infbuf[t-V_ibt0] : VEOF
    );
}

/* vread -- read values into memory, return number actually read	*/
int
vread(
VOLTAGE		*where,
int		howmany
) {
    register int	i;
    register int	c1, c2;

    for(i=0; i<howmany; i++) {
	if (sb) {
	    if (
		(EOF == (c1 = getc(infile))) ||
		(EOF == (c2 = getc(infile)))
	    ) break;
	}
	else {
	    if (
		(EOF == (c2 = getc(infile))) ||
		(EOF == (c1 = getc(infile)))
	    ) break;
	}
	/*
	 * extend sign of c2.  This should work even on a non-two's
	 * complement machine, regardless of word size.  Note that the
	 * values in the file are two's complement, regardless of native
	 * format.
	 */
	c2 &= 0xff;
	if (0 != (0x80 & c2)) c2 -= 0x100;
	*where++ = (0xff & c1) + (0x100 * c2);
    }
    return(i);
}

/* read the FETCHEX data file header
 *
 * Read the header and store it in space allocated from the heap,
 * pointed to by V_header.  Returns EOF if any problem occurs;
 */
int
readhdr()
{
    register int	i;
    int			n;

    V_header = ealloc(MAGMAX);
    if (0 >= (n = fread(V_header, 1, MAGMAX, infile))) {
	sprintf(emsg, "unable to read %s header", ifn);
	V_errmsg = emsg;
	return(EOF);
    }
    for(i=0; i<NTYPES; i++) {
	if (
	    (n >= filtyps[i].maglen) &&
	    (0 == bcmp(filtyps[i].magic, V_header, filtyps[i].maglen))
	) break;
    }
    if (i >= NTYPES) {
	sprintf(emsg, "file %s is of unknown format", ifn);
	V_errmsg = emsg;
	return(EOF);
    }
    format = filtyps[i].name;
    hdrsize = filtyps[i].hdrsize;
    hdrinfo = filtyps[i].hdrinfo;
    sb = filtyps[i].sb;
    valsize = filtyps[i].valsize;
    V_header = erealloc(V_header, hdrsize);
    if (1 != fread(V_header+n, hdrsize-n, 1, infile)) {
	sprintf(emsg, "unable to read %s header", ifn);
	V_errmsg = emsg;
	return(EOF);
    }
    V_ibt0 = 0;
    return(0);
}

/* routines to extract file info					*/
static void
fexhdr(hdr, info)
char	*hdr;
V_INFO	*info;
{
    info->V_format = format;
    info->V_comment = "";
    info->V_np = (filelen(infile) - hdrsize) / valsize;
    info->V_tlo = 0;
    info->V_thi = info->V_np;
    info->V_vlo = -0x1000;
    info->V_vhi = 0xfff;
    info->V_tscale = 1.0;
    info->V_vscale = 1.0;
}

/*
 * Wayne Davis's accbin format #1
 */
#define	AB_VLO	-0x800			/* assume a 12-bit ADC		*/
#define	AB_VHI	0x7ff
#define	AB_SGL	4			/* size of LabView SGL type	*/
#define	AB_I32	4			/* I32 type			*/
#define	AB_U16	2			/* U16 type			*/

static int
ab_I32(s)
char	*s;
{
    union {
	char	c[4];
	long	i;
    }			u;

#ifdef	WORDS_BIGENDIAN
    u.c[0] = s[0];
    u.c[1] = s[1];
    u.c[2] = s[2];
    u.c[3] = s[3];
#else	/* WORDS_BIGENDIAN */
    u.c[0] = s[1];			/* ****************************	*/
    u.c[1] = s[0];			/*           UNTESTED		*/
    u.c[2] = s[3];			/*	       CODE		*/
    u.c[3] = s[2];			/* ****************************	*/
#endif	/* WORDS_BIGENDIAN */
    return(u.i);
}

static float
ab_sgl(s)
char	*s;
{
    union {
	char	c[4];
	float	f;
    }			u;

#ifdef	WORDS_BIGENDIAN
    u.c[0] = s[0];
    u.c[1] = s[1];
    u.c[2] = s[2];
    u.c[3] = s[3];
#else	/* WORDS_BIGENDIAN */
    u.c[0] = s[1];			/* ****************************	*/
    u.c[1] = s[0];			/*           UNTESTED		*/
    u.c[2] = s[3];			/*	       CODE		*/
    u.c[3] = s[2];			/* ****************************	*/
#endif	/* WORDS_BIGENDIAN */
    return(u.f);
}

static void
abhdr(hdr, info)
char	*hdr;
V_INFO	*info;
{
    char		*t = hdr;
    char		*s;
    int			c;

    info->V_format = format;
    info->V_np = (filelen(infile) - hdrsize) / valsize;
    info->V_tlo = 0;
    info->V_thi = info->V_np;
    info->V_vlo = AB_VLO;
    info->V_vhi = AB_VHI;
    t += AB_MAGLEN;			/* skip magic number		*/
    t += ab_I32(t) + AB_I32;		/* skip channel string		*/
    s = t + AB_I32;			/* points to hardware config	*/
    t = s + ab_I32(t);			/* skip hardware config		*/
    s += AB_I32;			/* skip junk			*/
    s += ab_I32(s) + AB_I32;		/* skip channel string		*/
    s += 1 +				/* what the Hell is this?	*/
         AB_SGL +			/* skip upper input limit ...	*/
         AB_SGL +			/* ...lower input limit...	*/
	 AB_SGL +			/* ...range...			*/
	 AB_U16 +			/* ...polarity...		*/
	 AB_SGL +			/* ...gain...			*/
	 AB_U16 +			/* ...coupling...		*/
	 AB_U16;			/* ...input mode...		*/
    info->V_vscale = ab_sgl(s);		/* multiplier, at last		*/
					/* (t already skipped offset)	*/
    info->V_tscale = 1.0 / ab_sgl(t);	/* clock frequency		*/
    t += AB_SGL + AB_SGL;		/* skip clock info		*/
    s = hdr + hdrsize - 1;		/* strip junk from comment	*/
    for(c=*s; (s >= t) && (*s == c); s--) *s = '\0';
    info->V_comment = t;
}

#define	AB2_T0		(57)
#define	AB2_VMUL	(69)
#define	AB2_CLK		(637)
#define	AB2_COM		(645)
static void
ab2hdr(hdr, info)
char	*hdr;
V_INFO	*info;
{
    info->V_format = format;
    info->V_np = (filelen(infile) - hdrsize) / valsize;
    info->V_tlo = 0;
    info->V_thi = info->V_np;
    info->V_vlo = AB_VLO;
    info->V_vhi = AB_VHI;
    info->V_vscale = ab_sgl(hdr + AB2_VMUL);
    info->V_tscale = 1.0 / ab_sgl(hdr + AB2_CLK);
    info->V_comment = hdr + AB2_COM;
    hdr[hdrsize-1] = '\0';		/* (Just in case)		*/
}
