
#include "xplore.h"
#include "interface.h"

#include "callbacks.h"
#include "curdir.h"
#include "devmount.h"
#include "error.h"
#include "fileops.h"
#include "ftype.h"
#include "icons.h"
#include "interface.h"
#include "menus.h"
#include "parse.h"
#include "signals.h"
#include "util.h"

#include <X11/cursorfont.h>
#include <Xm/XmAll.h>
#include <Xm/TransferP.h>
#include <Tabs.h>

#if (XtSpecificationRelease > 4)
#ifndef _NO_XMU

/* support for editres protocol */
#include <X11/Xmu/Editres.h>

#endif
#endif

/* application context, shell and display */

XtAppContext app;
Widget app_shell;
Display *display;

/* translations and actions */

static char tree_trans_s[] = "#override <Btn2Down>: TreeStartDrag()";
static char file_trans_s[] = "#override <Btn2Down>: FileStartDrag()";

static XtTranslations tree_trans, file_trans;

static void TreeStartDrag(Widget, XEvent*);
static void FileStartDrag(Widget, XEvent*);

static XtActionsRec drag_actions[] = {
  { "TreeStartDrag", (XtActionProc)TreeStartDrag },
  { "FileStartDrag", (XtActionProc)FileStartDrag },
};

/* widgets */

Widget main_window, menubar, form, shelf_tabs, frame, work_area, file_view,
file_pane, file_scroll, tree_pane, tree_scroll, shelf_pane, shelf_scroll,
status_line, message_frame, message, indicators_frame, indicators, curgadget;

Widget file_pulldown, view_pulldown, options_pulldown, help_pulldown;
Widget command_pulldown, shelf_pulldown, file_popup;

Widget file_button, view_button, options_button, help_button;
Widget command_button, shelf_button;

Widget open_button, mkdir_button, chdir_button, parent_button,
home_button, rename_button, move_button, copy_button, link_button,
delete_button, properties_button, filter_button,
select_button, select_all_button, invert_all_button, quit_button;

Widget *command_menu_button = NULL;
int command_menu_size = 0;

Widget shelf_open_button, shelf_rename_button, shelf_move_button,
shelf_copy_button, shelf_link_button, shelf_delete_button,
shelf_properties_button, shelf_select_button, shelf_select_all_button,
shelf_invert_all_button;

Widget icons_button, text_button, tree_pane_button, file_pane_button,
shelf_pane_button,
sort_by_name_button, sort_by_size_button, sort_by_date_button,
reverse_button, dirs_first_button,
show_hidden_button, show_parent_dir_button,
show_files_button, show_dirs_button,
rescan_files_button, magic_headers_button,
update_button, reread_button, unmount_button, unmount_all_button;

Widget absolute_paths_button, target_dir_button, push_dir_button,
echo_commands_button, check_mounts_button, backups_button,
auto_updates_button,
confirm_drop_button, confirm_move_button, confirm_copy_button,
confirm_link_button, confirm_delete_button, confirm_deldir_button,
confirm_delfile_button, confirm_overwrite_button, confirm_quit_button,
save_setup_button, reload_config_button;

Widget about_button;

Cursor wait_cursor;

/* resource values: **********************************************************/

String progname;
char magicfile[MAXPATHLEN+1], configfile[MAXPATHLEN+1], resfile[MAXPATHLEN+1];
String rescmd;
String curshelf;
String cpp_options;
String iconpath;
String sh_list;

unsigned char keyboardFocusPolicy;

char rootdirname[MAXPATHLEN+1], curdirname[MAXPATHLEN+1],
shelfdirname[MAXPATHLEN+1];

Boolean show_getwd;

int update_time;
Boolean updates;

ViewType viewtype;
Boolean filepane, treepane, shelfpane;

Boolean absolute, dirtarget, pushdir;

Boolean
confirm_drop,
confirm_move,
confirm_copy,
confirm_link,
confirm_delete,
confirm_deldir,
confirm_delfile,
confirm_overwrt,
confirm_quit;

/* default options for directories */

static int stdoptions =
CHECK_FILES|SORT_BY_NAME|INCLUDE_UPDIR|INCLUDE_FILES|INCLUDE_DIRS|DIRS_FIRST;

static int shelfoptions =
CHECK_FILES|SORT_BY_NAME|INCLUDE_HIDDEN|INCLUDE_FILES|INCLUDE_DIRS|DIRS_FIRST;

/* resource data */

typedef enum _SortType {
    ByName, BySize, ByDate,
} SortType;

typedef struct _ResData {
  String rootdir, curdir, shelfdir, curshelf;
  String iconpath;
  String magicfile, configfile, resfile, rescmd;
  String cppoptions;
  String bourneShells;
  String metachars;
  Boolean getwd;
  Boolean updates;
  int update, multiclick;
  Boolean file, tree, shelf;
  ViewType view;
  Boolean updir, hidden, files, dirs;
  SortType sort;
  Boolean reverse, dirsfirst;
  Boolean rescan, magic;
  Boolean check;
  Boolean absolute, dirtarget, pushdir, echo, backups;
  Boolean drop, move, copy, link, delete, deldir, delfile, overwrt, quit;
} ResData;

static ResData resdata;

/* resource converters */

static void StringToViewType(XrmValue *args, Cardinal *n_args,
			     XrmValue *fromVal, XrmValue *toVal)
{
  static ViewType t;

  if (!(strcmp(fromVal->addr, "Icons")))
    t = IconView;
  else if (!(strcmp(fromVal->addr, "Text")))
    t = TextView;
  else {
    XtStringConversionWarning(fromVal->addr, "ViewType");
    return;
  }
  
  toVal->addr = (caddr_t) &t;
  toVal->size = sizeof(ViewType);
}

static void StringToSortType(XrmValue *args, Cardinal *n_args,
			     XrmValue *fromVal, XrmValue *toVal)
{
  static SortType t;

  if (!(strcmp(fromVal->addr, "ByName")))
    t = ByName;
  else if (!(strcmp(fromVal->addr, "BySize")))
    t = BySize;
  else if (!(strcmp(fromVal->addr, "ByDate")))
    t = ByDate;
  else {
    XtStringConversionWarning(fromVal->addr, "SortType");
    return;
  }
  
  toVal->addr = (caddr_t) &t;
  toVal->size = sizeof(ViewType);
}

/* command line options */

static XrmOptionDescRec options[] = {
  { "-f", ".configfile", XrmoptionSepArg, NULL },
  { "-m", ".magicfile", XrmoptionSepArg, NULL },
  { "-o", ".cppoptions", XrmoptionSepArg, NULL },
  { "-r", ".resfile", XrmoptionSepArg, NULL },
  { "-s", ".curshelf", XrmoptionSepArg, NULL },
};

/* fallback resources */

static String fallback_resources[] = {
#include "defaults.h"
  NULL
};

/* application-specific resource values with hard-coded defaults */

static XtResource resources[] = {
  { "iconpath", "Iconpath", XmRString, sizeof(String),
    XtOffsetOf(ResData, iconpath), XmRString,
    "/usr/lib/X11/xplore/icons" },
  { "rootdir", "Rootdir", XmRString, sizeof(String),
    XtOffsetOf(ResData, rootdir), XmRString,
    "/" },
  { "curdir", "Curdir", XmRString, sizeof(String),
    XtOffsetOf(ResData, curdir), XmRString,
    "." },
  { "shelfdir", "Shelfdir", XmRString, sizeof(String),
    XtOffsetOf(ResData, shelfdir), XmRString,
    "~/shelf" },
  { "curshelf", "Curshelf", XmRString, sizeof(String),
    XtOffsetOf(ResData, curshelf), XmRString,
    "" },
  { "magicfile", "MagicFile", XmRString, sizeof(String),
    XtOffsetOf(ResData, magicfile), XmRString,
    "~/.magic" },
  { "configfile", "ConfigFile", XmRString, sizeof(String),
    XtOffsetOf(ResData, configfile), XmRString,
    "~/.xplorerc" },
  { "resfile", "ResFile", XmRString, sizeof(String),
    XtOffsetOf(ResData, resfile), XmRString,
    "~/.Xresources" },
  { "rescmd", "ResCmd", XmRString, sizeof(String),
    XtOffsetOf(ResData, rescmd), XmRString,
    "xrdb -merge %s" },
  { "cppoptions", "CppOptions", XmRString, sizeof(String),
    XtOffsetOf(ResData, cppoptions), XmRString,
    "-I/usr/lib/X11/xplore" },
  { "bourneShells", "BourneShells", XmRString, sizeof(String),
    XtOffsetOf(ResData, bourneShells), XmRString,
    "AUTO" },
  { "metachars", "Metachars", XmRString, sizeof(String),
    XtOffsetOf(ResData, metachars), XmRString,
    NULL },
  { "getwd", "Getwd", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, getwd), XmRImmediate,
    (XtPointer) True },
  { "updates", "Updates", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, updates), XmRImmediate,
    (XtPointer) True },
  { "update", "Update", XmRInt, sizeof(int),
    XtOffsetOf(ResData, update), XmRImmediate,
    (XtPointer) 10000 },
  { "multiclick", "Multiclick", XmRInt, sizeof(int),
    XtOffsetOf(ResData, multiclick), XmRImmediate,
    (XtPointer) 300 },
  { "tree", "Panes", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, tree), XmRImmediate,
    (XtPointer) True },
  { "file", "Panes", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, file), XmRImmediate,
    (XtPointer) True },
  { "shelf", "Panes", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, shelf), XmRImmediate,
    (XtPointer) True },
  { "view", "View", "ViewType", sizeof(ViewType),
    XtOffsetOf(ResData, view), XmRImmediate,
    (XtPointer) IconView },
  { "updir", "Items", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, updir), XmRImmediate,
    (XtPointer) True },
  { "hidden", "Items", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, hidden), XmRImmediate,
    (XtPointer) False },
  { "files", "Items", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, files), XmRImmediate,
    (XtPointer) True },
  { "dirs", "Items", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, dirs), XmRImmediate,
    (XtPointer) True },
  { "sort", "Sort", "SortType", sizeof(SortType),
    XtOffsetOf(ResData, sort), XmRImmediate,
    (XtPointer) ByName },
  { "reverse", "Reverse", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, reverse), XmRImmediate,
    (XtPointer) False },
  { "dirsfirst", "DirsFirst", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, dirsfirst), XmRImmediate,
    (XtPointer) True },
  { "rescan", "Rescan", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, rescan), XmRImmediate,
    (XtPointer) True },
  { "magic", "Magic", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, magic), XmRImmediate,
    (XtPointer) True },
  { "check", "Check", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, check), XmRImmediate,
    (XtPointer) True },
  { "absolute", "Absolute", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, absolute), XmRImmediate,
    (XtPointer) False },
  { "dirtarget", "Dirtarget", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, dirtarget), XmRImmediate,
    (XtPointer) False },
  { "pushdir", "Pushdir", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, pushdir), XmRImmediate,
    (XtPointer) True },
  { "echo", "Echo", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, echo), XmRImmediate,
    (XtPointer) False },
  { "backups", "Backups", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, backups), XmRImmediate,
    (XtPointer) True },
  { "drop", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, drop), XmRImmediate,
    (XtPointer) True },
  { "move", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, move), XmRImmediate,
    (XtPointer) True },
  { "copy", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, copy), XmRImmediate,
    (XtPointer) True },
  { "link", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, link), XmRImmediate,
    (XtPointer) True },
  { "delete", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, delete), XmRImmediate,
    (XtPointer) True },
  { "deldir", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, deldir), XmRImmediate,
    (XtPointer) True },
  { "delfile", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, delfile), XmRImmediate,
    (XtPointer) True },
  { "overwrt", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, overwrt), XmRImmediate,
    (XtPointer) True },
  { "quit", "Confirm", XmRBoolean, sizeof(Boolean),
    XtOffsetOf(ResData, quit), XmRImmediate,
    (XtPointer) True },
};

/* menu descriptions *******************************************************/

enum {
    MENU_POPUP, MENU_PULLDOWN
};

enum {
    SEPARATOR_ID, PUSH_BUTTON_ID, CHECK_BUTTON_ID, RADIO_BUTTON_ID,
    CASCADE_BUTTON_ID
};

typedef struct _MenuItem {
  int id;
  String label;
  String mnemonic;
  String accelerator;
  String accelerator_text;
  struct _MenuItem *submenu;
  int submenu_size;
  XtCallbackProc callback, unmap_callback;
  XtPointer data;
  String button_name;
  Widget *button;
  String pulldown_name;
  Widget *pulldown;
} MenuItem, MenuDesc[];

#define SEPARATOR \
{ \
    SEPARATOR_ID, NULL, NULL, NULL, NULL, NULL, 0, \
    (XtCallbackProc)NULL, (XtCallbackProc)NULL, NULL, NULL, NULL, NULL, NULL \
}

#define PUSH_BUTTON(lbl,mnem,accel,accel_text,cb,data,but_name,but) \
{ \
    PUSH_BUTTON_ID, lbl, mnem, accel, accel_text, NULL, 0, \
    (XtCallbackProc)cb, (XtCallbackProc)NULL, data, but_name, but, NULL, NULL \
}

#define CHECK_BUTTON(lbl,mnem,accel,accel_text,cb,data,but_name,but) \
{ \
    CHECK_BUTTON_ID, lbl, mnem, accel, accel_text, NULL, 0, \
    (XtCallbackProc)cb, (XtCallbackProc)NULL, data, but_name, but, NULL, NULL \
}

#define RADIO_BUTTON(lbl,mnem,accel,accel_text,cb,data,but_name,but) \
{ \
    RADIO_BUTTON_ID, lbl, mnem, accel, accel_text, NULL, 0, \
    (XtCallbackProc)cb, (XtCallbackProc)NULL, data, but_name, but, NULL, NULL \
}

#define CASCADE_BUTTON(lbl,mnem,accel,accel_text,subm,subm_sz,map_cb,unmap_cb,\
but_name,but,pdn_name,pdn) \
{ \
    CASCADE_BUTTON_ID, lbl, mnem, accel, accel_text, subm, subm_sz, \
    (XtCallbackProc)map_cb, (XtCallbackProc)unmap_cb, \
    NULL, but_name, but, pdn_name, pdn \
}

#define COMMAND 2

static MenuDesc file_menu_desc = {
  PUSH_BUTTON("Open", NULL, NULL, NULL, OpenCB, NULL,
	      "Open", &open_button),
  SEPARATOR,
  CASCADE_BUTTON("Command", NULL, NULL, NULL, NULL, 0, NULL, NULL,
		 "Command", &command_button,
		 "CommandPulldown", &command_pulldown),
  SEPARATOR,
  PUSH_BUTTON("Mkdir...", NULL, NULL, NULL, MkdirCB, NULL,
	      "Mkdir", &mkdir_button),
  PUSH_BUTTON("Chdir...", NULL, NULL, NULL, ChdirCB, NULL,
	      "Chdir", &chdir_button),
  PUSH_BUTTON("Parent", NULL, NULL, NULL, ParentCB, NULL,
	      "Parent", &parent_button),
  PUSH_BUTTON("Home", NULL, NULL, NULL, HomeCB, NULL,
	      "Home", &home_button),
  SEPARATOR,
  PUSH_BUTTON("Rename...", NULL, NULL, NULL, RenameCB, NULL,
	      "Rename", &rename_button),
  PUSH_BUTTON("Move...", NULL, NULL, NULL, MoveCB, NULL,
	      "Move", &move_button),
  PUSH_BUTTON("Copy...", NULL, NULL, NULL, CopyCB, NULL,
	      "Copy", &copy_button),
  PUSH_BUTTON("Link...", NULL, NULL, NULL, LinkCB, NULL,
	      "Link", &link_button),
  PUSH_BUTTON("Delete", NULL, NULL, NULL, DeleteCB, NULL,
	      "Delete", &delete_button),
  SEPARATOR,
  PUSH_BUTTON("Properties...", NULL, NULL, NULL, PropertiesCB, NULL,
	      "Properties", &properties_button),
  SEPARATOR,
  PUSH_BUTTON("Filter...", NULL, NULL, NULL, FilterCB, NULL,
	      "Filter", &filter_button),
  PUSH_BUTTON("Select...", NULL, NULL, NULL, SelectCB, NULL,
	      "Select", &select_button),
  PUSH_BUTTON("Select all", NULL, NULL, NULL, SelectAllCB, NULL,
	      "Select_all", &select_all_button),
  PUSH_BUTTON("Invert all", NULL, NULL, NULL, InvertAllCB, NULL,
	      "Invert_all", &invert_all_button),
  SEPARATOR,
  PUSH_BUTTON("Quit", NULL, NULL, NULL, QuitCB, NULL,
	      "Quit", &quit_button),
};

static MenuItem *command_menu_desc = NULL;

static MenuDesc view_menu_desc = {
  RADIO_BUTTON("Icons", NULL, NULL, NULL, IconsCB, NULL,
	       "Icons", &icons_button),
  RADIO_BUTTON("Text", NULL, NULL, NULL, TextCB, NULL,
	       "Text", &text_button),
  SEPARATOR,
  CHECK_BUTTON("Shelf pane", NULL, NULL, NULL, ShelfPaneCB, NULL,
	       "Shelf_pane", &shelf_pane_button),
  CHECK_BUTTON("Tree pane", NULL, NULL, NULL, TreePaneCB, NULL,
	       "Tree_pane", &tree_pane_button),
  CHECK_BUTTON("File pane", NULL, NULL, NULL, FilePaneCB, NULL,
	       "File_pane", &file_pane_button),
  SEPARATOR,
  RADIO_BUTTON("Sort by name", NULL, NULL, NULL, SortByNameCB, NULL,
	       "Sort_by_name", &sort_by_name_button),
  RADIO_BUTTON("Sort by size", NULL, NULL, NULL, SortBySizeCB, NULL,
	       "Sort_by_size", &sort_by_size_button),
  RADIO_BUTTON("Sort by date", NULL, NULL, NULL, SortByDateCB, NULL,
	       "Sort_by_date", &sort_by_date_button),
  SEPARATOR,
  CHECK_BUTTON("Reverse order", NULL, NULL, NULL, ReverseCB, NULL,
	       "Reverse_order", &reverse_button),
  CHECK_BUTTON("Directories first", NULL, NULL, NULL, DirsFirstCB, NULL,
	       "Directories_first", &dirs_first_button),
  SEPARATOR,
  CHECK_BUTTON("Show hidden files", NULL, NULL, NULL,
	       ShowHiddenFilesCB, NULL,
	       "Show_hidden_files", &show_hidden_button),
  CHECK_BUTTON("Show files", NULL, NULL, NULL, ShowFilesCB, NULL,
	       "Show_files", &show_files_button),
  CHECK_BUTTON("Show directories", NULL, NULL, NULL, ShowDirsCB, NULL,
	       "Show_directories", &show_dirs_button),
  CHECK_BUTTON("Show parent dir", NULL, NULL, NULL, ShowParentDirCB, NULL,
	       "Show_parent_dir", &show_parent_dir_button),
  SEPARATOR,
  CHECK_BUTTON("Rescan files", NULL, NULL, NULL, RescanFilesCB, NULL,
	       "Rescan_files", &rescan_files_button),
  CHECK_BUTTON("Magic headers", NULL, NULL, NULL, MagicHeadersCB, NULL,
	       "Magic_headers", &magic_headers_button),
  SEPARATOR,
  PUSH_BUTTON("Update", NULL, NULL, NULL, UpdateCB, NULL,
	      "Update", &update_button),
  PUSH_BUTTON("Reread", NULL, NULL, NULL, RereadCB, NULL,
	      "Reread", &reread_button),
  PUSH_BUTTON("Unmount", NULL, NULL, NULL, UnmountCB, NULL,
	      "Unmount", &unmount_button),
  PUSH_BUTTON("Unmount all", NULL, NULL, NULL, UnmountAllCB, NULL,
	      "Unmount_all", &unmount_all_button),
};

static MenuDesc options_menu_desc = {
  CHECK_BUTTON("Absolute paths", NULL, NULL, NULL, AbsolutePathsCB, NULL,
	       "Absolute_paths", &absolute_paths_button),
  CHECK_BUTTON("Push in current dir", NULL, NULL, NULL, PushDirCB, NULL,
	       "Push_in_current_dir", &push_dir_button),
  CHECK_BUTTON("Drop in target dir", NULL, NULL, NULL, TargetDirCB, NULL,
	       "Drop_in_target_dir", &target_dir_button),
  CHECK_BUTTON("Echo commands", NULL, NULL, NULL, EchoCommandsCB, NULL,
	       "Echo_commands", &echo_commands_button),
  CHECK_BUTTON("Check mounts", NULL, NULL, NULL, CheckMountsCB, NULL,
	       "Check_mounts", &check_mounts_button),
  CHECK_BUTTON("Backups", NULL, NULL, NULL, BackupsCB, NULL,
	       "Backups", &backups_button),
  SEPARATOR,
  CHECK_BUTTON("Confirm drop", NULL, NULL, NULL, ConfirmDropCB, NULL,
	       "Confirm_drop", &confirm_drop_button),
  CHECK_BUTTON("Confirm move", NULL, NULL, NULL, ConfirmMoveCB, NULL,
	       "Confirm_move", &confirm_move_button),
  CHECK_BUTTON("Confirm copy", NULL, NULL, NULL, ConfirmCopyCB, NULL,
	       "Confirm_copy", &confirm_copy_button),
  CHECK_BUTTON("Confirm link", NULL, NULL, NULL, ConfirmLinkCB, NULL,
	       "Confirm_link", &confirm_link_button),
  CHECK_BUTTON("Confirm delete", NULL, NULL, NULL, ConfirmDeleteCB, NULL,
	       "Confirm_delete", &confirm_delete_button),
  CHECK_BUTTON("Confirm delete dir", NULL, NULL, NULL, ConfirmDelDirCB, NULL,
	       "Confirm_deldir", &confirm_deldir_button),
  CHECK_BUTTON("Confirm delete file", NULL, NULL, NULL, ConfirmDelFileCB, NULL,
	       "Confirm_delfile", &confirm_delfile_button),
  CHECK_BUTTON("Confirm overwrite", NULL, NULL, NULL, ConfirmOverwriteCB,
	       NULL,
	       "Confirm_overwrite", &confirm_overwrite_button),
  CHECK_BUTTON("Confirm quit", NULL, NULL, NULL, ConfirmQuitCB, NULL,
	       "Confirm_quit", &confirm_quit_button),
  SEPARATOR,
  CHECK_BUTTON("Automatic updates", NULL, NULL, NULL, AutoUpdatesCB, NULL,
	       "Automatic_updates", &auto_updates_button),
  SEPARATOR,
  PUSH_BUTTON("Save setup", NULL, NULL, NULL, SaveSetupCB, NULL,
	      "Save_setup", &save_setup_button),
  PUSH_BUTTON("Reload configuration file", NULL, NULL, NULL, ReloadConfigCB,
	      NULL,
	      "Reload_configuration_file", &reload_config_button),
};

static MenuDesc shelf_menu_desc = {
  PUSH_BUTTON("Open", NULL, NULL, NULL, ShelfOpenCB, NULL,
	      "Open", &shelf_open_button),
  SEPARATOR,
  PUSH_BUTTON("Rename...", NULL, NULL, NULL, ShelfRenameCB, NULL,
	      "Rename", &shelf_rename_button),
  PUSH_BUTTON("Move...", NULL, NULL, NULL, ShelfMoveCB, NULL,
	      "Move", &shelf_move_button),
  PUSH_BUTTON("Copy...", NULL, NULL, NULL, ShelfCopyCB, NULL,
	      "Copy", &shelf_copy_button),
  PUSH_BUTTON("Link...", NULL, NULL, NULL, ShelfLinkCB, NULL,
	      "Link", &shelf_link_button),
  PUSH_BUTTON("Delete", NULL, NULL, NULL, ShelfDeleteCB, NULL,
	      "Delete", &shelf_delete_button),
  SEPARATOR,
  PUSH_BUTTON("Properties...", NULL, NULL, NULL, ShelfPropertiesCB, NULL,
	      "Properties", &shelf_properties_button),
  SEPARATOR,
  PUSH_BUTTON("Select...", NULL, NULL, NULL, ShelfSelectCB, NULL,
	      "Select", &shelf_select_button),
  PUSH_BUTTON("Select all", NULL, NULL, NULL, ShelfSelectAllCB, NULL,
	      "Select_all", &shelf_select_all_button),
  PUSH_BUTTON("Invert all", NULL, NULL, NULL, ShelfInvertAllCB, NULL,
	      "Invert_all", &shelf_invert_all_button),
};

static MenuDesc help_menu_desc = {
  PUSH_BUTTON("About...", NULL, NULL, NULL, AboutCB, NULL,
	      "About", &about_button),
};

static MenuItem *file_popup_desc = NULL;
static Widget *file_popup_button = NULL;
static int file_popup_size = 0;

static MenuDesc file_popup_desc0 = {
  PUSH_BUTTON("Open", NULL, NULL, NULL, OpenCB, NULL,
	      "Open", NULL),
  SEPARATOR,
  PUSH_BUTTON("Rename...", NULL, NULL, NULL, RenameCB, NULL,
	      "Rename", NULL),
  PUSH_BUTTON("Move...", NULL, NULL, NULL, MoveCB, NULL,
	      "Move", NULL),
  PUSH_BUTTON("Copy...", NULL, NULL, NULL, CopyCB, NULL,
	      "Copy", NULL),
  PUSH_BUTTON("Link...", NULL, NULL, NULL, LinkCB, NULL,
	      "Link", NULL),
  PUSH_BUTTON("Delete", NULL, NULL, NULL, DeleteCB, NULL,
	      "Delete", NULL),
  SEPARATOR,
  PUSH_BUTTON("Properties...", NULL, NULL, NULL, PropertiesCB, NULL,
	      "Properties", NULL),
  SEPARATOR,
};

static MenuDesc shelf_popup_desc0 = {
  PUSH_BUTTON("Open", NULL, NULL, NULL, ShelfOpenCB, NULL,
	      "Open", NULL),
  SEPARATOR,
  PUSH_BUTTON("Rename...", NULL, NULL, NULL, ShelfRenameCB, NULL,
	      "Rename", NULL),
  PUSH_BUTTON("Move...", NULL, NULL, NULL, ShelfMoveCB, NULL,
	      "Move", NULL),
  PUSH_BUTTON("Copy...", NULL, NULL, NULL, ShelfCopyCB, NULL,
	      "Copy", NULL),
  PUSH_BUTTON("Link...", NULL, NULL, NULL, ShelfLinkCB, NULL,
	      "Link", NULL),
  PUSH_BUTTON("Delete", NULL, NULL, NULL, ShelfDeleteCB, NULL,
	      "Delete", NULL),
  SEPARATOR,
  PUSH_BUTTON("Properties...", NULL, NULL, NULL, ShelfPropertiesCB, NULL,
	      "Properties", NULL),
  SEPARATOR,
};

/* create the description of the command menu */

static void CreateCommandMenuDesc(void)
{
  int i;
  
  for (i = 0; i < command_menu_size; i++)
    FREE(command_menu_desc[i].button_name);
  FREE(command_menu_desc); FREE(command_menu_button);
  command_menu_size = n_cmd_actions();
  command_menu_desc = (MenuItem*) MALLOC(command_menu_size*sizeof(MenuItem));
  command_menu_button = (Widget*) MALLOC(command_menu_size*sizeof(Widget));
  for (i = 0; i < n_cmd_actions(); i++) 
    if (cmd_label(i)) {
      String button_name = alloca(strlen(cmd_label(i))+1);

      resname(button_name, cmd_label(i));
      command_menu_desc[i].id = PUSH_BUTTON_ID;
      command_menu_desc[i].label = cmd_label(i);
      command_menu_desc[i].mnemonic = command_menu_desc[i].accelerator =
	command_menu_desc[i].accelerator_text = NULL;
      command_menu_desc[i].submenu = NULL;
      command_menu_desc[i].submenu_size = 0;
      command_menu_desc[i].callback = (XtCallbackProc) CommandActionCB;
      command_menu_desc[i].unmap_callback = NULL;
      command_menu_desc[i].data = (XtPointer) i;
      command_menu_desc[i].button = &command_menu_button[i];
      command_menu_desc[i].button_name = NEWSTRING(button_name);
      command_menu_desc[i].pulldown = NULL;
      command_menu_desc[i].pulldown_name = NULL;
    } else {
      command_menu_desc[i].id = SEPARATOR_ID;
      command_menu_desc[i].label = NULL;
      command_menu_desc[i].mnemonic = command_menu_desc[i].accelerator =
	command_menu_desc[i].accelerator_text = NULL;
      command_menu_desc[i].submenu = NULL;
      command_menu_desc[i].submenu_size = 0;
      command_menu_desc[i].callback = NULL;
      command_menu_desc[i].unmap_callback = NULL;
      command_menu_desc[i].data = NULL;
      command_menu_desc[i].button = NULL;
      command_menu_desc[i].button_name = NULL;
      command_menu_desc[i].pulldown = NULL;
      command_menu_desc[i].pulldown_name = NULL;
    }
}

/* create the description of the file popup */

static void CreateFilePopupDesc(DirPtr dir, int item)
{
  FilePtr file = dirFile(dir, item);
  int i, n = XtNumber(file_popup_desc0), type = fileType(file);
  
  for (i = n; i < file_popup_size; i++)
    FREE(file_popup_desc[i].button_name);
  FREE(file_popup_desc); FREE(file_popup_button);
  if (!n_menu_actions(type)) n--;
  file_popup_size = n+n_menu_actions(type);
  file_popup_desc = (MenuItem*) MALLOC(file_popup_size*sizeof(MenuItem));
  file_popup_button = (Widget*) MALLOC(file_popup_size*sizeof(Widget));
  for (i = 0; i < n; i++) {
    file_popup_desc[i] = (dir == curdir) ? file_popup_desc0[i] :
      shelf_popup_desc0[i];
    file_popup_desc[i].button = &file_popup_button[i];
  }
  for (i = 0; i < n_menu_actions(type); i++)
    if (menu_label(type, i)) {
      String button_name = alloca(strlen(menu_label(type, i))+1);

      resname(button_name, menu_label(type, i));
      file_popup_desc[i+n].id = PUSH_BUTTON_ID;
      file_popup_desc[i+n].label = menu_label(type, i);
      file_popup_desc[i+n].mnemonic = file_popup_desc[i+n].accelerator =
	file_popup_desc[i+n].accelerator_text = NULL;
      file_popup_desc[i+n].submenu = NULL;
      file_popup_desc[i+n].submenu_size = 0;
      file_popup_desc[i+n].callback = (XtCallbackProc)
	((dir == curdir) ? MenuActionCB : ShelfMenuActionCB);
      file_popup_desc[i+n].unmap_callback = NULL;
      file_popup_desc[i+n].data = (XtPointer) i;
      file_popup_desc[i+n].button = &file_popup_button[i+n];
      file_popup_desc[i+n].button_name = NEWSTRING(button_name);
      file_popup_desc[i+n].pulldown = NULL;
      file_popup_desc[i+n].pulldown_name = NULL;
    } else {
      file_popup_desc[i+n].id = SEPARATOR_ID;
      file_popup_desc[i+n].label = NULL;
      file_popup_desc[i+n].mnemonic = file_popup_desc[i+n].accelerator =
	file_popup_desc[i+n].accelerator_text = NULL;
      file_popup_desc[i+n].submenu = NULL;
      file_popup_desc[i+n].submenu_size = 0;
      file_popup_desc[i+n].callback = NULL;
      file_popup_desc[i+n].unmap_callback = NULL;
      file_popup_desc[i+n].data = NULL;
      file_popup_desc[i+n].button = NULL;
      file_popup_desc[i+n].button_name = NULL;
      file_popup_desc[i+n].pulldown = NULL;
      file_popup_desc[i+n].pulldown_name = NULL;
    }
}

/* convenience functions to create popup and pulldown menus from
   descriptions */

static void CreateMenuButtons(Widget menu, MenuDesc desc, int size);

static void CreateMenu(Widget parent,
		       int type,
		       String name,
		       String mnemonic,
		       String accelerator, String accelerator_text,
		       MenuDesc desc, int size,
		       XtCallbackProc map_callback,
		       XtCallbackProc unmap_callback,
		       String menu_name, Widget *menu,
		       String button_name, Widget *button)
{
  Widget pdn, but;
  XmString label;
  int nargs = 0;
  Arg args[3];
  
  if (type == MENU_PULLDOWN)
    *menu = XmCreatePulldownMenu(parent, menu_name, NULL, 0);
  else
    *menu = XmCreatePopupMenu(parent, menu_name, NULL, 0);
  if (map_callback)
    XtAddCallback(*menu,
		  XmNmapCallback, map_callback,
		  NULL);
  if (unmap_callback)
    XtAddCallback(*menu,
		  XmNunmapCallback, unmap_callback,
		  NULL);
  if (mnemonic) {
    XtSetArg(args[0],
	     XmNmnemonic, XStringToKeysym(mnemonic));
    nargs++;
  }
  if (accelerator) {
    XtSetArg(args[nargs],
	     XmNaccelerator, accelerator);
    nargs++;
  }
  if (accelerator_text) {
    label = XmStringCreateLocalized(accelerator_text);
    XtSetArg(args[nargs], XmNacceleratorText, label);
    nargs++;
  }
  if (accelerator_text)
    XmStringFree(label);
  if (type == MENU_PULLDOWN) {
    *button = XmCreateCascadeButton(parent, button_name, args, nargs);
    label = XmStringCreateLocalized(name);
    XtVaSetValues(*button,
		  XmNlabelString, label,
		  XmNsubMenuId, *menu,
		  NULL);
    XmStringFree(label);
    XtManageChild(*button);
  }
  CreateMenuButtons(*menu, desc, size);
}

static void CreateMenuButtons(Widget menu, MenuDesc desc, int size)
{
  XmString label;
  int i, sepcount = 0;
  int nargs;
  Arg args[3];
  
  for (i = 0; i < size; i++) {
    if (desc[i].id == SEPARATOR_ID)
      XtManageChild(XmCreateSeparator(menu, "Separator", NULL, 0));
    else {
      nargs = 0;
      if (desc[i].mnemonic) {
	XtSetArg(args[nargs],
		 XmNmnemonic, XStringToKeysym(desc[i].mnemonic));
	nargs++;
      }
      if (desc[i].accelerator) {
	XtSetArg(args[nargs],
		 XmNaccelerator, desc[i].accelerator);
	nargs++;
      }
      if (desc[i].accelerator_text) {
	label = XmStringCreateLocalized(desc[i].accelerator_text);
	XtSetArg(args[nargs],
		 XmNacceleratorText, label);
	nargs++;
      }
      switch (desc[i].id) {
      case PUSH_BUTTON_ID:
	*desc[i].button =
	  XmCreatePushButton(menu, desc[i].button_name,
			     args, nargs);
	break;
      case CHECK_BUTTON_ID:
      case RADIO_BUTTON_ID:
	*desc[i].button =
	  XmCreateToggleButton(menu, desc[i].button_name,
			       args, nargs);
	XtVaSetValues(*desc[i].button,
		      XmNindicatorType,
		      desc[i].id == CHECK_BUTTON_ID?XmN_OF_MANY:
		      XmONE_OF_MANY,
		      NULL);
	break;
      case CASCADE_BUTTON_ID:
	CreateMenu(menu, MENU_PULLDOWN,
		   desc[i].label,
		   desc[i].mnemonic,
		   desc[i].accelerator,
		   desc[i].accelerator_text,
		   desc[i].submenu, desc[i].submenu_size,
		   desc[i].callback,
		   desc[i].unmap_callback,
		   desc[i].pulldown_name, desc[i].pulldown,
		   desc[i].button_name, desc[i].button);
	break;
      default:
	/* shouldn't happen */
	fatal("invalid menu description");
      }
      if (desc[i].accelerator_text)
	XmStringFree(label);
      label = XmStringCreateLocalized(desc[i].label);
      XtVaSetValues(*desc[i].button,
		    XmNlabelString, label,
		    NULL);
      XmStringFree(label);
      if (desc[i].callback)
	switch(desc[i].id) {
	case PUSH_BUTTON_ID:
	  XtAddCallback(*desc[i].button,
			XmNactivateCallback, desc[i].callback,
			(XtPointer)desc[i].data);
	  break;
	case CHECK_BUTTON_ID:
	case RADIO_BUTTON_ID:
	  XtAddCallback(*desc[i].button,
			XmNvalueChangedCallback, desc[i].callback,
			(XtPointer)desc[i].data);
	  break;
	}
      XtManageChild(*desc[i].button);
    }
  }
}

/* file window: *************************************************************/

/* adjust the cell size of a container */

static void AdjustCellSize(Widget cont)
{
  unsigned char type;
  int count, i, n;
  WidgetList gadgets;
  Dimension cell_wd, cell_ht;

  XtVaGetValues(cont, XmNentryViewType, &type, NULL);
  if (type == XmLARGE_ICON)
    XtVaGetValues(cont, XmNlargeCellWidth, &cell_wd, XmNlargeCellHeight,
		  &cell_ht, NULL);
  else
    XtVaGetValues(cont, XmNsmallCellWidth, &cell_wd, XmNsmallCellHeight,
		  &cell_ht, NULL);
  XtVaGetValues(cont, XmNnumChildren, &n, XmNchildren, &gadgets, NULL);
  for (count = i = 0; i < n; i++)
    if (XtIsManaged(gadgets[i]) &&
	XtClass(gadgets[i]) == xmIconGadgetClass) {
      Dimension gadget_wd, gadget_ht, border_wd;
      
      XtVaGetValues(gadgets[i], XmNwidth, &gadget_wd, XmNheight, &gadget_ht,
		    XmNborderWidth, &border_wd, NULL);
      if (gadget_wd + 2 * border_wd > cell_wd)
	cell_wd = gadget_wd + 2 * border_wd;
      if (gadget_ht + 2 * border_wd > cell_ht)
	cell_ht = gadget_ht + 2 * border_wd;
      count++;
    }
  if (type == XmLARGE_ICON)
    XtVaSetValues(cont, XmNlargeCellWidth, cell_wd, XmNlargeCellHeight,
		  cell_ht, NULL);
  else
    XtVaSetValues(cont, XmNsmallCellWidth, cell_wd, XmNsmallCellHeight,
		  cell_ht, NULL);
}

/* calculate the number of icon gadget children and the cell size of a
   container with spatial layout type */

static void GetCellParams(Widget cont, Dimension *w, Dimension *h, int *count)
{
  unsigned char type;
  Cardinal numChildren;
  WidgetList children;
  int i;

  XtVaGetValues(cont, XmNentryViewType, &type, NULL);
  if (type == XmLARGE_ICON)
    XtVaGetValues(cont, XmNlargeCellWidth, w, XmNlargeCellHeight,
		  h, NULL);
  else
    XtVaGetValues(cont, XmNsmallCellWidth, w, XmNsmallCellHeight,
		  h, NULL);
  XtVaGetValues(cont,
		XmNnumChildren, &numChildren,
		XmNchildren, &children,
		NULL);
  for (*count = i = 0; i < numChildren; i++)
    if (XtIsManaged(children[i]) &&
	XtClass(children[i]) == xmIconGadgetClass)
      (*count)++;
}

/* calculate the height of a container in outline or detail mode */

static int ContHeight(Widget cont)
{
  Cardinal numChildren;
  WidgetList children;
  int i;
  Dimension ht, wd, cont_ht;

  XtVaGetValues(cont,
		XmNnumChildren, &numChildren,
		XmNchildren, &children,
		NULL);
  if (cont == file_scroll) {
    for (cont_ht = 0, i = 0; i < numChildren; i++)
      if (XtIsManaged(children[i]) &&
	  XtClass(children[i]) == xmIconGadgetClass) {
	XtVaGetValues(children[i], XmNheight, &ht, XmNborderWidth, &wd, NULL);
	cont_ht += ht + 2 * wd;
      }
  } else {
    /* same as above, but ignore placeholder gadgets in tree container */
    for (cont_ht = 0, i = 0; i < numChildren; i++)
      if (XtIsManaged(children[i]) &&
	  XtClass(children[i]) == xmIconGadgetClass) {
	DirPtr dir;

	XtVaGetValues(children[i], XmNuserData, &dir,
		      XmNheight, &ht, XmNborderWidth, &wd, NULL);
	if (dir)
	  cont_ht += ht + 2 * wd;
      }
  }
  return cont_ht;
}

/* resize a container according to the current size of its clip window */

static void Resize(Widget scroll)
{
  unsigned char type;
  Dimension clipw, cliph, scrollw, scrollh, margw, margh, cellw, cellh;
  int n, ncols, nlines;

  /* determine the layout type of the container */

  XtVaGetValues(scroll, XmNlayoutType, &type, NULL);

  /* get the current geometry of clip window and scroll */

  XtVaGetValues(XtParent(scroll),
		XmNwidth, &clipw,
		XmNheight, &cliph,
		NULL);
  XtVaGetValues(scroll,
		XmNwidth, &scrollw,
		XmNheight, &scrollh,
		XmNmarginWidth, &margw,
		XmNmarginHeight, &margh,
		NULL);

  /* Setting the new width or height of the scroll to the exact value of
     clipw or cliph sometimes creates a scrollbar (Motif bug?). Decrementing
     these values by 2 seems to work though. */

  clipw -= 2;
  cliph -= 2;

  if (type == XmSPATIAL) {

    /* calculate the new number of columns and lines to display */

    GetCellParams(scroll, &cellw, &cellh, &n);
    if (cellw)
      ncols = (clipw - 2 * margw) / cellw;
    else
      ncols = 1;
    if (ncols * cellw + 2 * margw > clipw)
      ncols--;
    if (!ncols) ncols = 1;
    nlines = n / ncols;
    if (nlines * ncols < n) nlines++;
  
    /* set the new width and height of the scroll */
  
    scrollh = nlines * cellh + 2 * margh;
    XtVaSetValues(scroll,
		  XmNwidth, clipw,
		  XmNheight, (scrollh > cliph) ? scrollh : cliph,
		  NULL);

  } else {
    scrollh = ContHeight(scroll);
    XtVaSetValues(scroll,
		  XmNheight, (scrollh > cliph) ? scrollh : cliph,
		  NULL);
  }
  
}

static Boolean ResizeWorkProc(Widget scroll)
{
  /* make sure that the scroll is still valid */
  if (scroll == tree_scroll || scroll == file_scroll ||
      scroll == shelf_scroll)
    Resize(scroll);
  return True;
}

static void ResizeCB(Widget clip, Widget scroll, XtPointer widget_data)
{
  if (XtIsRealized(scroll)) {
    /* be sure to invoke this as a background procedure, otherwise scrolled
       window widget is not updated properly */
    XtAppAddWorkProc(app, (XtWorkProc)ResizeWorkProc, scroll);
  }
}

/* traverse to obscured widgets in scrolled windows */

static void TravObscuredCB(Widget w, XtPointer app_data,
		    XmTraverseObscuredCallbackStruct *tocs)
{
  XmScrollVisible(w, tocs->traversal_destination, 0, 0);
}

/* Terrible kludge to force setting of a pane's geometry during update
   or remapping of a pane. This is achieved by setting XmNpane{Min,Max}imum
   accordingly. */

static void FreezePaneGeom(Widget pane, Dimension *save_min,
			   Dimension *save_max)
{
  unsigned char orientation;
  Dimension wd, ht;

  XtVaGetValues(XtParent(pane), XmNorientation, &orientation, NULL);
  XtVaGetValues(pane,
		XmNpaneMinimum, save_min,
		XmNpaneMaximum, save_max,
		XmNwidth, &wd,
		XmNheight, &ht,
		NULL);
  if (orientation == XmVERTICAL)
    XtVaSetValues(pane,
		  XmNpaneMinimum, ht,
		  XmNpaneMaximum, ht,
		  NULL);
  else
    XtVaSetValues(pane,
		  XmNpaneMinimum, wd,
		  XmNpaneMaximum, wd,
		  NULL);
}

/* revert to previous XmNpane{Min,Max}imum settings */

static void UnfreezePaneGeom(Widget pane, Dimension save_min,
			     Dimension save_max)
{
  XtVaSetValues(pane,
		XmNpaneMinimum, save_min,
		XmNpaneMaximum, save_max,
		NULL);
}

/* save and restore the slider positions of the scrollbars during update
   of a pane */

static void SaveScrollBarValue(Widget pane, Boolean vert, Dimension *save_pos)
{
  Widget scroll, scrollbar;
  Dimension scroll_sz;
  int sb_min, sb_max, sb_val;

  XtVaGetValues(pane, XmNworkWindow, &scroll,
		vert?XmNverticalScrollBar:XmNhorizontalScrollBar,
		&scrollbar, NULL);
  if (scroll != None)
    XtVaGetValues(scroll, vert?XmNheight:XmNwidth, &scroll_sz, NULL);
  else
    scroll_sz = 0;
  if (scrollbar != None)
    XtVaGetValues(scrollbar, XmNminimum, &sb_min, XmNmaximum, &sb_max,
		  XmNvalue, &sb_val, NULL);
  else
    sb_min = 0, sb_max = scroll_sz, sb_val = 0;
  /* translate the slider position to a position in the scroll */
  if (sb_max > sb_min)
    *save_pos = sb_val * scroll_sz / (sb_max - sb_min);
  else
    *save_pos = 0;
}

static void SaveScrollBarValues(Widget pane, Dimension *vert_pos,
				Dimension *horiz_pos)
{
  SaveScrollBarValue(pane, True, vert_pos);
  SaveScrollBarValue(pane, False, horiz_pos);
}

static void RestoreScrollBarValue(Widget pane, Boolean vert,
				  Dimension save_pos)
{
  Widget scroll, scrollbar;
  Dimension scroll_sz;
  int sb_min, sb_max, sb_val, sb_sld;

  XtVaGetValues(pane, XmNworkWindow, &scroll,
		vert?XmNverticalScrollBar:XmNhorizontalScrollBar,
		&scrollbar, NULL);
  if (scroll != None)
    XtVaGetValues(scroll, vert?XmNheight:XmNwidth, &scroll_sz, NULL);
  else
    scroll_sz = 0;
  if (scroll_sz > 0 && scrollbar != None) {
    XtVaGetValues(scrollbar, XmNminimum, &sb_min, XmNmaximum, &sb_max,
		  XmNsliderSize, &sb_sld, NULL);
    /* translate the saved position value to a slider position in the
       new scrollbar */
    sb_val = save_pos * (sb_max - sb_min) / scroll_sz;
    if (sb_val > sb_max - sb_sld)
      sb_val = sb_max - sb_sld;
    /* establish the new slider position; this requires that we reset the
       slider position first */
    XmScrollBarSetValues(scrollbar, sb_min, 0, 0, 0, False);
    XmScrollBarSetValues(scrollbar, sb_val, 0, 0, 0, True);
  }
}

static void RestoreScrollBarValues(Widget pane, Dimension vert_pos,
				   Dimension horiz_pos)
{
  RestoreScrollBarValue(pane, True, vert_pos);
  RestoreScrollBarValue(pane, False, horiz_pos);
}

/* reset the slider position to the minimum value */

static void ResetScrollBarValue(Widget pane, Boolean vert)
{
  Widget scrollbar;
  int sb_min;

  XtVaGetValues(pane, vert?XmNverticalScrollBar:XmNhorizontalScrollBar,
		&scrollbar, NULL);
  if (scrollbar != None) {
    XtVaGetValues(scrollbar, XmNminimum, &sb_min, NULL);
    XmScrollBarSetValues(scrollbar, sb_min, 0, 0, 0, True);
  }
}

static void ResetScrollBarValues(Widget pane)
{
  ResetScrollBarValue(pane, True);
  ResetScrollBarValue(pane, False);
}

/* file popup handler */

static void FilePopupDestroyCB(Widget w, XtPointer app_data,
			       XtPointer widget_data)
{
  file_popup = NULL;
}

static void FilePopupH(Widget w, XtPointer client_data, XEvent *event,
		       Boolean *continue_to_dispatch_return)
{
  XButtonPressedEvent *button_event = (XButtonPressedEvent *) event;
  Widget gadget = XmObjectAtPoint(w, button_event->x, button_event->y);
  DirPtr dir = (w == file_scroll)?curdir:shelfdir;

  if (button_event->button == 3 && gadget != None) {
    int item, i, n;
    WidgetList gadgets;

    XtVaGetValues(gadget, XmNuserData, &item, NULL);
    if (item == NONE) return;
    if (file_popup)
      XtDestroyWidget(file_popup);
    XmProcessTraversal(gadget, XmTRAVERSE_CURRENT);
    XtVaGetValues(w, XmNselectedObjectCount, &n,
		  XmNselectedObjects, &gadgets, NULL);
    for (i = 0; i < n; i++)
      XtVaSetValues(gadgets[i], XmNvisualEmphasis, XmNOT_SELECTED, NULL);
    XtVaSetValues(gadget, XmNvisualEmphasis, XmSELECTED, NULL);
    XmUpdateDisplay(w);
    dirDeselect(dir, ALL, NULL);
    dirSelect(dir, item, NULL);
    UpdateStatusLine();
    CreateFilePopupDesc(dir, item);
    CreateMenu(w, MENU_POPUP, NULL, NULL, NULL, NULL,
	       file_popup_desc, file_popup_size,
	       NULL, UnmapCB,
	       "file_popup", &file_popup,
	       NULL, NULL);
    XtAddCallback(file_popup, XmNdestroyCallback, FilePopupDestroyCB, NULL);
    XmMenuPosition(file_popup, button_event);
    lock = True;
    XtManageChild(file_popup);
  }
}

/* drag and drop */

/* finish a drag and drop transaction */

static void DragDropFinishCB(Widget w, Widget drag_icon, XtPointer widget_data)
{
  if (drag_icon)
    XtDestroyWidget(drag_icon);
}

/* start a drag */

static void FileStartDrag(Widget w, XEvent *event)
{
  XButtonPressedEvent *button_event = (XButtonPressedEvent *) event;
  Widget gadget = XmObjectAtPoint(w, button_event->x, button_event->y);
  DirPtr dir = (w == file_scroll)?curdir:shelfdir;

  if (gadget) {
    int i, n;
    WidgetList gadgets;
    Arg args[10];
    FilePtr file;
    IconPtr icon;
    Widget drag_context, drag_icon;

    XtVaGetValues(gadget, XmNuserData, &i, NULL);
    if (i == NONE) return;
    /* set focus to the dragged icon gadget and select it if necessary */
    XmProcessTraversal(gadget, XmTRAVERSE_CURRENT);
    file = dirFile(dir, i);
    if (!fileSelected(file)) {
      int j, m;

      XtVaGetValues(w, XmNselectedObjectCount, &m,
		    XmNselectedObjects, &gadgets, NULL);
      for (j = 0; j < m; j++)
	XtVaSetValues(gadgets[j], XmNvisualEmphasis, XmNOT_SELECTED, NULL);
      XtVaSetValues(gadget, XmNvisualEmphasis, XmSELECTED, NULL);
      dirDeselect(dir, ALL, NULL);
      dirSelect(dir, i, NULL);
      UpdateStatusLine();
    }
    XmUpdateDisplay(w);
    /* create the drag icon */
    if (dirNFilesSelected(dir) > 1)
      icon = &std_icons[ICON_FILES];
    else
      icon = fileLargeIcon(file);
    n = 0;
    XtSetArg(args[n], XmNpixmap, icon->pixmap); n++;
    XtSetArg(args[n], XmNdepth,
	     DefaultDepth(display, DefaultScreen(display))); n++;
    if (icon->mask != None) {
      XtSetArg(args[n], XmNmask, icon->mask); n++;
    }
    XtSetArg(args[n], XmNwidth, icon->width); n++;
    XtSetArg(args[n], XmNheight, icon->height); n++;
    drag_icon = XmCreateDragIcon(w, "drag_icon", args, n);
    /* create the drag context */
    n = 0;
    XtSetArg(args[n], XmNdragOperations,
	     XmDROP_MOVE|XmDROP_COPY|XmDROP_LINK); n++;
    XtSetArg(args[n], XmNsourcePixmapIcon, drag_icon); n++;
    drag_context = XmeDragSource(w, NULL, event, args, n);
    XtAddCallback(drag_context, XmNdragDropFinishCallback,
		  (XtCallbackProc)DragDropFinishCB, drag_icon);
    XtAddCallback(drag_context, XmNdropFinishCallback,
		  (XtCallbackProc)DropFinishCB, NULL);
    /* make sure we don't get an update during dragging */
    lock = True;
  }
}

DirPtr tree_drag_dir;

static void TreeStartDrag(Widget w, XEvent *event)
{
  XButtonPressedEvent *button_event = (XButtonPressedEvent *) event;
  Widget gadget = XmObjectAtPoint(w, button_event->x, button_event->y);

  if (gadget) {
    int n;
    WidgetList gadgets;
    Arg args[10];
    DirPtr dir;
    IconPtr icon;
    Widget drag_context, drag_icon;

    XtVaGetValues(gadget, XmNuserData, &dir, NULL);
    if (!dir || !strcmp(dirName(dir), "/")) return;
    XmProcessTraversal(gadget, XmTRAVERSE_CURRENT);
    /* create the drag icon */
    icon = dirLargeIcon(dir);
    n = 0;
    XtSetArg(args[n], XmNpixmap, icon->pixmap); n++;
    XtSetArg(args[n], XmNdepth,
	     DefaultDepth(display, DefaultScreen(display))); n++;
    if (icon->mask != None) {
      XtSetArg(args[n], XmNmask, icon->mask); n++;
    }
    XtSetArg(args[n], XmNwidth, icon->width); n++;
    XtSetArg(args[n], XmNheight, icon->height); n++;
    drag_icon = XmCreateDragIcon(w, "drag_icon", args, n);
    /* create the drag context */
    tree_drag_dir = dir;
    n = 0;
    XtSetArg(args[n], XmNdragOperations,
	     XmDROP_MOVE|XmDROP_COPY|XmDROP_LINK); n++;
    XtSetArg(args[n], XmNsourcePixmapIcon, drag_icon); n++;
    drag_context = XmeDragSource(w, dir, event, args, n);
    XtAddCallback(drag_context, XmNdragDropFinishCallback,
		  (XtCallbackProc)DragDropFinishCB, drag_icon);
    XtAddCallback(drag_context, XmNdropFinishCallback,
		  (XtCallbackProc)DropFinishCB, NULL);
    /* make sure we don't get an update during dragging */
    lock = True;
  }
}

/* register drop sites */

static void DropRegister(Widget w)
{
  Atom FILES = XInternAtom(display, "FILES", False);
  Atom FILE = XInternAtom(display, "FILE", False);
  Atom targets[2];
  int n = 0;
  Arg args[10];

  targets[0] = FILES;
  targets[1] = FILE;
  XtSetArg(args[n], XmNdropSiteOperations,
	   XmDROP_MOVE|XmDROP_COPY|XmDROP_LINK); n++;
  XtSetArg(args[n], XmNimportTargets, targets); n++;
  XtSetArg(args[n], XmNnumImportTargets, 2); n++;
  XtSetArg(args[n], XmNdragProc, DragProcCB); n++;
  /* disable drag under effects, in Motif 2.0 they're buggy anyhow */
  XtSetArg(args[n], XmNanimationStyle, XmDRAG_UNDER_NONE); n++;
  if (XmDropSiteRegistered(w))
    XmDropSiteUnregister(w);
  XmeDropSink(w, args, n);
}

/* update window title and icon name */

static void UpdateTitle(void)
{
  char title[3*MAXPATHLEN+7], icon_name[MAXPATHLEN+1];

  if (!treepane && !filepane) {
    *title = *icon_name = '\0';
  } else {
    if (show_getwd && dirGetwd(curdir) &&
	strcmp(dirName(curdir), dirGetwd(curdir))) {
      sprintf(title, "%s [%s]", dirName(curdir), dirGetwd(curdir));
    } else
      strcpy(title, dirName(curdir));
    strcpy(icon_name, dirName(curdir));
  }
  if (!*title) {
    strcpy(title, progname);
    strcpy(icon_name, progname);
  }
  XtVaSetValues(app_shell, XmNtitle, title, NULL);
  XtVaSetValues(app_shell, XmNiconName, icon_name, NULL);
}

/* icon gadget creation */

#define ICON_FILE_LINK_BAD_NAME	"file_link_bad_icon"
#define ICON_DIR_LINK_NAME	"dir_link_icon"
#define ICON_DIR_UP_NAME	"dir_up_icon"
#define ICON_DIR_NAME		"dir_icon"
#define ICON_EXEC_LINK_NAME	"exec_link_icon"
#define ICON_EXEC_NAME		"exec_icon"
#define ICON_FILE_LINK_NAME	"file_link_icon"
#define ICON_FILE_NAME		"file_icon"

static String icon_name(String name, StatPtr lstats, StatPtr stats)
{
  if (S_ISLNK(stats->st_mode))
    return ICON_FILE_LINK_BAD_NAME;
  else if (S_ISDIR(stats->st_mode))
    if (S_ISLNK(lstats->st_mode))
      return ICON_DIR_LINK_NAME;
    else if (!(strcmp(name, "..")))
      return ICON_DIR_UP_NAME;
    else
      return ICON_DIR_NAME;
  else if (stats->st_mode & (S_IXUSR | S_IXGRP | S_IXOTH))
    if (S_ISLNK(lstats->st_mode))
      return ICON_EXEC_LINK_NAME;
    else
      return ICON_EXEC_NAME;
  else if (S_ISLNK(lstats->st_mode))
    return ICON_FILE_LINK_NAME;
  else
    return ICON_FILE_NAME;
}

static Widget CreateDummyGadget(Widget parent, String name, XtPointer data)
{
  Widget icon_gadget =
    XtVaCreateManagedWidget(name, xmIconGadgetClass, parent, NULL);
  Pixel background;
  
  /* The following kludge removes shadows from the label. Setting the
     shadow thickness to zero does not work here since the pixmap may
     be clipped when the label string is shorter than the pixmap's width
     and the highlight thickness is greater than the shadow thickness
     (Motif bug?). */
  XtVaGetValues(icon_gadget, XmNbackground, &background, NULL);
  XtVaSetValues(icon_gadget,
		XmNtopShadowColor, background,
		XmNbottomShadowColor, background,
		XmNuserData, data,
		NULL);
  return icon_gadget;
}

static Widget CreateIconGadget(Widget parent, DirPtr dir, int i)
{
  FilePtr file = dirFile(dir, i);
  XmString label = XmStringCreateLocalized(fileName(file));
  String name = icon_name(fileName(file), fileLstats(file), fileStats(file));
  Widget icon_gadget =
    XtVaCreateManagedWidget(name, xmIconGadgetClass, parent,
			    XmNlabelString, label,
			    XmNlargeIconPixmap,
			    fileLargeIcon(file)->pixmap,
			    XmNlargeIconMask,
			    fileLargeIcon(file)->mask,
			    NULL);
  Pixel background;
  
  XtVaGetValues(icon_gadget, XmNbackground, &background, NULL);
  XtVaSetValues(icon_gadget,
		XmNtopShadowColor, background,
		XmNbottomShadowColor, background,
		XmNvisualEmphasis,
		fileSelected(file)?XmSELECTED:XmNOT_SELECTED,
		XmNuserData, i,
		NULL);
  XmStringFree(label);
  return icon_gadget;
}

static Widget CreateTextGadget(Widget parent, DirPtr dir, int i)
{
  FilePtr file = dirFile(dir, i);
  XmString label = XmStringCreateLocalized(fileName(file));
  String name = icon_name(fileName(file), fileLstats(file), fileStats(file));
  Widget icon_gadget =
    XtVaCreateManagedWidget(name, xmIconGadgetClass, parent,
			    XmNlabelString, label,
			    XmNsmallIconPixmap,
			    fileSmallIcon(file)->pixmap,
			    XmNsmallIconMask,
			    fileSmallIcon(file)->mask,
			    NULL);
  Pixel background;
  XmString details[5];
  int j, n = 0;
  struct passwd *pw;
  struct group *gr;
  char s[100];

  /* size */
  sprintf(s, "%lu", (unsigned long) fileLstats(file)->st_size);
  details[n++] = XmStringCreateLocalized(s);

  /* owner */
  if ((pw = getpwuid(fileLstats(file)->st_uid)) == NULL) {
    sprintf(s, "%lu", (unsigned long) fileLstats(file)->st_uid);
    details[n++] = XmStringCreateLocalized(s);
  } else
    details[n++] = XmStringCreateLocalized(pw->pw_name);

  /* group */
  if ((gr = getgrgid(fileLstats(file)->st_gid)) == NULL) {
    sprintf(s, "%lu", (unsigned long) fileLstats(file)->st_gid);
    details[n++] = XmStringCreateLocalized(s);
  } else
    details[n++] = XmStringCreateLocalized(gr->gr_name);

  /* permissions */
  sprintf(s, "%c%c%c%c%c%c%c%c%c%c",
	  S_ISLNK(fileLstats(file)->st_mode) ? 'l' :
	  S_ISDIR(fileLstats(file)->st_mode) ? 'd' : '-',
	  (S_IRUSR & fileLstats(file)->st_mode) ? 'r' : '-',
	  (S_IWUSR & fileLstats(file)->st_mode) ? 'w' : '-',
	  (S_IXUSR & fileLstats(file)->st_mode) ? 'x' : '-',
	  (S_IRGRP & fileLstats(file)->st_mode) ? 'r' : '-',
	  (S_IWGRP & fileLstats(file)->st_mode) ? 'w' : '-',
	  (S_IXGRP & fileLstats(file)->st_mode) ? 'x' : '-',
	  (S_IROTH & fileLstats(file)->st_mode) ? 'r' : '-',
	  (S_IWOTH & fileLstats(file)->st_mode) ? 'w' : '-',
	  (S_IXOTH & fileLstats(file)->st_mode) ? 'x' : '-');
  if (S_ISUID & fileLstats(file)->st_mode)
    s[3] = (s[3] == 'x')?'s':'S';
  if (S_ISGID & fileLstats(file)->st_mode)
    s[6] = (s[6] == 'x')?'s':'S';
  details[n++] = XmStringCreateLocalized(s); 

  /* date */
  strcpy(s, ctime(&(fileLstats(file)->st_mtime)));
  if (*s)
    s[strlen(s)-1] = '\0';
  details[n++] = XmStringCreateLocalized(s);

  XtVaGetValues(icon_gadget, XmNbackground, &background, NULL);
  XtVaSetValues(icon_gadget,
		XmNtopShadowColor, background,
		XmNbottomShadowColor, background,
		XmNdetail, details,
		XmNdetailCount, n,
		XmNvisualEmphasis,
		fileSelected(file)?XmSELECTED:XmNOT_SELECTED,
		XmNuserData, i,
		NULL);
  XmStringFree(label);
  for (j = 0; j < n; j++)
    XmStringFree(details[j]);
  return icon_gadget;
}

static void CreateFileGadgets(Widget parent, DirPtr dir, ViewType viewtype)
{
  int i;
  
  /* make sure we have at least a dummy gadget in the container; otherwise
     Motif breaks (bug?) */
  if (!dirIsReadable(dir) || !dirHasFileInfo(dir))
    CreateDummyGadget(parent, "unreadable", (XtPointer)NONE);
  else if (dirNFiles(dir) == 0)
    CreateDummyGadget(parent, "empty", (XtPointer)NONE);
  else if (viewtype == IconView)
    for (i = 0; i < dirNFiles(dir); i++)
      CreateIconGadget(parent, dir, i);
  else
    for (i = 0; i < dirNFiles(dir); i++)
      CreateTextGadget(parent, dir, i);
}

static Widget CreateTreeGadget(Widget parent, Widget parent_entry,
			       DirPtr dir)
{
  char dirname[MAXPATHLEN+1];
  XmString label;
  String name;
  Widget icon_gadget;
  Pixel background;
  
  filepart(dirname, dirName(dir));
  label = XmStringCreateLocalized((dir != rootdir)?dirname:dirName(dir));
  name = icon_name(dirname, dirLstats(dir), dirStats(dir));
  icon_gadget =
    XtVaCreateManagedWidget(name, xmIconGadgetClass, parent,
			    XmNlabelString, label,
			    XmNsmallIconPixmap,
			    dirSmallIcon(dir)->pixmap,
			    XmNsmallIconMask,
			    dirSmallIcon(dir)->mask,
			    XmNentryParent, parent_entry,
			    XmNoutlineState,
			    dirHasSubdirInfo(dir)?XmEXPANDED:XmCOLLAPSED,
			    NULL);
  XtVaGetValues(icon_gadget, XmNbackground, &background, NULL);
  XtVaSetValues(icon_gadget,
		XmNtopShadowColor, background,
		XmNbottomShadowColor, background,
		XmNuserData, dir,
		NULL);
  XmStringFree(label);
  if (dir == curdir)
    curgadget = icon_gadget;
  return icon_gadget;
}

static void CreateTreeGadgets(Widget parent, Widget parent_entry, DirPtr dir)
{
  int i;
  Widget entry = CreateTreeGadget(parent, parent_entry, dir);
  
  if (dirHasSubdirInfo(dir))
    for (i = 0; i < dirNSubdirs(dir); i++)
      CreateTreeGadgets(parent, entry, dirSubdir(dir, i));
  else if (dirHasSubdirs(dir)) {
    /* create a placeholder entry */
    Widget subdir_entry = CreateDummyGadget(parent, "subdir", NULL);
    XtVaSetValues(subdir_entry, XmNentryParent, entry, NULL);
  }
}

/* default dimensions of the panes */

static Dimension view_wd, view_ht, tree_wd, tree_ht, file_wd, file_ht, 
  shelf_wd, shelf_ht;

/* create scrolls and the corresponding panes */

static Widget new_file_scroll;

static void UpdateFileScroll(Widget parent)
{
  if (viewtype == IconView)
    new_file_scroll =
      XtVaCreateWidget("file_scroll_icons",
		       xmContainerWidgetClass,
		       parent,
		       XmNentryViewType, XmLARGE_ICON,
		       XmNlayoutType, XmSPATIAL,
		       XmNspatialStyle, XmGRID,
		       XmNspatialResizeModel, XmGROW_MINOR,
		       XmNspatialSnapModel, XmCENTER,
		       XmNselectionPolicy, XmEXTENDED_SELECT,
		       XmNautomaticSelection, XmNO_AUTO_SELECT,
		       XmNprimaryOwnership, XmOWN_NEVER,
		       XmNtranslations, file_trans,
		       NULL);
  else
    new_file_scroll =
      XtVaCreateWidget("file_scroll_text",
		       xmContainerWidgetClass,
		       parent,
		       XmNentryViewType, XmSMALL_ICON,
		       XmNlayoutType, XmDETAIL,
		       XmNselectionPolicy, XmEXTENDED_SELECT,
		       XmNautomaticSelection, XmNO_AUTO_SELECT,
		       XmNoutlineButtonPolicy, XmOUTLINE_BUTTON_ABSENT,
		       XmNprimaryOwnership, XmOWN_NEVER,
		       XmNtranslations, file_trans,
		       NULL);
  CreateFileGadgets(new_file_scroll, curdir, viewtype);
  if (viewtype == IconView)
    AdjustCellSize(new_file_scroll);
  DropRegister(new_file_scroll);
  XtAddCallback(new_file_scroll, XmNdefaultActionCallback,
		(XtCallbackProc) FileActionCB, NULL);
  XtAddCallback(new_file_scroll, XmNselectionCallback,
		(XtCallbackProc) FileSelectionCB, NULL);
  XtAddCallback(new_file_scroll, XmNconvertCallback,
		(XtCallbackProc) FileConvertCB, NULL);
  XtAddCallback(new_file_scroll, XmNdestinationCallback,
		(XtCallbackProc) FileDestinationCB, NULL);
  XtRemoveAllCallbacks(XtParent(new_file_scroll), XmNresizeCallback);
  XtAddCallback(XtParent(new_file_scroll), XmNresizeCallback,
		(XtCallbackProc) ResizeCB, new_file_scroll);
  XtAddEventHandler(new_file_scroll, ButtonPressMask, False, FilePopupH, NULL);
}

static void RedrawFileScroll(void)
{
  if (file_scroll != None)
    XtDestroyWidget(file_scroll);
  file_scroll = new_file_scroll;
  XtManageChild(file_scroll);
  UpdateTitle();
}

static void CreateFileScroll(Widget parent)
{
  UpdateFileScroll(parent);
  RedrawFileScroll();
}

static void CreateFilePane(Widget parent)
{
  file_pane = XtVaCreateWidget("file_pane",
			       xmScrolledWindowWidgetClass,
			       parent,
			       XmNscrollingPolicy, XmAUTOMATIC,
			       NULL);
  XtVaGetValues(file_pane, XmNwidth, &file_wd, XmNheight, &file_ht, NULL);
  XtAddCallback(file_pane, XmNtraverseObscuredCallback,
		(XtCallbackProc) TravObscuredCB, NULL);
  CreateFileScroll(file_pane);
  if (filepane)
    XtManageChild(file_pane);
}

static Widget new_shelf_scroll;

static void UpdateShelfScroll(Widget parent)
{
  new_shelf_scroll =
    XtVaCreateWidget("shelf_scroll",
		     xmContainerWidgetClass,
		     parent,
		     XmNentryViewType, XmLARGE_ICON,
		     XmNlayoutType, XmSPATIAL,
		     XmNspatialStyle, XmGRID,
		     XmNspatialResizeModel, XmGROW_MINOR,
		     XmNspatialSnapModel, XmCENTER,
		     XmNselectionPolicy, XmEXTENDED_SELECT,
		     XmNautomaticSelection, XmNO_AUTO_SELECT,
		     XmNprimaryOwnership, XmOWN_NEVER,
		     XmNtranslations, file_trans,
		     NULL);
  CreateFileGadgets(new_shelf_scroll, shelfdir, IconView);
  AdjustCellSize(new_shelf_scroll);
  DropRegister(new_shelf_scroll);
  XtAddCallback(new_shelf_scroll, XmNdefaultActionCallback,
		(XtCallbackProc) ShelfActionCB, NULL);
  XtAddCallback(new_shelf_scroll, XmNselectionCallback,
		(XtCallbackProc) ShelfSelectionCB, NULL);
  XtAddCallback(new_shelf_scroll, XmNconvertCallback,
		(XtCallbackProc) ShelfConvertCB, NULL);
  XtAddCallback(new_shelf_scroll, XmNdestinationCallback,
		(XtCallbackProc) ShelfDestinationCB, NULL);
  XtRemoveAllCallbacks(XtParent(new_shelf_scroll), XmNresizeCallback);
  XtAddCallback(XtParent(new_shelf_scroll), XmNresizeCallback,
		(XtCallbackProc) ResizeCB, new_shelf_scroll);
  XtAddEventHandler(new_shelf_scroll, ButtonPressMask, False, FilePopupH,
		    NULL);
}

static void RedrawShelfScroll(void)
{
  if (shelf_scroll != None)
    XtDestroyWidget(shelf_scroll);
  shelf_scroll = new_shelf_scroll;
  XtManageChild(shelf_scroll);
}

static void CreateShelfScroll(Widget parent)
{
  UpdateShelfScroll(parent);
  RedrawShelfScroll();
}

static void CreateShelfPane(Widget parent)
{
  shelf_pane = XtVaCreateWidget("shelf_pane",
			       xmScrolledWindowWidgetClass,
			       parent,
			       XmNscrollingPolicy, XmAUTOMATIC,
			       NULL);
  XtVaGetValues(shelf_pane, XmNwidth, &shelf_wd, XmNheight, &shelf_ht, NULL);
  XtAddCallback(shelf_pane, XmNtraverseObscuredCallback,
		(XtCallbackProc) TravObscuredCB, NULL);
  CreateShelfScroll(shelf_pane);
  if (shelfpane)
    XtManageChild(shelf_pane);
}

static Widget new_tree_scroll;

static void UpdateTreeScroll(Widget parent)
{
  new_tree_scroll =
    XtVaCreateWidget("tree_scroll",
		     xmContainerWidgetClass,
		     parent,
		     XmNentryViewType, XmSMALL_ICON,
		     XmNlayoutType, XmOUTLINE,
		     XmNselectionPolicy, XmBROWSE_SELECT,
		     XmNautomaticSelection, XmNO_AUTO_SELECT,
		     XmNprimaryOwnership, XmOWN_NEVER,
		     XmNtranslations, tree_trans,
		     NULL);
  CreateTreeGadgets(new_tree_scroll, None, rootdir);
  XtVaSetValues(curgadget, XmNvisualEmphasis, XmSELECTED, NULL);
  DropRegister(new_tree_scroll);
  XtAddCallback(new_tree_scroll, XmNoutlineChangedCallback,
		(XtCallbackProc) TreeOutlineCB, NULL);
  XtAddCallback(new_tree_scroll, XmNselectionCallback,
		(XtCallbackProc) TreeSelectionCB, NULL);
  XtAddCallback(new_tree_scroll, XmNconvertCallback,
		(XtCallbackProc) TreeConvertCB, NULL);
  XtAddCallback(new_tree_scroll, XmNdestinationCallback,
		(XtCallbackProc) TreeDestinationCB, NULL);
  XtRemoveAllCallbacks(XtParent(new_tree_scroll), XmNresizeCallback);
  XtAddCallback(XtParent(new_tree_scroll), XmNresizeCallback,
		(XtCallbackProc) ResizeCB, new_tree_scroll);
}

static void RedrawTreeScroll(void)
{
  if (tree_scroll != None)
    XtDestroyWidget(tree_scroll);
  tree_scroll = new_tree_scroll;
  XtManageChild(tree_scroll);
  UpdateTitle();
}

static void CreateTreeScroll(Widget parent)
{
  UpdateTreeScroll(parent);
  RedrawTreeScroll();
}

static void CreateTreePane(Widget parent)
{
  tree_pane = XtVaCreateWidget("tree_pane",
			       xmScrolledWindowWidgetClass,
			       parent,
			       XmNscrollingPolicy, XmAUTOMATIC,
			       NULL);
  XtVaGetValues(tree_pane, XmNwidth, &tree_wd, XmNheight, &tree_ht, NULL);
  XtAddCallback(tree_pane, XmNtraverseObscuredCallback,
		(XtCallbackProc) TravObscuredCB, NULL);
  CreateTreeScroll(tree_pane);
  if (treepane)
    XtManageChild(tree_pane);
}

/* create and update the shelf tabs */

static void CreateShelfTabs(Widget parent)
{
  String *labels = (String*)alloca(n_shelves()*sizeof(String));
  int i;
  for (i = 0; i < n_shelves(); i++)
    labels[i] = shelf_label(i)?shelf_label(i):"";
  shelf_tabs = XtVaCreateWidget("shelf_tabs",
				xgTabsWidgetClass,
				parent,
				XgNtabLabels, labels,
				XgNcurrentTab, default_shelf,
				XgNtabCount, n_shelves(),
				NULL);
  XtAddCallback(shelf_tabs, XmNactivateCallback, (XtCallbackProc) ShelfCB,
		NULL);
  if (shelfpane)
    XtManageChild(shelf_tabs);
}

static void UpdateShelfTabs()
{
  String *labels = (String*)alloca(n_shelves()*sizeof(String));
  int i;
  for (i = 0; i < n_shelves(); i++)
    labels[i] = shelf_label(i)?shelf_label(i):"";
  XtVaSetValues(shelf_tabs,
		XgNtabLabels, labels,
		XgNcurrentTab, default_shelf,
		XgNtabCount, n_shelves(),
		NULL);
}

/* create the file view */

static void CreateFileView(Widget parent)
{
  file_view = XtVaCreateWidget("file_view",
			       xmPanedWindowWidgetClass,
			       parent,
			       NULL);
  XtVaGetValues(file_view, XmNwidth, &view_wd, XmNheight, &view_ht, NULL);
  CreateTreePane(file_view);
  CreateFilePane(file_view);
  if (treepane || filepane)
    XtManageChild(file_view);
}

/* create the work area */

static void CreateWorkArea(Widget parent)
{
  frame = XtVaCreateManagedWidget("frame",
				  xmFrameWidgetClass,
				  parent,
				  NULL);
  work_area = XtVaCreateManagedWidget("work_area",
				      xmPanedWindowWidgetClass,
				      frame,
				      NULL);
  CreateShelfPane(work_area);
  CreateFileView(work_area);
}

/* create attachments for the main form */

static void CreateAttachments(void)
{
  if (shelfpane) {
    int loc;
    XtVaGetValues(shelf_tabs, XgNtabLocation, &loc, NULL);
    switch (loc) {
    case XgTOP:
      XtVaSetValues(shelf_tabs,
		    XmNtopAttachment, XmATTACH_FORM,
		    XmNbottomAttachment, XmATTACH_NONE,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNrightAttachment, XmATTACH_FORM,
		    NULL);
      XtVaSetValues(frame,
		    XmNtopAttachment, XmATTACH_WIDGET,
		    XmNtopWidget, shelf_tabs,
		    XmNbottomAttachment, XmATTACH_FORM,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNrightAttachment, XmATTACH_FORM,
		    NULL);
      break;
    case XgBOTTOM:
      XtVaSetValues(shelf_tabs,
		    XmNbottomAttachment, XmATTACH_FORM,
		    XmNtopAttachment, XmATTACH_NONE,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNrightAttachment, XmATTACH_FORM,
		    NULL);
      XtVaSetValues(frame,
		    XmNbottomAttachment, XmATTACH_WIDGET,
		    XmNbottomWidget, shelf_tabs,
		    XmNtopAttachment, XmATTACH_FORM,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNrightAttachment, XmATTACH_FORM,
		    NULL);
      break;
    case XgLEFT:
      XtVaSetValues(shelf_tabs,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNrightAttachment, XmATTACH_NONE,
		    XmNtopAttachment, XmATTACH_FORM,
		    XmNbottomAttachment, XmATTACH_FORM,
		    NULL);
      XtVaSetValues(frame,
		    XmNleftAttachment, XmATTACH_WIDGET,
		    XmNleftWidget, shelf_tabs,
		    XmNrightAttachment, XmATTACH_FORM,
		    XmNtopAttachment, XmATTACH_FORM,
		    XmNbottomAttachment, XmATTACH_FORM,
		    NULL);
      break;
    case XgRIGHT:
      XtVaSetValues(shelf_tabs,
		    XmNrightAttachment, XmATTACH_FORM,
		    XmNleftAttachment, XmATTACH_NONE,
		    XmNtopAttachment, XmATTACH_FORM,
		    XmNbottomAttachment, XmATTACH_FORM,
		    NULL);
      XtVaSetValues(frame,
		    XmNrightAttachment, XmATTACH_WIDGET,
		    XmNrightWidget, shelf_tabs,
		    XmNleftAttachment, XmATTACH_FORM,
		    XmNtopAttachment, XmATTACH_FORM,
		    XmNbottomAttachment, XmATTACH_FORM,
		    NULL);
      break;
    }
  } else
    XtVaSetValues(frame,
		  XmNtopAttachment, XmATTACH_FORM,
		  XmNbottomAttachment, XmATTACH_FORM,
		  XmNleftAttachment, XmATTACH_FORM,
		  XmNrightAttachment, XmATTACH_FORM,
		  NULL);
}

/* create the main form */

static void CreateForm(Widget parent)
{
  form = XtVaCreateManagedWidget("form",
				 xmFormWidgetClass,
				 parent,
				 NULL);
  CreateShelfTabs(form);
  CreateWorkArea(form);
  CreateAttachments();
}

/* create the status line */

static char status[1000];

static void update_status(void)
{
  int i;
  long bsize, ksize, free, free_blocks;
  long n_bytes, n_files, n_bytes_sel, n_files_sel;
  int res;
#ifdef USE_STATVFS
  struct statvfs stat;
#else
  struct statfs stat;
#endif

  status[0] = 0;

  if (!filepane && !treepane)
    return;

  if (filepane && dirFilter(curdir))
    sprintf(status, "[%s] ", dirFilter(curdir));

  n_files = dirNFiles(curdir);
  n_bytes = dirNBytes(curdir);
  n_files_sel = dirNFilesSelected(curdir);
  n_bytes_sel = dirNBytesSelected(curdir);

  sprintf(status+strlen(status),
	  "%ld byte%s in %d item%s",
	  n_bytes, n_bytes == 1 ? "" : "s",
	  n_files, n_files == 1 ? "" : "s");
  if (filepane && n_files_sel)
     sprintf(status+strlen(status), ", %ld byte%s in %d selected item%s",
	     n_bytes_sel, n_bytes_sel == 1 ? "" : "s",
	     n_files_sel, n_files_sel == 1 ? "" : "s");

  /* the following has been pilfered from moxfm -- thanks to Oliver Mai */

#ifdef USE_STATVFS
  res = statvfs(dirName(curdir), &stat);
#else
  res = statfs(dirName(curdir), &stat);
#endif
#ifdef OSF1
  bsize = (long) stat.f_fsize / 1024L;
  if (!bsize)  ksize = 1024L / (long) stat.f_fsize;
#else
  bsize = (long) stat.f_bsize / 1024L;
  if (!bsize)  ksize = 1024L / (long) stat.f_bsize;
#endif

  if (!res) {
    if (geteuid())
      free_blocks = stat.f_bavail;
    else
      free_blocks = stat.f_bfree;

    if (bsize) free = free_blocks * bsize;
    else  free = free_blocks / ksize;

    sprintf(status+strlen(status), ", %ld KB available", free);
  }
}

static int n_dev_indicators = 0;
static Widget *dev_indicators = NULL;
static int *devs = NULL;

static Pixel foreground, highlightColor;

static void UpdateDeviceIndicators(void)
{
  int i;
  if (check_mnt_table())
    get_mnt_table();
  for (i = 0; i < n_dev_indicators; i++) {
    if (premounted(devs[i]))
      XtVaSetValues(dev_indicators[i], XmNforeground, highlightColor, NULL);
    else
      XtVaSetValues(dev_indicators[i], XmNforeground, foreground, NULL);
    if (devcount(devs[i]))
      XtSetSensitive(dev_indicators[i], True);
    else
      XtSetSensitive(dev_indicators[i], False);
  }
}

static void CreateDeviceIndicators(Widget parent)
{
  int i, d;
  if (dev_indicators) {
    for (i = 0; i < n_dev_indicators; i++)
      XtDestroyWidget(dev_indicators[i]);
    FREE(dev_indicators);
    FREE(devs);
  }
  n_dev_indicators = 0;
  for (d = 0; d < ndevs(); d++)
    if (devlabel(d))
      n_dev_indicators++;
  if (n_dev_indicators) {
    XtManageChild(indicators_frame);
    XtVaSetValues(message_frame,
		  XmNrightAttachment, XmATTACH_WIDGET,
		  XmNrightWidget, indicators_frame,
		  XmNrightOffset, 3,
		  NULL);
    dev_indicators = (Widget*)MALLOC(n_dev_indicators*sizeof(Widget));
    devs = (int*)MALLOC(n_dev_indicators*sizeof(int));
  } else {
    XtUnmanageChild(indicators_frame);
    XtVaSetValues(message_frame,
		  XmNrightAttachment, XmATTACH_FORM,
		  XmNrightOffset, 0,
		  NULL);
    dev_indicators = NULL;
    devs = NULL;
  }
  for (i = d = 0; d < ndevs(); d++)
    if (devlabel(d)) {
      XmString labelstr = XmStringCreateLocalized(devlabel(d));
      Pixel background;
      if (i > 0)
	dev_indicators[i] =
	  XtVaCreateManagedWidget("device_indicator", xmIconGadgetClass,
				  parent,
				  XmNlabelString, labelstr,
				  XmNviewType, XmSMALL_ICON,
				  XmNleftAttachment, XmATTACH_WIDGET,
				  XmNleftWidget, dev_indicators[i-1],
				  XmNleftOffset, 5,
				  XmNrightAttachment, XmATTACH_NONE,
				  XmNtopAttachment, XmATTACH_NONE,
				  XmNbottomAttachment, XmATTACH_NONE,
				  NULL);
      else
	dev_indicators[i] =
	  XtVaCreateManagedWidget("device_indicator", xmIconGadgetClass,
				  parent,
				  XmNlabelString, labelstr,
				  XmNviewType, XmSMALL_ICON,
				  XmNtraversalOn, False,
				  XmNleftAttachment, XmATTACH_FORM,
				  XmNrightAttachment, XmATTACH_NONE,
				  XmNtopAttachment, XmATTACH_NONE,
				  XmNbottomAttachment, XmATTACH_NONE,
				  NULL);
      if (devicon(d))
	XtVaSetValues(dev_indicators[i],
		      XmNsmallIconPixmap, devicon(d)->pixmap,
		      XmNsmallIconMask, devicon(d)->mask,
		      NULL);
      XtVaGetValues(dev_indicators[i], XmNbackground, &background, NULL);
      XtVaSetValues(dev_indicators[i],
		    XmNtopShadowColor, background,
		    XmNbottomShadowColor, background,
		    NULL);
      XmStringFree(labelstr);
      devs[i++] = d;
    }
  XtVaGetValues(indicators, XmNhighlightColor, &highlightColor, NULL);
  XtVaGetValues(indicators, XmNforeground, &foreground, NULL);
  UpdateDeviceIndicators();
}

static void CreateStatusLine(Widget parent)
{
  status_line =
    XtVaCreateManagedWidget("status_line", xmFormWidgetClass, parent, NULL);
  XtVaSetValues(parent, XmNmessageWindow, status_line, NULL);
  message_frame =
    XtVaCreateManagedWidget("message_frame", xmFrameWidgetClass, status_line,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    NULL);
  message =
    XtVaCreateManagedWidget("message", xmLabelGadgetClass, message_frame,
			    XmNlabelType, XmSTRING,
			    XmNframeChildType, XmFRAME_WORKAREA_CHILD,
			    XmNchildHorizontalAlignment, XmALIGNMENT_BEGINNING,
			    NULL);
  indicators_frame =
    XtVaCreateWidget("indicators_frame", xmFrameWidgetClass,
		     status_line,
		     XmNleftAttachment, XmATTACH_NONE,
		     XmNrightAttachment, XmATTACH_FORM,
		     XmNtopAttachment, XmATTACH_FORM,
		     XmNbottomAttachment, XmATTACH_FORM,
		     NULL);
  indicators =
    XtVaCreateManagedWidget("indicators", xmFormWidgetClass, indicators_frame,
			    XmNframeChildType, XmFRAME_WORKAREA_CHILD,
			    XmNchildHorizontalAlignment, XmALIGNMENT_BEGINNING,
			    NULL);
  CreateDeviceIndicators(indicators);
}

/* interface routines: ******************************************************/

/* hide and show the panes */

void HideTreePane(void)
{
  if (treepane) {
    XtUnmanageChild(tree_pane);
    treepane = False;
    if (!filepane)
      XtUnmanageChild(file_view);
    UpdateTitle();
    UpdateStatusLine();
  }
}

void ShowTreePane(void)
{
  if (!treepane) {
    Dimension save_min, save_max;

    treepane = True;
    UpdateTitle();
    UpdateStatusLine();
    if (!filepane) {
      XtVaSetValues(file_view, XmNwidth, view_wd, XmNheight, view_ht, NULL);
      FreezePaneGeom(file_view, &save_min, &save_max);
      XtManageChild(file_view);
      UnfreezePaneGeom(file_view, save_min, save_max);
    }
    XtVaSetValues(tree_pane, XmNwidth, tree_wd, XmNheight, tree_ht, NULL);
    FreezePaneGeom(tree_pane, &save_min, &save_max);
    XtManageChild(tree_pane);
    UnfreezePaneGeom(tree_pane, save_min, save_max);
    XtAppAddWorkProc(app, (XtWorkProc)ResizeWorkProc, tree_scroll);
  }
}

void HideFilePane(void)
{
  if (filepane) {
    XtUnmanageChild(file_pane);
    filepane = False;
    if (!treepane)
      XtUnmanageChild(file_view);
    UpdateTitle();
    UpdateStatusLine();
  }
}

void ShowFilePane(void)
{
  if (!filepane) {
    Dimension save_min, save_max;

    filepane = True;
    UpdateTitle();
    UpdateStatusLine();
    if (!treepane) {
      XtVaSetValues(file_view, XmNwidth, view_wd, XmNheight, view_ht, NULL);
      FreezePaneGeom(file_view, &save_min, &save_max);
      XtManageChild(file_view);
      UnfreezePaneGeom(file_view, save_min, save_max);
    }
    XtVaSetValues(file_pane, XmNwidth, file_wd, XmNheight, file_ht, NULL);
    FreezePaneGeom(file_pane, &save_min, &save_max);
    XtManageChild(file_pane);
    UnfreezePaneGeom(file_pane, save_min, save_max);
    XtAppAddWorkProc(app, (XtWorkProc)ResizeWorkProc, file_scroll);
  }
}

void HideShelfPane(void)
{
  if (shelfpane) {
    XtUnmanageChild(shelf_tabs);
    XtUnmanageChild(shelf_pane);
    shelfpane = False;
    CreateAttachments();
    UpdateStatusLine();
  }
}

void ShowShelfPane(void)
{
  if (!shelfpane) {
    Dimension save_min, save_max;

    shelfpane = True;
    UpdateStatusLine();
    XtManageChild(shelf_tabs);
    XtVaSetValues(shelf_pane, XmNwidth, shelf_wd, XmNheight, shelf_ht, NULL);
    FreezePaneGeom(shelf_pane, &save_min, &save_max);
    XtManageChild(shelf_pane);
    UnfreezePaneGeom(shelf_pane, save_min, save_max);
    CreateAttachments();
    XtAppAddWorkProc(app, (XtWorkProc)ResizeWorkProc, shelf_scroll);
  }
}

/* update the panes */

void UpdatePanes(Boolean do_shelf, Boolean do_tree, Boolean do_file,
		 Boolean save_shelf, Boolean save_tree, Boolean save_file)
{
  Dimension tree_min, tree_max, tree_vert, tree_horiz;
  Dimension file_min, file_max, file_vert, file_horiz;
  Dimension shelf_min, shelf_max, shelf_vert, shelf_horiz;

  if (!do_shelf && !do_tree && !do_file)
    return;
  if (do_shelf) {
    if (save_shelf)
      SaveScrollBarValues(shelf_pane, &shelf_vert, &shelf_horiz);
    FreezePaneGeom(shelf_pane, &shelf_min, &shelf_max);
    UpdateShelfScroll(shelf_pane);
  }
  if (do_tree) {
    if (save_tree)
      SaveScrollBarValues(tree_pane, &tree_vert, &tree_horiz);
    FreezePaneGeom(tree_pane, &tree_min, &tree_max);
    UpdateTreeScroll(tree_pane);
  }
  if (do_file) {
    if (save_file)
      SaveScrollBarValues(file_pane, &file_vert, &file_horiz);
    FreezePaneGeom(file_pane, &file_min, &file_max);
    UpdateFileScroll(file_pane);
  }
  if (do_shelf) {
    RedrawShelfScroll();
    UnfreezePaneGeom(shelf_pane, shelf_min, shelf_max);
    if (shelfpane)
      Resize(shelf_scroll);
    if (save_shelf)
      RestoreScrollBarValues(shelf_pane, shelf_vert, shelf_horiz);
    else
      ResetScrollBarValues(shelf_pane);
  }
  if (do_tree) {
    RedrawTreeScroll();
    UnfreezePaneGeom(tree_pane, tree_min, tree_max);
    if (treepane)
      Resize(tree_scroll);
    if (save_tree)
      RestoreScrollBarValues(tree_pane, tree_vert, tree_horiz);
    else
      ResetScrollBarValues(tree_pane);
  }
  if (do_file) {
    RedrawFileScroll();
    UnfreezePaneGeom(file_pane, file_min, file_max);
    if (filepane)
      Resize(file_scroll);
    if (save_file)
      RestoreScrollBarValues(file_pane, file_vert, file_horiz);
    else
      ResetScrollBarValues(file_pane);
  }
  if (treepane) {
    XmScrollVisible(tree_pane, curgadget, 20, 20);
    XmProcessTraversal(curgadget, XmTRAVERSE_CURRENT);
    XtVaSetValues(tree_scroll, XmNinitialFocus, curgadget, NULL);
  }
  UpdateStatusLine();
}

/* delete childs in the tree container */

/* this could be implemented more efficiently (the current implementation
   needs time O(nm) where n is the number of icon gadgets and m the number
   of items to be deleted), but we count on the fact that the tree container
   has not more than, say, a few hundred items which should be manageable */

static void DeleteSubtree(Widget w, WidgetList gadgets, int n)
{
  int i;

  for (i = 0; i < n; i++) {
    Widget gadget = gadgets[i];

    if (XtIsManaged(gadget) && XtClass(gadget) == xmIconGadgetClass) {
      Widget parent_entry;

      XtVaGetValues(gadget, XmNentryParent, &parent_entry, NULL);
      if (parent_entry == w) {
	DeleteSubtree(gadget, gadgets, n);
	XtDestroyWidget(gadget);
      }
    }
  }
}
    
static void DeleteChilds(Widget w)
{
  int n;
  WidgetList gadgets;

  XtVaGetValues(tree_scroll, XmNnumChildren, &n, XmNchildren, &gadgets, NULL);
  DeleteSubtree(w, gadgets, n);
}

/* change outline state in tree pane */

void Expand(Widget w)
{
  Dimension save_min, save_max, save_vert, save_horiz;
  DirPtr dir;
  int i;

  SaveScrollBarValues(tree_pane, &save_vert, &save_horiz);
  FreezePaneGeom(tree_pane, &save_min, &save_max);
  XtUnmanageChild(tree_scroll);
  DeleteChilds(w);
  XtVaGetValues(w, XmNuserData, &dir, NULL);
  for (i = 0; i < dirNSubdirs(dir); i++)
    CreateTreeGadgets(tree_scroll, w, dirSubdir(dir, i));
  XtVaSetValues(w, XmNoutlineState, XmEXPANDED, NULL);
  XtManageChild(tree_scroll);
  UnfreezePaneGeom(tree_pane, save_min, save_max);
  RestoreScrollBarValues(tree_pane, save_vert, save_horiz);
  UpdateStatusLine();
  if (keyboardFocusPolicy == XmEXPLICIT)
    XmProcessTraversal(w, XmTRAVERSE_CURRENT);
}

void Collapse(Widget w)
{
  Dimension save_min, save_max, save_vert, save_horiz;
  DirPtr dir;

  SaveScrollBarValues(tree_pane, &save_vert, &save_horiz);
  FreezePaneGeom(tree_pane, &save_min, &save_max);
  XtUnmanageChild(tree_scroll);
  DeleteChilds(w);
  XtVaGetValues(w, XmNuserData, &dir, NULL);
  if (dirHasSubdirs(dir)) {
    Widget subdir_entry = CreateDummyGadget(tree_scroll, "subdir", NULL);
    XtVaSetValues(subdir_entry, XmNentryParent, w, NULL);
  }
  XtVaSetValues(w, XmNoutlineState, XmCOLLAPSED, NULL);
  XtManageChild(tree_scroll);
  UnfreezePaneGeom(tree_pane, save_min, save_max);
  UpdateStatusLine();
  if (keyboardFocusPolicy == XmEXPLICIT)
    XmProcessTraversal(w, XmTRAVERSE_CURRENT);
}

/* status line */

void UpdateStatusLine(void)
{
  update_status();
  if (*status) {
    XmString labelstr;

    if (!XtIsManaged(status_line))
      XtManageChild(status_line);
    labelstr = XmStringCreateLocalized(status);
    XtVaSetValues(message,
		  XmNlabelString, labelstr,
		  XmNchildHorizontalAlignment, XmALIGNMENT_BEGINNING,
		  NULL);
    XmStringFree(labelstr);
    UpdateDeviceIndicators();
  } else {
    if (XtIsManaged(status_line))
      XtUnmanageChild(status_line);
  }
}

void RestoreStatusLine(void)
{
  XmString labelstr;

  labelstr = XmStringCreateLocalized(status);
  XtVaSetValues(message,
		XmNlabelString, labelstr,
		XmNchildHorizontalAlignment, XmALIGNMENT_BEGINNING,
		NULL);
  XmStringFree(labelstr);
}

void StatusMessage(String msg)
{
  XmString labelstr;

  labelstr = XmStringCreateLocalized(msg);
  XtVaSetValues(message,
		XmNlabelString, labelstr,
		XmNchildHorizontalAlignment, XmALIGNMENT_BEGINNING,
		NULL);
  XmStringFree(labelstr);
}

/* initialize the application */

void init_app(int argc, char **argv)
{
  XtSetLanguageProc(NULL, NULL, NULL);
  app_shell = XtAppInitialize(&app,
			      "Xplore",
			      options, XtNumber(options),
			      &argc, argv,
			      fallback_resources,
			      (ArgList) NULL, 0);
  display = XtDisplay(app_shell);

  /* make sure we can close-on-exec the display connection */
  if (fcntl(ConnectionNumber(display), F_SETFD, 1) == -1)
    fatal("Couldn't mark display connection as close-on-exec");

#if (XtSpecificationRelease > 4)
#ifndef _NO_XMU

  /* add support for editres messages */
  XtAddEventHandler(app_shell, (EventMask)0, True, _XEditResCheckMessages,
		    NULL);

#endif
#endif

  /* register converters for application-specific resources */

  XtAppAddConverter(app, XmRString, "ViewType", StringToViewType,
		    NULL, 0);
  XtAppAddConverter(app, XmRString, "SortType", StringToSortType,
		    NULL, 0);

  /* get application-specific resource values */

  XtGetApplicationResources(app_shell, &resdata, resources,
			    XtNumber(resources), NULL, 0);

  /* retrieve global settings from resources */

  XtVaGetValues(app_shell, XmNkeyboardFocusPolicy, &keyboardFocusPolicy, NULL);

  abspath(rootdirname, basedir, resdata.rootdir);
  abspath(curdirname, basedir, resdata.curdir);
  abspath(shelfdirname, basedir, resdata.shelfdir);

  abspath(magicfile, basedir, resdata.magicfile);
  abspath(configfile, basedir, resdata.configfile);
  abspath(resfile, basedir, resdata.resfile);
  rescmd = resdata.rescmd;
  curshelf = resdata.curshelf;
  cpp_options = resdata.cppoptions;
  iconpath = resdata.iconpath;
  sh_list = resdata.bourneShells;
  metachars = resdata.metachars;

  show_getwd = resdata.getwd;

  update_time = resdata.update;
  updates = resdata.updates && update > 0;

  viewtype = resdata.view;
  treepane = resdata.tree;
  filepane = resdata.file;
  shelfpane = resdata.shelf;

  stdoptions = 0;

  if (resdata.rescan)
    stdoptions |= CHECK_FILES;
  
  switch (resdata.sort) {
  case BySize:
    stdoptions |= SORT_BY_SIZE;
    break;
  case ByDate:
    stdoptions |= SORT_BY_DATE;
    break;
  default:
    stdoptions |= SORT_BY_NAME;
  }

  if (resdata.updir)
    stdoptions |= INCLUDE_UPDIR;
  if (resdata.hidden)
    stdoptions |= INCLUDE_HIDDEN;
  if (resdata.files)
    stdoptions |= INCLUDE_FILES;
  if (resdata.dirs)
    stdoptions |= INCLUDE_DIRS;

  if (resdata.reverse)
    stdoptions |= REVERSE;
  if (resdata.dirsfirst)
    stdoptions |= DIRS_FIRST;
  if (!resdata.magic)
    stdoptions |= NOMAGIC;

  check = resdata.check;
  absolute = resdata.absolute;
  dirtarget = resdata.dirtarget;
  pushdir = resdata.pushdir;
  echo = resdata.echo;
  backups = resdata.backups;

  confirm_drop = resdata.drop;
  confirm_move = resdata.move;
  confirm_copy = resdata.copy;
  confirm_link = resdata.link;
  confirm_delete = resdata.delete;
  confirm_deldir = resdata.deldir;
  confirm_delfile = resdata.delfile;
  confirm_overwrt = resdata.overwrt;
  confirm_quit = resdata.quit;

  /* set the multi-click time */

  if (resdata.multiclick)
    XtSetMultiClickTime(display, resdata.multiclick);

  /* initialize wait cursor */
  wait_cursor = XCreateFontCursor(display, XC_watch);

  /* parse command line parameters */

  if (argc)
    progname = argv[0];
  else
    progname = "xplore";
  
  if (argc >= 2)
    abspath(curdirname, basedir, argv[1]);

}

/* create application widgets */

void create_app(void)
{
  int i;
  DirPtr ret;
  Atom WM_DELETE_WINDOW;

  /* if the shelf directory does not exist, presumably xplore is invoked
     for the first time; run the xplore-setup script */

  if (!exists(shelfdirname))
    fileExec(libsetup, NULL, NULL);

  /* try to create shelf directory and its subdirectories if they still do not
     exist */

  if (!exists(shelfdirname))
    mkdir(shelfdirname, user_umask & 0777);
  if (!chdir(shelfdirname))
    for (i = 0; i < n_shelves(); i++)
      if (shelf_label(i)) {
	String path = shelf_dir(i)?shelf_dir(i):shelf_label(i);
	if (!exists(path))
	  mkdir(path, user_umask & 0777);
      }

  /* set current shelf from resource value if possible */

  if (curshelf)
    for (i = 0; i < n_shelves(); i++)
      if (shelf_label(i) && !strcmp(curshelf, shelf_label(i))) {
	default_shelf = i;
	break;
      }

  /* read in initial directory information */

  cur_init(rootdirname, curdirname, stdoptions);
  shelf_init(shelfdirname, shelfoptions);

  /* parse translations and register drag action routines */

  tree_trans = XtParseTranslationTable(tree_trans_s);
  file_trans = XtParseTranslationTable(file_trans_s);
  XtAppAddActions(app, drag_actions, XtNumber(drag_actions));

  /* create the main window */

  main_window = XtVaCreateManagedWidget("main_window",
					xmMainWindowWidgetClass, app_shell,
					NULL);

  /* create the menu bar */
  
  menubar = XmCreateMenuBar(main_window, "menubar", NULL, 0);

  CreateCommandMenuDesc();

  file_menu_desc[COMMAND].submenu = command_menu_desc;
  file_menu_desc[COMMAND].submenu_size = command_menu_size;

  CreateMenu(menubar, MENU_PULLDOWN, "File", NULL, NULL, NULL,
	     file_menu_desc, XtNumber(file_menu_desc),
	     FileMapCB, FileUnmapCB,
	     "FilePulldown", &file_pulldown,
	     "File", &file_button);
  CreateMenu(menubar, MENU_PULLDOWN, "Shelf", NULL, NULL, NULL,
	     shelf_menu_desc, XtNumber(shelf_menu_desc),
	     ShelfMapCB, ShelfUnmapCB,
	     "ShelfPulldown", &shelf_pulldown,
	     "Shelf", &shelf_button);
  CreateMenu(menubar, MENU_PULLDOWN, "View", NULL, NULL, NULL,
	     view_menu_desc, XtNumber(view_menu_desc),
	     ViewMapCB, ViewUnmapCB,
	     "ViewPulldown", &view_pulldown,
	     "View", &view_button);
  CreateMenu(menubar, MENU_PULLDOWN, "Options", NULL, NULL, NULL,
	     options_menu_desc, XtNumber(options_menu_desc),
	     OptionsMapCB, OptionsUnmapCB,
	     "OptionsPulldown", &options_pulldown,
	     "Options", &options_button);
  CreateMenu(menubar, MENU_PULLDOWN, "Help", NULL, NULL, NULL,
	     help_menu_desc, XtNumber(help_menu_desc),
	     HelpMapCB, UnmapCB,
	     "HelpPulldown", &help_pulldown,
	     "Help", &help_button);

  XtVaSetValues(menubar, XmNmenuHelpWidget, help_button, NULL);

  XtManageChild(menubar);

  /* create the work area */

  CreateForm(main_window);

  /* create the status line */

  CreateStatusLine(main_window);

  /* set the application icon and install WM_DELETE_WINDOW handler */

  XtVaSetValues(app_shell, XmNiconPixmap, std_icons[ICON_XPLORE].pixmap,
		XmNiconMask, std_icons[ICON_XPLORE].mask,
		XmNdeleteResponse, XmDO_NOTHING, NULL);

  WM_DELETE_WINDOW = XmInternAtom(display, "WM_DELETE_WINDOW", False);
  XmAddWMProtocolCallback(app_shell, WM_DELETE_WINDOW, DeleteWindowCB, NULL);

}

/* run the application */

XtIntervalId timer_id;

void run_app(void)
{

  /* realize the shell */

  XtRealizeWidget(app_shell);
  /* adjust the width of the shelf and the file scroll */
  if (shelfpane)
    Resize(shelf_scroll);
  if (treepane)
    Resize(tree_scroll);
  if (filepane)
    Resize(file_scroll);

  /* initialize the status line */

  UpdateStatusLine();

  /* initialize the update cycle */

  if (updates)
    timer_id = XtAppAddTimeOut(app, update_time, TimerCB, NULL);

  /* show the current directory in tree pane */

  XmScrollVisible(tree_pane, curgadget, 20, 20);
  XmProcessTraversal(curgadget, XmTRAVERSE_CURRENT);
  XtVaSetValues(tree_scroll, XmNinitialFocus, curgadget, NULL);

  /* enter the event loop */

  XtAppMainLoop(app);

}

/* terminate application */

void exit_app(int val)
{
  XtDestroyApplicationContext(app);
  exit(val);
}

/* reinitialize */

void reinit(void)
{
  int i, n;
  WidgetList buttons;

  if (!exists(shelfdirname))
    mkdir(shelfdirname, user_umask & 0777);
  if (!chdir(shelfdirname))
    for (i = 0; i < n_shelves(); i++)
      if (shelf_label(i)) {
	String path = shelf_dir(i)?shelf_dir(i):shelf_label(i);
	if (!exists(path))
	  mkdir(path, user_umask & 0777);
      }
  XtVaGetValues(command_pulldown, XmNnumChildren, &n, XmNchildren, &buttons,
		NULL);
  XtUnmanageChildren(buttons, n);
  for (i = 0; i < n; i++)
    XtDestroyWidget(buttons[i]);
  CreateCommandMenuDesc();
  CreateMenuButtons(command_pulldown, command_menu_desc,
		    command_menu_size);
  shelf_init(shelfdirname, shelfoptions);
  UpdateShelfTabs(form);
  CreateAttachments();
  CreateShelfScroll(shelf_pane);
  CreateTreeScroll(tree_pane);
  CreateFileScroll(file_pane);
  ResetScrollBarValues(shelf_pane);
  ResetScrollBarValues(tree_pane);
  ResetScrollBarValues(file_pane);
  if (shelfpane)
    Resize(shelf_scroll);
  if (treepane)
    Resize(tree_scroll);
  if (filepane)
    Resize(file_scroll);
  CreateDeviceIndicators(indicators);
  UpdateStatusLine();
  if (treepane) {
    XmScrollVisible(tree_pane, curgadget, 20, 20);
    XmProcessTraversal(curgadget, XmTRAVERSE_CURRENT);
    XtVaSetValues(tree_scroll, XmNinitialFocus, curgadget, NULL);
  }
}
