/* --------------------------------- getstr.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* User input with editing. This is built around C code from a mag (forget
 * which one) but the original author's note follows.
*/

/* Get input string, with VMS-style input line editing
 * and previous-command scrolling.
 *
 * Written for Turbo C 2.0 / Borland C++ 2.0
 * Bob Bybee, 2/91
*/

/* Modified for usual msdos keys,
 * Eyal Lebedinsky, Feb 1992
 *
 * Use microsoft C. Original program compiled and run without any changes
 *   with this compiler.
 * Dont loose strings that look like part of an escape sequence [push_back].
 * Show insert/overtype mode as cursor size[sys_show].
 * Initial mode is overwrite!
 * User can select length for shortest string kept [shortest_saved].
 * If an earlier line is selected (with up/down arrows) and entered then it
 *   is not saved again [last_recalled].
 * Provide history search with PAGE_UP key: Previous entry that matches up
 *   to the cursor is retrieved.
 * Added keys:
 *   HOME		move to start of line (CTRL-B)
 *   END		move to end   of line (CTRL-E)
 *   ESC		clear line (CTRL-X)
 *   INSERT		toggle mode (CTRL-A)
 * Added keys with new functions:
 *   C_RIGHT_ARROW	move to next word start
 *   C_LEFT_ARROW	move to prev word start
 *   DEL		delete char at right of cursor
 *   C_END		delete right of cursor to end of line
 *   C_HOME		delete left of cursor to start of line
 *   PAGE_UP		search line prefix
*/

/* Modified for fly8
 * Eyal Lebedinsky, Nov 1992
 *
 * Removed escape-sequences for input.
 * Removed init function.
 * Modified to use fly8 key names.
 * Modified to interact with fly8 input/output.
*/

#include "fly.h"


/* ASCII key definitions
*/
#define ESC_KEY			K_ESC
#define DELETE_KEY		K_DEL
#define BACKSPACE_KEY		K_RUBOUT
#define RETURN_KEY 		K_ENTER
#define CTRL(x)			((x) & 0x1f)

/* Values for tracking cursor key entry
*/
#define UP_ARROW	K_UP
#define DOWN_ARROW	K_DOWN
#define RIGHT_ARROW	K_RIGHT
#define LEFT_ARROW	K_LEFT
#define HOME		K_HOME
#define END		K_END
#define INSERT		K_INS
#define C_LEFT_ARROW	(K_LEFT  | K_CTRL)
#define C_RIGHT_ARROW	(K_RIGHT | K_CTRL)
#define C_END		(K_END   | K_CTRL)	
#define C_HOME		(K_HOME  | K_CTRL)	
#define Del		K_DEL	
#define PAGE_UP		K_PGUP	

/* MAX_RECALL is two greater than the number of lines
 * we want in the "lastlines" recall buffer.
*/
#define MAX_RECALL	22
#define RECSUB(x)	(((x) + MAX_RECALL) % MAX_RECALL)
#define RECALL_LEN	100
static char	lastlines[MAX_RECALL][RECALL_LEN] = {{0}};
static int	push_back[10] = {0}, push_back_index = 0;

#define	SHORTEST_SAVED	3		/* don't save short strings */
#define INIT_MODE	1		/* 1=insert 0=overwrite */

static int	num_got = 0;		/* # chars in input buffer */
static int	cursor_pos = 0;		/* cursor position on line */
static int	lastptr = 0;		/* ptr to last line entered */
static char	FAR* str_ptr = 0;	/* ptr to current input string */
static int	insert_mode = INIT_MODE;
static char	FAR* Prompt = 0;


/* prototypes for this file */
static void	NEAR clear_line (void);
static int	NEAR cursor_right (void);
static int	NEAR cursor_left (void);
static int	NEAR get_char_esc (void);
static void	NEAR put_str (char FAR* str);

/* Put a character to the output device.
*/
static void NEAR
sys_put (int ch)
{
	if ('\a' == ch && st.quiet)
		Snd->Beep (880, 100);
}

/* Get a character from the input device.
*/
static int NEAR
sys_get (void)
{
	return (mgetch ());
}

/* Set cursor shape to indicate insert/overtype mode.
*/
static void NEAR
sys_show (int insert_mode)
{}

/* display the prompt and the user input if getstr is active.
*/
extern void FAR
show_prompt (VIEW *view, int orgx, int orgy, int maxx, int maxy, int bss)
{
	int	i, x, dx, xl, xr, y, tick;
	char	*p;

	if (!str_ptr)
		return;

	dx = stroke_size ("x", bss);
	tick = bss/2;

	xl = orgx - maxx + bss + 2;
	xr = orgx + maxx - bss - 2;
	y = orgy + maxy - 4;

	add_line (xl-2,   y+2,     T_MOVE);
	add_line (xr+bss, y+2,     st.lgray);
	add_line (xr+bss, y-1-bss, st.lgray);
	add_line (xl-2,   y-1-bss, st.lgray);
	add_line (xl-2,   y+2,     st.lgray);

	x = xl;
	if (Prompt) {
		for (p = Prompt; *p; ++p)
			x += stroke_char (x, y, *p,  bss, st.lgray);
			x += dx;
	}
	for (i = 0;; ++i) {
		if (i == cursor_pos) {
			add_line (x, y+3, T_MOVE);
			add_line (x, y-2-bss, st.white);
			if (insert_mode) {
				add_line (x-tick, y-2-bss-tick, T_MOVE);
				add_line (x,      y-2-bss,      st.white);
				add_line (x+tick, y-2-bss-tick, st.white);
			}
		}
		if (i >= num_got)
			break;
		if (x > xr)
			break;
		x += stroke_char (x, y, str_ptr[i], bss, st.white);
	}
}

/* getstr() is called to get a line of input.
 * The input line is placed in "str" and will be no
 * more than "len" characters.
*/
extern void FAR
getstr (char *prompt, char FAR* str, int len)
{
	int i, c, curptr, last_recalled = -1;

	Prompt = prompt;
	push_back_index = 0;
	sys_show (insert_mode = INIT_MODE);
	num_got = 0;
	cursor_pos = 0;
	str_ptr = str;				/* copy the buffer pointer */
	curptr = RECSUB(lastptr + 1);
	lastlines[curptr][0] = '\0';
	lastlines[RECSUB(curptr + 1)][0] = '\0';
	if (len > RECALL_LEN - 1)		/* limit len to RECALL_LEN */
		len = RECALL_LEN - 1;

	for (;;) {
		c = get_char_esc();
		if (c == RETURN_KEY)
			break;
		else if (c == DELETE_KEY || c == BACKSPACE_KEY || c == Del) {
			if ((c == Del && cursor_pos < num_got) ||
			    (c != Del && cursor_left())) {
				++cursor_pos;
				for (i = cursor_pos; i < num_got; ++i) {
					str[i - 1] = str[i];
					(*sys_put)(str[i]);
				}
				(*sys_put)(' ');
				for (i = cursor_pos; i <= num_got; ++i)
					(*sys_put)('\b');
				--num_got;
				--cursor_pos;
			}
		} else if (c == C_END) {
			if (cursor_pos < num_got) {
				for (i = cursor_pos; i < num_got; ++i) {
					str[i] = ' ';
					(*sys_put)(' ');
				}
				for (i = cursor_pos; i < num_got; ++i)
					(*sys_put)('\b');
				num_got = cursor_pos;
			}
		} else if (c == C_HOME) {
			if (cursor_pos > 0) {
				for (i = 0; i < cursor_pos; ++i)
					(*sys_put)('\b');
				num_got -= cursor_pos;
				for (i = 0; i < num_got; ++i) {
					str[i] = str[cursor_pos+i];
					(*sys_put)(str[i]);
				}
				for (i = 0; i < cursor_pos; ++i)
					(*sys_put)(' ');
				for (i = cursor_pos + num_got; i-- > 0;)
					(*sys_put)('\b');
				cursor_pos = 0;
			}
		} else if (c == CTRL('X') ||	/* erase line? */
			 c == ESC_KEY)
			clear_line();
		else if (c == CTRL('A') ||	/* insert/overtype? */
			 c == INSERT)
			sys_show (insert_mode ^= 1);
		else if (c == CTRL('B') ||	/* beginning-of-line? */
			 c == HOME)
			{
			while (cursor_left())
				;
			}
		else if (c == CTRL('E') ||	/* end-of-line? */
			 c == END)
			{
			while (cursor_right())
				;
			}
		else if (c == CTRL('R'))	/* recall last line? */
			{
			clear_line();
			strcpy(str, lastlines[lastptr]);
			if ((num_got = strlen(str)) > 0)
				{
				put_str(str);
				last_recalled = lastptr;
				break;
				}
			}
		else if (c == UP_ARROW)
			{
			clear_line();
			if (lastlines[curptr][0] != '\0' ||
				lastlines[RECSUB(curptr - 1)][0] != '\0')
				{
				curptr = RECSUB(curptr - 1);
				strcpy(str, lastlines[curptr]);
				last_recalled = curptr;
				put_str(str);
				cursor_pos = num_got = strlen(str);
				}
			}
		else if (c == DOWN_ARROW)
			{
			clear_line();
			if (lastlines[curptr][0] != '\0' ||
				lastlines[RECSUB(curptr + 1)][0] != '\0')
				{
				curptr = RECSUB(curptr + 1);
				strcpy(str, lastlines[curptr]);
				last_recalled = curptr;
				put_str(str);
				cursor_pos = num_got = strlen(str);
				}
			}
		else if (c == LEFT_ARROW)
			{
			if (cursor_pos > 0)
				{
				(*sys_put)('\b');
				--cursor_pos;
				}
			}
		else if (c == RIGHT_ARROW)
			cursor_right();
		else if (c == C_LEFT_ARROW) {
			while (cursor_pos > 0 && str[cursor_pos-1] == ' ') {
				(*sys_put)('\b');
				--cursor_pos;
			}
			while (cursor_pos > 0 && str[cursor_pos-1] != ' ') {
				(*sys_put)('\b');
				--cursor_pos;
			}
		} else if (c == C_RIGHT_ARROW) {
			while (cursor_pos < num_got &&
			       str[cursor_pos] != ' ')
				cursor_right();
			while (cursor_pos < num_got &&
			       str[cursor_pos] == ' ')
				cursor_right();
		} else if (c == PAGE_UP) {
			for (i = curptr; (i = RECSUB(i-1)) != curptr;) {
				if (memcmp (str, lastlines[i], cursor_pos))
					continue;
				c = cursor_pos;
				clear_line();
				strcpy(str, lastlines[i]);
				if ((num_got = strlen(str)) > 0) {
					curptr = last_recalled = i;
					put_str(str);
					cursor_pos = num_got;
					while (cursor_pos > c)
						cursor_left ();
				}
				c = ~PAGE_UP;
				break;
			}
			if (c == PAGE_UP)
				(*sys_put) ('\a');
		} else if (' ' <= c && c < 0x7f && num_got < len - 1)
			{
			if (insert_mode)
				{
				/* Move right, all the characters
				 * to the right of cursor_pos.
				 */
				for (i = num_got; i > cursor_pos; --i)
					str[i] = str[i - 1];
				str[cursor_pos] = (char)c;
				for (i = cursor_pos; i <= num_got; ++i)
					(*sys_put)(str[i]);
				for (i = cursor_pos; i < num_got; ++i)
					(*sys_put)('\b');
				++num_got;
				++cursor_pos;
				}
			else	/* insert is off, use overtype mode */
				{
				str[cursor_pos] = (char)c;
				(*sys_put)(c);
				if (cursor_pos == num_got)
					++num_got;
				++cursor_pos;
				}
			}
		}

	str[num_got] = '\0';
	(*sys_put)('\n');

	/* If this line is non-empty, and different
	 * from the last one accepted, store it into
	 * the recall buffer.
	 */
	if (num_got >= SHORTEST_SAVED && strcmp(str, lastlines[lastptr]) &&
	    (last_recalled < 0 || strcmp(str, lastlines[last_recalled]))) {
		lastptr = RECSUB(lastptr + 1);
		strcpy(lastlines[lastptr], str);
	}
	Prompt = 0;
	str_ptr = 0;
}

/* Move the cursor right one position, by echoing the
 * character it's currently over.
 * Return 1-OK, 0-can't move.
*/
static int NEAR
cursor_right (void)
{
	if (cursor_pos < num_got) {
		(*sys_put)(str_ptr[cursor_pos]);
		++cursor_pos;
		return (1);
	}
	return (0);
}

/* Move the cursor left one position, by echoing
 * a backspace.  Return 1-OK, 0-can't move.
*/
static int NEAR
cursor_left (void)
{
	if (cursor_pos > 0) {
		(*sys_put)('\b');
		--cursor_pos;
		return (1);
	}
	return (0);
}

/* Erase all characters on the current line.
*/
static void NEAR
clear_line (void)
{
	int	i;

	while (cursor_left())		/* move to begining of line */
		;

	for (i = 0; i < num_got; ++i)
		(*sys_put)(' ');
	for (i = 0; i < num_got; ++i)
		(*sys_put)('\b');
	cursor_pos = num_got = 0;
}

/* Get a character, with escape processing.
 * Handles special sequences like "ESC [ A" for up-arrow.
 * This function would need to be modified to handle
 * keyboards that are neither PC's nor VT-100's.
*/
static int NEAR
get_char_esc (void)
{
	int	ch;

	if (push_back_index > 0)
		return (push_back[--push_back_index]);

	ch = (*sys_get)();
	if (ch != ESC_KEY)
		return (ch);

	return (ch);	/* no escapes! */
}

/* Put a string to (*sys_put)().
*/
static void NEAR
put_str (char FAR* str)
{
	while (*str != '\0')
		(*sys_put)(*str++);
}

#undef ESC_KEY
#undef DELETE_KEY
#undef BACKSPACE_KEY
#undef RETURN_KEY
#undef CTRL
#undef UP_ARROW
#undef DOWN_ARROW
#undef RIGHT_ARROW
#undef LEFT_ARROW
#undef HOME
#undef END
#undef INSERT
#undef C_LEFT_ARROW
#undef C_RIGHT_ARROW
#undef C_END
#undef C_HOME
#undef Del
#undef PAGE_UP
#undef MAX_RECALL
#undef RECSUB
#undef RECALL_LEN
#undef SHORTEST_SAVED
#undef INIT_MODE
