/* --------------------------------- memory.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Memory manager.
*/

#include "fly.h"


typedef struct block	BLOCK;
struct block {
	BLOCK	*next;
};

#define	MAXBSIZE	256			/* max size is MAXSIZE*GRAIN */
#define	GRAIN		4

static BLOCK	**blocks = 0;
static long	ngrains = 0;			/* stats */
static long	*nblocks = 0;			/* alloced - freed */
static int	malloc_dead = 0;		/* malloc failed! */

extern void * FAR
xcalloc (Uint count, Uint size)
{
	Ulong	flags;
	void	*p;

	if (malloc_dead)
		return (0);
	flags = Sys->Disable ();
	p = calloc (count, size);
	Sys->Enable (flags);
	if (!p)
		malloc_dead = 1;

	return (p);
}

extern void * FAR
xmalloc (Uint size)
{
	Ulong	flags;
	void	*p;

	if (malloc_dead)
		return (0);
	flags = Sys->Disable ();
	p = malloc (size);
	Sys->Enable (flags);
	if (!p)
		malloc_dead = 1;
	return (p);
}

extern void * FAR
xfree (void *block)
{
	Ulong	flags;

	if (!block)
		return (0);

	flags = Sys->Disable ();
	free (block);
	Sys->Enable (flags);
	malloc_dead = 0;

	return (0);
}

#define	CHUNKSIZE	(8*1024)
#define	LOW_MEMORY	(CHUNKSIZE/4)

typedef struct chunk	CHUNK;
struct chunk {
	CHUNK	*next;
	Uint	avail;
	Uchar	*mem;
	Uchar	buff[CHUNKSIZE];
};

static CHUNK	*chunks = 0;

/* 'bytes' already rounded and limited!
*/
static void * NEAR
my_malloc (Uint bytes)
{
	register CHUNK	*p;
	void		*b;

/* best-fit or first-fit? for speed, we use first-fit.
*/
	for (p = chunks; p; p = p->next)
		if (p->avail >= bytes)
			break;
	if (!p) {
		if (malloc_dead)
			return (0);
		if (F(p = (CHUNK *)malloc (sizeof (*p)))) {
			malloc_dead = 1;
			return (0);
		}
		p->next = chunks;
		chunks = p;
		p->avail = sizeof (p->buff);
		p->mem = p->buff;
	}
	b = p->mem;
	p->mem += bytes;
	p->avail -= bytes;
	return (b);
}

/* Ensure we are not low on memory.
*/
extern void FAR
memory_check (void)
{
	CHUNK	*p, *m;
	Ulong	flags;

	if (malloc_dead)
		return;

	for (m = p = chunks; p; p = p->next)
		if (p->avail > m->avail)
			m = p;

	if (!m || m->avail < LOW_MEMORY) {
		flags = Sys->Disable ();
		if (T(p = (CHUNK *)malloc (sizeof (*p)))) {
			p->next = chunks;
			chunks = p;
			p->avail = sizeof (p->buff);
			p->mem = p->buff;
		} else
			malloc_dead = 1;
		Sys->Enable (flags);
	}
}

extern void * FAR
memory_alloc (Uint bytes)
{
	BLOCK	*p;
	Uint	i;
	Ulong	flags;

	if (!bytes || !blocks)
		return (0);

	if (bytes < sizeof (BLOCK))
		bytes = sizeof (BLOCK);
	i = (bytes-1) / GRAIN;

	if (i >= MAXBSIZE) {
		p = xmalloc (bytes);
		goto ret;
	}

	flags = Sys->Disable ();
	if (T(p = blocks[i]))
		blocks[i] = p->next;
	else if (T(p = (BLOCK *)my_malloc ((i+1)*GRAIN)))
		ngrains += i+1;
	Sys->Enable (flags);
	if (p)
		++nblocks[i];
ret:
	if (p)
		memset (p, 0, bytes);
	return ((void *)p);
}

extern void * FAR
memory_free (void *block, int bytes)
{
	BLOCK	*p;
	Ulong	flags;
	int	i;

	if (!block)
		return (0);

	if (bytes) {
		if (bytes < sizeof (BLOCK))
			bytes = sizeof (BLOCK);
		i = (bytes-1) / GRAIN;
	} else
		i = 0;		/* avoid compiler warning */

	p = ((BLOCK *)block)->next;

	flags = Sys->Disable ();
	if (!bytes || i >= MAXBSIZE || !blocks)
		xfree (block);
	else {
		((BLOCK *)block)->next = blocks[i];
		blocks[i] = (BLOCK *)block;
		--nblocks[i];
	}
	Sys->Enable (flags);
	if (nblocks[i] < 0)
		MsgEPrintf (-10, "bad free (%u)", bytes);

	return (p);
}

extern int FAR
memory_init (void)
{
	ngrains = 0;
	if (F(blocks = (BLOCK **)xcalloc (MAXBSIZE, sizeof (BLOCK *))))
		return (1);
	if (F(nblocks = (long *)xcalloc (MAXBSIZE, sizeof (long)))) {
		blocks = xfree (blocks);
		return (1);
	}

	chunks = 0;
	memory_check ();

	return (0);
}

extern void FAR
memory_term (void)
{
	BLOCK	*p;
	CHUNK	*c, *c1;
	int	i, n;
	long	tt, avail, freed;

	if (!blocks)
		return;

	freed = 0;
	for (i = 0; i < MAXBSIZE; ++i) {
		n = 0;
		for (p = blocks[i]; p; p = p->next)
			++n;
		if (n || nblocks[i]){
			tt = (i+1)*(long)GRAIN*n;
			freed += tt;
			LogPrintf ("Mem[%u] %u %lu", (i+1)*GRAIN, n, tt);
			if (nblocks[i])
				LogPrintf (" (%ld)\n", nblocks[i]);
			else
				LogPrintf ("\n");
		}
	}

	n = 0;
	avail = 0;
	for (c1 = chunks; T(c = c1);) {
		c1 = c->next;
		avail += c->avail;
		xfree (c);
		++n;
	}
	chunks = 0;
	LogPrintf ("Chunks   %lu (%u)\n", n*(long)CHUNKSIZE, n);
	LogPrintf ("Unused   %lu\n", avail);
	LogPrintf ("Alloc'ed %lu\n", ngrains*GRAIN);
	LogPrintf ("Freed    %lu\n", freed);
	LogPrintf ("Leakage  %ld\n", ngrains*GRAIN-freed);

	blocks = xfree (blocks);
	nblocks = xfree (nblocks);
	ngrains = 0;
}

extern int FAR
memory_get (int bytes, int count)
{
	BLOCK	*p;
	Uint	i;
	int	n;
	Ulong	flags;

	if (!bytes || !blocks)
		return (0);

	if (bytes < sizeof (BLOCK))
		bytes = sizeof (BLOCK);
	i = (bytes-1) / GRAIN;

	if (i >= MAXBSIZE)
		return (0);			/* nothing to do */

	for (n = 0; n < count; ++n) {
		flags = Sys->Disable ();
		if (T(p = (BLOCK *)malloc ((i+1)*GRAIN))) {
			p->next = blocks[i];
			blocks[i] = p;
			ngrains += i+1;
			Sys->Enable (flags);
		} else {
			Sys->Enable (flags);
			break;
		}
	}

	return (n);
}

extern char * FAR
xstrdup (const char *s)
{
	char	*p;
	Ulong	flags;

	flags = Sys->Disable ();
	p = strdup (s);
	Sys->Enable (flags);

	return (p);
}

#if 0
extern void * FAR
list_del (void *member, void **head, void **tail)
{
	BLOCK	*p, *prev;

	if (!member || !head)
		return (0);

	for (prev = 0, p = *(BLOCK **)head; p; prev = p, p = p->next) {
		if (p == (BLOCK *)member) {
			p = p->next;
			if (prev)
				prev->next = p;
			else
				*head = p;
			if (!p && tail)
				*(BLOCK **)tail = prev;
			break;
		}
	}

	return (p);
}

extern void FAR
fifo_put (void *head, void *new)
{
	void	**p;

	for (p = (void **)head; *p; p = (void **)*p)
		;
	*p = new;
	*(void **)new = 0;
}

extern void * FAR
fifo_get (void *head)
{
	void	*p;

	p = *(void **)head;
	if (p)
		*(void **)head = *(void **)p;
	return (p);
}
#endif

#undef MAXBSIZE
#undef GRAIN
#undef CHUNKSIZE
#undef LOW_MEMORY

