/* util.c
 *
 * Miscellaneous functions used by both the client and server
 */
#include "net3d.h"

/* caches for jsin and jcos calls */
static struct cache *sc;
static struct cache *cc;
#define SCSIZE 5
#define CCSIZE 5

/* data on cache efficiency */
int sincalls=0;
int sincachehits=0;

/* Returns the vehicle that va has collided with, drawn from the
 * vehicle list given by vh, or NULL if no collision.
 */
struct vehicle *collide(struct vehicle *va, struct vehicle *vh)
{
struct vehicle *vb;

/* Vehicles that are shrapnel, scenery or seedpods, or are already dead
 * cannot collide with anything.
 */
if (va->type == t_shrapnel || va->type == t_scenery || va->type == t_seedpod ||
 va->hp < 0)
	return NULL;

/* Loop through all other vehicles, looking for a collision.
 */
for(vb=vh; vb; vb=vb->next) {
	/* Once again, prevent collisions with scenery, shrapnel, seedpods
	 * or dead vehicles waiting to be freed.
	 */
	if (vb->type == t_shrapnel || vb->type == t_scenery ||
	 vb->type == t_seedpod || vb->hp < 0)
		continue;

	/* Also prevent collisions between a vehicle and itself, collisions
	 * with a vehicle's own ammo, and collisions between bullets.
	 */
	if (va == vb || va->firer == vb || vb->firer == va ||
	 (va->type == t_bullet && vb->type == t_bullet))
		continue;

	/* now actually compare bounding boxes.. */
	if ((va->bmin.x>=vb->bmin.x && va->bmin.x<=vb->bmax.x) ||
	 (va->bmax.x>=vb->bmin.x && va->bmax.x<=vb->bmax.x)) {
	  if ((va->bmin.y>=vb->bmin.y && va->bmin.y<=vb->bmax.y) ||
	   (va->bmax.y>=vb->bmin.y && va->bmax.y<=vb->bmax.y)) {
	    if ((va->bmin.z>=vb->bmin.z && va->bmin.z<=vb->bmax.z) ||
	     (va->bmax.z>=vb->bmin.z && va->bmax.z<=vb->bmax.z)) {
	/* we have a collision!
	 */
	      return vb;
	      }
	    }
	  }
	}

/* no collisions were found */
return NULL;
}

/* find the vehicle with id given by vid, in the list pointed to by
 * vhead.
 */
struct vehicle *findbyvid(struct vehicle *vhead, int vid)
{
struct vehicle *vp;

for(vp = vhead; vp && vp->vid != vid; vp=vp->next)
	;
return vp;
}

void initcaches(void)
{
sc = (struct cache *)calloc(SCSIZE,sizeof(struct cache));
cc = (struct cache *)calloc(CCSIZE,sizeof(struct cache));
}

double jsin(double a)
{
int i;
register int oldest;			/* best place to put the new value */
register long lowesttime;
static long call=1;

sincalls++;
call++;
lowesttime = call;
/* first, check the cache to see if the value is already in */
for(i=0; i<SCSIZE; i++) {
	if (sc[i].inuse && sc[i].val == a) {
		/* found in the cache! */
		sincachehits++;
		return sc[i].res;
		}
	if (!sc[i].inuse) {
		/* this cache line is empty.. use it */
		oldest = i;
		lowesttime = 0;
		}
	if (sc[i].lastused < lowesttime) {
		/* this is the oldest one found so far */
		lowesttime = sc[i].lastused;
		oldest = i;
		}
	}

/* the value wasn't found... store it in the cache */
sc[oldest].val = a;
sc[oldest].inuse = true;
sc[oldest].lastused = call;
return (sc[oldest].res = sin(a));
}

struct object *findturret(struct vehicle *v)
{
int i;

for(i=0; i<v->partcount; i++)
	if (v->parts[i]->turret)
		return v->parts[i];
return NULL;
}

void addpts(struct point p1, struct point p2, struct point *ret)
{
ret->x = p1.x + p2.x;
ret->y = p1.y + p2.y;
ret->z = p1.z + p2.z;
}

#if NO_STRERROR
char *strerror(int e)
{
static char buf[100];

sprintf(buf,"error %d",e);
return buf;
}
#endif

