## TFSA-2020-004: Out of bounds access in TFLite implementation of segment sum

### CVE Number
CVE-2020-15212

### Impact
In TensorFlow Lite models using segment sum can trigger [writes outside of
bounds of heap allocated
buffers](https://github.com/tensorflow/tensorflow/blob/0e68f4d3295eb0281a517c3662f6698992b7b2cf/tensorflow/lite/kernels/internal/reference/reference_ops.h#L2625-L2631)
by inserting negative elements in the segment ids tensor:
```cc
  for (int i = 0; i < input_shape.Dims(0); i++) {
    int output_index = segment_ids_data[i];
    for (int j = 0; j < segment_flat_size; ++j) {
      output_data[output_index * segment_flat_size + j] +=
          input_data[i * segment_flat_size + j];
    }
  }
```

Users having access to `segment_ids_data` can alter `output_index` and then
write to outside of `output_data` buffer.

This might result in a segmentation fault but it can also be used to further
corrupt the memory and can be chained with other vulnerabilities to create more
advanced exploits.

### Vulnerable Versions
TensorFlow 2.2.0, 2.3.0.

### Patches
We have patched the issue in
[204945b](https://github.com/tensorflow/tensorflow/commit/204945b) and will
release patch releases for all affected versions.

We recommend users to upgrade to TensorFlow 2.2.1, or 2.3.1.

### Workarounds
A potential workaround would be to add a custom `Verifier` to the model loading
code to ensure that the segment ids are all positive, although this only handles
the case when the segment ids are stored statically in the model.

A similar validation could be done if the segment ids are generated at runtime
between inference steps.

If the segment ids are generated as outputs of a tensor during inference steps,
then there are no possible workaround and users are advised to upgrade to
patched code.

### For more information
Please consult [our security
guide](https://github.com/tensorflow/tensorflow/blob/master/SECURITY.md) for
more information regarding the security model and how to contact us with issues
and questions.

### Attribution
This vulnerability has been discovered through a variant analysis of [a
vulnerability reported by members of the Aivul Team from Qihoo
360](https://github.com/tensorflow/tensorflow/blob/master/tensorflow/security/advisory/tfsa-2020-002.md).
