/*
 * PAM module for kerberized coda only.
 * This module should be used as "optional" in both "auth" and "session",
 * but NOT for authentication,
 * it just acquires a Coda token if there is a Kerberos one.
 *
 * ver 0.4, Sep 2002
 *  free() alloced memory
 *  changed argument name from "kclog" to "clog"
 *  run clog always with "-nokinit"
 *  added "realm" arguments
 * now the options accepted by auth are:
 *  ignore_root
 *  clog /path/to/clog/binary
 *  realm <realm> [realm <realm> ...]
 *
 * ver 0.3, Nov 2001
 *  moved token acuisition from open_session() to setcred()
 *  to suit existing pam framework for credentials setting
 * arguments (a minor change):
 *  auth: ignore_root, kclog /path/to/kclog/binary
 *  session: nocunlog, cunlog /path/to/cunlog/binary
 *
 * ver 0.2
 *  aaargh! memory allocation errors fixed
 *
 * ver 0.1
 * arguments:
 *  open_session: ignore_root, clog /path/to/kclog/binary
 *  close_session: nocunlog, cunlog /path/to/cunlog/binary
 *
 * it runs clog program at setcred() to aquire Coda tokens
 * and optionally cunlog at close_session() to destroy them.
 *
 * use kerberos pam module for authentication, then the things will work
 * transparently.
 *
 * Ivan Popov <pin@math.chalmers.se>, May 2001
 *
 * The module is inspired by and based on
 *
 * Pam Coda Module
 * ALPHA - V0.2 
 * Robin.Gareus@urz.uni-heidelberg.de
 * http://www.rzuser.uni-heidelberg.de/~x42/coda/index.html
 *
 * This Software is distributed under the terms of the GNU 
 * General Public License (GPL)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <grp.h>
#include <pwd.h>
#include <time.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/wait.h>
#include <sys/types.h>


#define CLOG_PROGRAM "/usr/local/bin/clog"
#define CUNLOG_PROGRAM "/usr/local/bin/cunlog"

#include <security/pam_modules.h>
#include <security/_pam_macros.h>

PAM_EXTERN
int pam_sm_authenticate(pam_handle_t * pamh, int flags,
                        int argc, const char **argv)
{
  return PAM_IGNORE;
}

PAM_EXTERN
int pam_sm_setcred(pam_handle_t * pamh, int flags,
                   int argc, const char **argv)
{
    int retval;
    const char *user = NULL;
    struct passwd *pw = NULL;
    /* Flags for options */
    int ignore_root = 0;
    pid_t code;
    int status, pipedes[2];
    char *execargs[16];
    char *clog_program;
    char *realms[16]; /* arbitrary limitation */
    int  nrealms = 0;
    char **en;
    int result = PAM_SUCCESS;

    if (flags != PAM_ESTABLISH_CRED)
      return PAM_CRED_ERR;

    retval = pam_get_user(pamh, &user, NULL);
    if (retval != PAM_SUCCESS)
        return PAM_USER_UNKNOWN;

    if (user == NULL || !isalnum(*user)) {
        if (user)
            syslog(LOG_ERR, "pam_kcoda: bad username [%s]", user);
        return PAM_USER_UNKNOWN;
    }

/* Get User Information */
    pw = getpwnam(user);

/* User does not exist */
    if (pw == NULL)
        return PAM_USER_UNKNOWN;

    clog_program = malloc ((strlen(CLOG_PROGRAM)+1)*sizeof(char));
    strcpy (clog_program,CLOG_PROGRAM);

/* Parse parameters for module */
    for (; argc-- > 0; argv++) {
	if (strcmp(*argv, "ignore_root") == 0)
	    ignore_root = 1;
        else if (strcmp(*argv,"clog") == 0) {
            if (argc-- > 0) {
            clog_program = realloc (clog_program,(strlen(*(++argv))+1)*sizeof(char));
            strcpy (clog_program,*argv);
            }
        } else if (strcmp(*argv,"realm") == 0) {
            if (argc-- > 0) {
              if (nrealms < 16) {
                realms[nrealms] = malloc ((strlen(*(++argv))+strlen(user)+2)*sizeof(char));
                strcpy (realms[nrealms],user);
                strcat (realms[nrealms],"@");
                strcat (realms[nrealms],*argv);
                ++nrealms;
              } else ++argv;
            }
	}
    }

/* Handle Root */
    if (pw->pw_uid == 0 && ignore_root == 1){
      result = PAM_SUCCESS;
      goto out;
    }

/* set up arguments for clog */
    execargs[0] = "clog";
    execargs[1] = "-nokinit";

    execargs[3] = NULL;

/* a loop over all realms */
    if( nrealms == 0 ){
      realms[0] = malloc ((strlen(user)+1)*sizeof(char));
      strcpy( realms[0], user );
      nrealms = 1;
    }
    while( nrealms > 0 ){
/* set up arguments for clog */
      execargs[2] = realms[nrealms-1];

/* set up pipe for communication with clog */
      if (pipe(pipedes) != 0) {
        syslog(LOG_CRIT, "pam_kcoda: Can't open pipe");
        result = PAM_CRED_ERR;
        goto out; /* a bad error, abort */
      }

      code = fork();
      if (code == -1) {
        syslog(LOG_ERR, "pam_kcoda: fork failed");
        result = PAM_CRED_ERR;
        goto out; /* a bad error, abort */
      }
      if (code == 0) {
	if (setgid(pw->pw_gid) < 0) {
	  syslog(LOG_ERR, "pam_kcoda: setgid(%d) failed",
		  pw->pw_gid);
	  exit( 1 );
	}
	if (setuid(pw->pw_uid) < 0) {
	  syslog(LOG_ERR, "pam_kcoda: setuid(%d) failed",
		  pw->pw_uid);
	  exit( 1 );
	}
	close(0);		/* stdin */
	dup(pipedes[0]);
	close(pipedes[0]);
	close(1);		/* stdout */
	dup(pipedes[1]);
	close(pipedes[1]);
	close(2);
	dup(1);
        if( (en=pam_getenvlist(pamh)) != NULL )
          execve(clog_program, execargs, en);
        else
          execv(clog_program, execargs);
	/* close pipe */
	close(0);		/* stdin */
	close(1);		/* stdout */
	close(2);		/* stdout */
	/* Error: exec did return! */
	syslog(LOG_CRIT, "pam_kcoda: exec returned");
	exit( 1 );
      }
/* close pipe */
      close(pipedes[1]);
/* read what clog tries to tells us :) -- pin */
      {char b[512]; read(pipedes[0],b,512);}
      close(pipedes[0]);
/* wait for child */
      while (code != wait(&status) && code != -1) ;
      if( code == -1 ){
/* we did't get the child status (why?)
   or got interrupt - it might be ok but we are paranoid :-) */
	result = PAM_CRED_ERR;
        /* but continue the realm iteration */
      }
      if (WIFEXITED(status) &&
	  WEXITSTATUS(status) == 0){
          /* ok, continue the loop */
          ;
      } else {
        syslog(LOG_NOTICE, "pam_kcoda: pam_clog[%d]: unsuccessful",
	        getpid());
        result = PAM_CRED_ERR;
        /* continue the loop */
      }
/* partial cleanup */
      free( realms[--nrealms] );
    }
out:
/* clean up */
    free(clog_program);
    while( nrealms > 0 ) free( realms[--nrealms] );
    return result;
}

PAM_EXTERN
int pam_sm_open_session(pam_handle_t * pamh, int flags,
                        int argc ,const char **argv)
{
  return PAM_SUCCESS;
}

PAM_EXTERN
int pam_sm_close_session(pam_handle_t * pamh, int flags,
                         int argc ,const char **argv)
{
    int retval;
    const char *user = NULL;
    struct passwd *pw = NULL;
    pid_t code;
    int status, pipedes[2];
    char *execargs[16];
    char *cunlog_program;
    char **en;
    int result = PAM_SUCCESS;

    retval = pam_get_user(pamh, &user, NULL);
    if (retval != PAM_SUCCESS)
        return PAM_SESSION_ERR;

    if (user == NULL || !isalnum(*user)) {
        if (user)
            syslog(LOG_ERR, "pam_kcoda: bad username [%s]", user);
        return PAM_SESSION_ERR;
    }

/* Get User Information */
    pw = getpwnam(user);

/* User does not exist */
    if (pw == NULL)
        return PAM_SESSION_ERR;

    cunlog_program = malloc ((strlen(CUNLOG_PROGRAM)+1)*sizeof(char));
    strcpy (cunlog_program,CUNLOG_PROGRAM);

/* Parse parameters for module */
    for (; argc-- > 0; argv++) {
        if (strcmp(*argv,"nocunlog") == 0){
            result = PAM_SUCCESS;
            goto out;
        } else if (strcmp(*argv,"cunlog") == 0) {
            if (argc-- > 0) {
              cunlog_program = realloc (cunlog_program,(strlen(*(++argv))+1)*sizeof(char));
              strcpy (cunlog_program,*argv);
            }
        }
    }

/* set up arguments for cunlog */
    execargs[0] = "cunlog";
    execargs[1] = NULL;

    code = fork();
    if (code == -1) {
	syslog(LOG_ERR, "pam_kcoda: fork faild");
	result = PAM_SESSION_ERR;
        goto out;
    }
    if (code == 0) {
	if (setgid(pw->pw_gid) < 0) {
	    syslog(LOG_ERR, "pam_kcoda: setgid(%d) failed",
		   pw->pw_gid);
	    exit( 1 );
	}
	if (setuid(pw->pw_uid) < 0) {
	    syslog(LOG_ERR, "pam_kcoda: setuid(%d) failed",
		   pw->pw_uid);
	    exit( 1 );
	}
	close(0);		/* stdin */
	dup(pipedes[0]);
	close(pipedes[0]);
	close(1);		/* stdout */
	dup(pipedes[1]);
	close(pipedes[1]);
	close(2);
	dup(1);
        if( (en=pam_getenvlist(pamh)) != NULL )
          execve(cunlog_program, execargs, en);
        else
          execv(cunlog_program, execargs);
	/* close pipe */
	close(0);		/* stdin */
	close(1);		/* stdout */
	close(2);		/* stdout */
	/* Error: exec did return! */
	syslog(LOG_CRIT, "pam_kcoda: exec returned");
	exit( 1 );
    }
/* close pipe */
    close(pipedes[1]);
/* read what cunlog tries to tells us :) -- pin */
    {char b[512]; read(pipedes[0],b,512);}
    close(pipedes[0]);
/* wait for child */
    while (code != wait(&status) && code != -1) ;
    if( code == -1 ){
/* we did't get the child status (why?)
   or got interrupt - it might be ok but we are paranoid :-) */
        result = PAM_SESSION_ERR;
        goto out;
    }
    if (WIFEXITED(status) &&
	WEXITSTATUS(status) == 0){
	    result = PAM_SUCCESS;
	    goto out;
    }
    syslog(LOG_NOTICE, "pam_kcoda: pam_cunlog[%d]: unsuccessful",
	   getpid());
    result = PAM_SESSION_ERR;
    goto out;
out:
    free( cunlog_program );
    return result;
}

/* end of module definition */
