/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "databaselistviewitem.h"
#include "tables/tablegrouplistviewitem.h"
#include "views/viewgrouplistviewitem.h"
#include "functions/functiongrouplistviewitem.h"
#include "sequences/sequencegrouplistviewitem.h"
#include "../utils/debugger.h"
#include "../utils/stringutils.h"
#include "common/infodialogs.h"
#include "serverlistviewitem.h"
#include "databaseconnectform.h"

#include <iostream.h>

	DatabaseListViewItem::DatabaseListViewItem(QListViewItem *pqoListViewItem)
		: BaseListViewItem(pqoListViewItem)
	{
		// setup table manager
		m_oDBMasterMgr.setDBConnection(&m_oDBConnection);
		m_strDatabaseName = "";
		m_strUserName = "";
		m_strPassword = "";
		m_bPasswordEntered = false;		
		updateDescription();
	} // end constructor

	DatabaseListViewItem::DatabaseListViewItem(QListViewItem *pqoListViewItem, QListViewItem *pqoAfterItem)
		: BaseListViewItem(pqoListViewItem, pqoAfterItem)
	{
		// setup table manager	
		m_oDBMasterMgr.setDBConnection(&m_oDBConnection);	
		m_strDatabaseName = "";
		m_strUserName = "";
		m_strPassword = "";
		m_bPasswordEntered = false;			
		updateDescription();		
	} // end constructor

	/**
	 * This method is used to set the database name.
	 */
	void DatabaseListViewItem::setDatabaseName(const string &rstrDatabaseName)
	{
		m_strDatabaseName = rstrDatabaseName;
	} // end setDatabaseName

	/**
	 * This method is used to get the database name.
	 */
	const string & DatabaseListViewItem::getDatabaseName() const
	{
		return m_strDatabaseName;
	} // end getDatabaseName

	/**
	 * This method is used to set the user name for the connection.
	 */
	void DatabaseListViewItem::setUserName(const string &rstrUserName)
	{
		m_strUserName = rstrUserName;
	} // end setUserName

	/**
	 * This method is used to get the user name.
	 */
	const string & DatabaseListViewItem::getUserName() const
	{
		return m_strUserName;
	} // end getUserName

	/**
	 * This method is used to set the password for the connection.
	 */
	void DatabaseListViewItem::setPassword(const string &rstrPassword)
	{
		m_strPassword = rstrPassword;
	} // end setPassword

	/**
	 * This method is used to get the password.
	 */
	const string & DatabaseListViewItem::getPassword() const
	{
		return m_strPassword;
	} // end getPassword

	/**
	 * This method is used to indicate that the user has entered their password.
	 * n.b. that when the user saves the database configuration file, the passwords
	 * are not saved hence each time the program is opened and used again it must
	 * prompt the user for the password.
	 */
	void DatabaseListViewItem::setPasswordEntered(bool bPasswordEntered)
	{
		m_bPasswordEntered = bPasswordEntered;
	} // end setPasswordEntered

	/**
	 * This method is used to see if the password has been entered.
	 */
	bool DatabaseListViewItem::isPasswordEntered() const
	{
		return m_bPasswordEntered;
	} // end isPasswordEntered

	/**
	 * This method is used to create the connection to the database.
	 */
	void DatabaseListViewItem::connect() throw (DBConnectionException)
	{
		ServerListViewItem * poServerListViewItem = (ServerListViewItem*)parent();
		string strServer = "";
		if (poServerListViewItem->isLocalServer() == false)
		{
			strServer = poServerListViewItem->getServerName();
		} // end if is local server
		m_oDBConnection.connect(getUserName(), getPassword(), getDatabaseName(), strServer, poServerListViewItem->getPortNumber());
	  // call default method to setup all grand children
	  BaseListViewItem::initGrandChildren();		  				
	
	} // end connect

	/**
 	 * Displays the right pressed menu for this item.
   */			    	
	void DatabaseListViewItem::showRightPressedMenu(const QPoint & rqoLocation, int nColumn)
	{
		
		// check to see if menu has been brought up previously
		if (m_pqoPopupMenu == 0)
		{
			m_pqoPopupMenu = new DatabaseItemPopup(listView(), this);
		} // end if popup already created
		m_pqoPopupMenu->popup(rqoLocation);
		
	} // end showRightPressedMenu
							
	/**
 	 * Used to update the description.
   */			    	
	void DatabaseListViewItem::updateDescription()
	{
		string strDescription = "<h1>Database</h1><hr><p><strong>Name</strong><br>" +
								StringUtils::htmlstr(getDatabaseName()) + "</p><p><strong>Username</strong><br>" +
								StringUtils::htmlstr(getUserName()) + "</p>";	
								
		m_qstrDescription = strDescription.c_str();	// convert over
	} // end updateDescription
			
	void DatabaseListViewItem::initGrandChildren()
	{	
		string strMethodName = "DatabaseListViewItem::initGrandChildren";	
		
		Debugger::entered(strMethodName);
		
		// only need to do if not expanded before
		if (m_bExpanded == false)		
		{		
  		// establish database connection
  		try
  		{
  			if (isPasswordEntered() == true)
  			{
  				connect();
  			} // end if password has been specified
  			else
  			{
					if (m_pqoPopupMenu == 0)
					{
						m_pqoPopupMenu = new DatabaseItemPopup(listView(), this);
					} // end if popup already created  			
					DatabaseConnectForm * poConnectForm = new DatabaseConnectForm(m_pqoPopupMenu, true, true);
					poConnectForm->show();  				
  			} // end else need to prompt user for password
  		} // end try
  		catch (DBConnectionException e)
  		{
  			Debugger::logException(strMethodName, e);
  			// report error to user
  			InfoDialogs::infoDBConnectionException(listView(), e);
				// lets close up the expanded list
	  		setExpanded(false);
  			// set to be not open		  	
  			setOpen(false);	  	  			
  		} // end catch
		} // end if not expanded before
		
		Debugger::exited(strMethodName);
		
	} // end initGrandChildren
	
	/**
 	 * Used to create all the first level children of the current item.
   */			    						
	void DatabaseListViewItem::initChildren()
	{	
		string strMethodName = "DatabaseListViewItem::initChildren";
		
		Debugger::entered(strMethodName);

		// create all children
		TableGroupListViewItem *poTableGroupItem = new TableGroupListViewItem(this);
		// set database connection
		poTableGroupItem->setDBMasterManager(&m_oDBMasterMgr);		

		ViewGroupListViewItem *poViewGroupItem = new ViewGroupListViewItem(this);
		// set database connection
		poViewGroupItem->setDBMasterManager(&m_oDBMasterMgr);		

		FunctionGroupListViewItem *poFunctionGroupItem = new FunctionGroupListViewItem(this);
		// set database connection
		poFunctionGroupItem->setDBMasterManager(&m_oDBMasterMgr);		

		SequenceGroupListViewItem *poSequenceGroupItem = new SequenceGroupListViewItem(this);
		// set database connection
		poSequenceGroupItem->setDBMasterManager(&m_oDBMasterMgr);		
						
		Debugger::exited(strMethodName);			
	} // end initChildren
	