#include "postgres.h"
#include "executor/spi.h"       /* this is what you need to work with SPI */
#include "commands/trigger.h"   /* ... and triggers */

#include "libpq/pqformat.h"
#include "miscadmin.h"
#include "utils/array.h"
#include "utils/builtins.h"
#include "utils/timestamp.h"
#include "utils/date.h"
#include "utils/informix.h"

PG_FUNCTION_INFO_V1(informix_serial);

/* args are the name of the column to increment, and the name of the sequence to use */

Datum
informix_serial(PG_FUNCTION_ARGS)
{
    TriggerData *trigdata = (TriggerData *) fcinfo->context;
    Trigger *trigger = trigdata->tg_trigger;

    int nargs = trigger->tgnargs;

    char * sequence;
    char * attr;

    TupleDesc   tupdesc;
    HeapTuple   rettuple;
    bool isnull;

    int i,ret, numberOfAttributes;

    Datum *values;
    char  *nulls, *incoming;
    int   attnums = 0, attnum;
    char select[256];

    if ( nargs != 2 ) elog( ERROR, "informix_serial needs two arguments, the attribute, and sequence");
 
    attr  = trigger->tgargs[0];
    sequence = trigger->tgargs[1];

    /* make sure it's called as a trigger at all */
    if (!CALLED_AS_TRIGGER(fcinfo))
        elog(ERROR, "informix_serial: not called by trigger manager");

    
    /* tuple to return to executor */
    rettuple = trigdata->tg_trigtuple;
    tupdesc = trigdata->tg_relation->rd_att;
    attnum = SPI_fnumber( tupdesc, attr);

    if (attnum == SPI_ERROR_NOATTRIBUTE ) elog( ERROR, "informix_serial: attribute not found");

    incoming = SPI_getvalue(rettuple, tupdesc, 1);

    /* make sure this is trying to insert a 0 */
    if (strcmp(incoming, "0") != 0 ) 
       return PointerGetDatum(rettuple);

    numberOfAttributes = tupdesc->natts;
    SPI_connect();

    values = (Datum *) palloc(numberOfAttributes * sizeof(Datum));
    nulls = (char *) palloc(numberOfAttributes * sizeof(char));
                                                                                                                                                                           
    /* fetch old values and nulls */
    for (i = 0; i < numberOfAttributes; i++)
    {
    	values[i] = SPI_getbinval(trigdata->tg_trigtuple, tupdesc, i + 1, &isnull);
        nulls[i] = (isnull) ? 'n' : ' ';
    }

    /* get the nextval from the named sequence */ 
    snprintf( select, 255 , "select nextval(\'%s\')", sequence);
   /*  elog(NOTICE, select ); */
    ret = SPI_exec(select, 0);
 
    if ( ret == SPI_OK_SELECT )
    {

       SPITupleTable *tuptable = SPI_tuptable;
       HeapTuple tuple = tuptable->vals[0];

       values[attnum-1] = SPI_getbinval(tuple, tupdesc, 1 , &isnull);
       /* elog(NOTICE, SPI_getvalue(tuple, tupdesc, 1 ) );*/

       attnums = attnum;
    }

    rettuple = SPI_modifytuple( trigdata->tg_relation, trigdata->tg_trigtuple, 1, &attnums, values, nulls);

    pfree(values);
    pfree(nulls);

    SPI_finish();
    return PointerGetDatum(rettuple);
}

Datum extend_in( PG_FUNCTION_ARGS )
{
        char       *inputText = PG_GETARG_CSTRING(0);
        text       *result;
        int                     len;
                                                                                                                                                                            
        /* verify encoding */
        len = strlen(inputText);
        pg_verifymbstr(inputText, len, false);
                                                                                                                                                                            
        result = (text *) palloc(len + VARHDRSZ);
        VARATT_SIZEP(result) = len + VARHDRSZ;
                                                                                                                                                                            
        memcpy(VARDATA(result), inputText, len);
                                                                                                                                                                            
        PG_RETURN_TEXT_P(result);
}

/*
 *              extend_out                 - converts internal representation to "..."
 */
Datum
extend_out(PG_FUNCTION_ARGS)
{
        text       *t = PG_GETARG_TEXT_P(0);
        int                     len;
        char       *result;
                                                                                                                                                                            
        len = VARSIZE(t) - VARHDRSZ;
        result = (char *) palloc(len + 1);
        memcpy(result, VARDATA(t), len);
        result[len] = '\0';
                                                                                                                                                                            
        PG_RETURN_CSTRING(result);
}

/* support for informix type date_extract ( date, from, to )
*
*	intervals can be year through to second 
* 	the return will be a string which includes the date parts specified inclusive 
*	ie.
*	year to second would return yymmdd hh:mm:ss
* 	hour to minute would return hh:mm
*/
Datum 
timestamp_extract( PG_FUNCTION_ARGS )
{
	struct tm	tt,
		*tm	= &tt;
	char	buf[MAXDATELEN + 1];
		
	int retval, len;
	text *result;
	fsec_t          fsec;
	char * from =  PG_GETARG_CSTRING(1);
	char * to   =  PG_GETARG_CSTRING(2);
	Timestamp timestamp = PG_GETARG_TIMESTAMP(0);
	memset(buf, 0, MAXDATELEN );
	
	if ( timestamp2tm(timestamp, NULL, tm, &fsec, NULL) == 0)		
	{
		retval = EncodeExtend( tm, from, to, buf,'-' );
	
		if ( retval < 0 )
		{
			elog(ERROR, "error code %d in timestamp_extract check the qualifiers %s, %s ", retval, from, to );
		}
	}
	else
	{
		elog(ERROR, "invalid timestamp");
	}


        len = (strlen(buf) + VARHDRSZ);
                                                                                                                                                       
        result = palloc(len);
                                                                                                                                                       
        VARATT_SIZEP(result) = len;
        memmove(VARDATA(result), buf, (len - VARHDRSZ));
                                                                                                                                                       
                                                                                                                                                       
        PG_RETURN_TEXT_P(result);

}


/* support for informix type date_extract ( date, from, to )
*
*	intervals can be year through to second 
* 	the return will be a string which includes the date parts specified inclusive 
*	ie.
*	year to second would return yymmdd hh:mm:ss
* 	hour to minute would return hh:mm
*/
Datum 
date_extract( PG_FUNCTION_ARGS )
{
	struct tm	tt,
		*tm	= &tt;
	char	buf[MAXDATELEN + 1];
	
	int retval, len;
	text *result;

	char * from =  PG_GETARG_CSTRING(1);
	char * to   =  PG_GETARG_CSTRING(2);
	
	DateADT		dateVal = PG_GETARG_DATEADT(0);
	
	memset(buf, 0, MAXDATELEN );
	
	j2date(dateVal + POSTGRES_EPOCH_JDATE,
		   &(tm->tm_year), &(tm->tm_mon), &(tm->tm_mday));

	if (IS_VALID_UTIME(tm->tm_year, tm->tm_mon, tm->tm_mday))

	{
		retval = EncodeExtend( tm, from, to, buf, '/' );
	
		if ( retval < 0 )
		{
			elog(ERROR, "error code %d in date_extract check the qualifiers %s, %s ", retval, from, to );
		}
	}
	else
	{
		elog(ERROR, "invalid date");
	}


        len = (strlen(buf) + VARHDRSZ);
                                                                                                                                                       
        result = palloc(len);
                                                                                                                                                       
        VARATT_SIZEP(result) = len;
        memmove(VARDATA(result), buf, (len - VARHDRSZ));
                                                                                                                                                       
                                                                                                                                                       
        PG_RETURN_TEXT_P(result);

}

