/*
	sql.h

	Common SQL client functionality.

	Project: pgjobs
	Author: Zlatko Michailov
	Created:  7-May-2004
	Updated: 10-May-2004
	Updated: 26-May-2004
	Updated: 27-May-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2004, Zlatko Michailov

*/



#ifndef __SQL_H__
#define __SQL_H__



//--------------------------------------------------------------------------------

#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <string>
#include <vector>
#include <fstream>
#include "defs.h"


using namespace std;



//--------------------------------------------------------------------------------

const int	MAX_PSQL_OUTPUT_LINE	= 1024;
const char	PSQL_BREAK_CHAR			= '+';



typedef vector< string >	SqlRow;
typedef vector< SqlRow >	SqlRowset;
typedef vector< int >		SqlColBreaks;



class SQL
{
// Supplemental types
public:
	typedef vector< string >	Row;
	typedef vector< SqlRow >	Rowset;



// Data members
protected:
	string	DbRefOptions;
	string	User;



// Constructors
public:
	SQL( const char* dbrefOptions, const char* user = 0 )
		: DbRefOptions( dbrefOptions )
	{
		if ( user )
		{
			User = user;
		}
	}



// Public methods
public:
	bool ExecFile( const char* file )
	{
		string	cmdOptions;

		// Command options: -X -f "<file>"
		cmdOptions  = "-X -f \"";
		cmdOptions += file;
		cmdOptions += "\"";

		return Psql( cmdOptions );
	}



	bool ExecCmd( const char* cmd )
	{
		string	cmdOptions;

		// Command options: -X -c "<command>"
		cmdOptions  = "-X -c \"";
		cmdOptions += cmd;
		cmdOptions += "\"";

		return Psql( cmdOptions );
	}



	bool FetchCmdRowset( const char* cmd, Rowset& rowset )
	{
		bool	ok;
		string	cmdOptions;
		string	tempFile;

		ok = MakeTempFileName( tempFile );
		if ( ok )
		{
			// Command options: -X -c "<command>" -o "<temp file>"
			cmdOptions  = "-X -c \"";
			cmdOptions += cmd;
			cmdOptions += "\" -o \"";
			cmdOptions += tempFile;
			cmdOptions += "\"";

			// Exec command
			ok = Psql( cmdOptions );

			// Parse output and build Rowset
			if ( ok )
			{
				ok = ParsePsqlOutput( tempFile, rowset );
			}

			// Remove temp file
			unlink( tempFile.c_str() );
		}

		return ok;
	}



// Support methods
protected:
	bool Psql( const string& cmdOptions )
	{
		int		execStatus;
		string	psqlCmdLine;

		// Construct command line: [su "user" -c '] psql <dbref> <command> [']
		psqlCmdLine = "";
		
		if ( !User.empty() )
		{
			psqlCmdLine += "su \"";
			psqlCmdLine += User;
			psqlCmdLine += "\" -c '";
		}

		psqlCmdLine += "psql ";
		psqlCmdLine += DbRefOptions;
		psqlCmdLine += " ";
		psqlCmdLine += cmdOptions;
		psqlCmdLine += " > ";
		psqlCmdLine += NullDevice;

		if ( !User.empty() )
		{
			psqlCmdLine += "'";
		}

		// Execute command line
		execStatus = system( psqlCmdLine.c_str() );

		return !execStatus;
	}



	bool ParsePsqlOutput( const string& fileName, SqlRowset& rowset )
	{
		bool			ok;
		ifstream		input;
		SqlColBreaks	colBreaks;

		// Open input
		input.open( fileName.c_str() );
		ok = input.is_open();

		// Compute column breaks
		if ( ok )
		{
			ok = ComputeColBreaks( input, colBreaks );
		}

		// Build rowset
		if ( ok )
		{
			ok = BuildRowset( input, colBreaks, rowset );
		}

		// Close input
		input.close();

		return ok;
	}



	bool ComputeColBreaks( ifstream& input, SqlColBreaks& colBreaks )
	{
		char	buff[ MAX_PSQL_OUTPUT_LINE + 1 ];
		bool	ok;

		colBreaks.clear();

		// Skip first line
		ok = input.getline( buff, MAX_PSQL_OUTPUT_LINE );

		// Use second line
		if ( ok )
		{
			ok = input.getline( buff, MAX_PSQL_OUTPUT_LINE );
		}

		// The line looks like:
		// -----+-----------+---+-----
		if ( ok )
		{
			char* breakPoint = buff;

			while ( breakPoint )
			{
				breakPoint = strchr( breakPoint, PSQL_BREAK_CHAR );
				if ( breakPoint )
				{
					colBreaks.push_back( breakPoint - buff );
					breakPoint++;
				}
			}
			
			// For simplicity add the trailing '\0'
			colBreaks.push_back( strlen( buff ) );
		}

		return ok;
	}



	bool BuildRowset( ifstream& input, const SqlColBreaks& colBreaks, SqlRowset& rowset )
	{
		char	buff[ MAX_PSQL_OUTPUT_LINE + 1 ];
		bool	hasInput;
		int		colCount;

		rowset.clear();
		colCount = colBreaks.size();

		// Traverse lines
		hasInput = input.getline( buff, MAX_PSQL_OUTPUT_LINE );
		while ( hasInput )
		{
			SqlRow	row;
			int		breakPoint = 0;

			// Parse columns except
			for ( int i = 0; i < colCount; i++ )
			{
				string colVal;

				// Construct a trimmed string instance
				TrimStr( colVal, buff + breakPoint, colBreaks[ i ] - breakPoint );

				// Add column value to row
				row.push_back( colVal );

				breakPoint = colBreaks[ i ] + 1;
			}

			// Ad row to rowset
			rowset.push_back( row );

			// Read next line
			hasInput = input.getline( buff, MAX_PSQL_OUTPUT_LINE );
		}

		// Remove the last 2 rows, it looks like:
		// (xx rows)
		//
		if ( rowset.size() > 0 )
		{
			rowset.pop_back();
			rowset.pop_back();
		}

		return true;
	}



	bool TrimStr( string& trim, const char* source, int sourceLen = -1 )
	{
		const char* trimSource;
		const char*	trimEnd;

		// Make sure we have a good length
		if ( sourceLen < 0 )
		{
			sourceLen = strlen( source );
		}

		trimSource	= source;
		trimEnd		= source + sourceLen - 1;

		// Trim from front
		while ( trimSource <= trimEnd )
		{
			if ( *trimSource != ' ' &&
				 *trimSource != '\t' &&
				 *trimSource != '\r' &&
				 *trimSource != '\n' )
			{
				break;
			}

			trimSource++;
		}

		// Trim from end
		while ( trimSource <= trimEnd )
		{
			if ( *trimEnd != ' ' &&
				 *trimEnd != '\t' &&
				 *trimEnd != '\r' &&
				 *trimEnd != '\n' )
			{
				break;
			}

			trimEnd--;
		}

		// Build result
		trim.clear();
		if ( trimSource <= trimEnd )
		{
			trim .assign( trimSource, trimEnd - trimSource + 1 );
		}

		return true;
	}



	bool MakeTempFileName( string& tempFile )
	{
		char	buff[ 512 + 1 ];
		timeval	now;

		gettimeofday( &now, 0 );

		sprintf( buff, TempFileTemplate,
						P_tmpdir,
						( long ) getpid(),
						( long ) now.tv_sec,
						( long ) now.tv_usec );

		tempFile += buff;

		return true;
	}

};



//--------------------------------------------------------------------------------



#endif	// __SQL_H__
