#include "euler.h"

/*!
  \file
  \brief Euler transformation functions
*/


/*!
  \brief create a zero matrix
  \param m pointer to matrix
  \param n count of cells
*/
#define matrix_zero(m,n)	memset( (void*)(m), 0, (n)*sizeof(float8) )


#ifndef DOXYGEN_SHOULD_SKIP_THIS
  PG_FUNCTION_INFO_V1(spheretrans_in);
  PG_FUNCTION_INFO_V1(spheretrans_from_float8);
  PG_FUNCTION_INFO_V1(spheretrans_from_float8_and_type);
  PG_FUNCTION_INFO_V1(spheretrans_equal);
  PG_FUNCTION_INFO_V1(spheretrans_not_equal);
  PG_FUNCTION_INFO_V1(spheretrans_phi);
  PG_FUNCTION_INFO_V1(spheretrans_theta);
  PG_FUNCTION_INFO_V1(spheretrans_psi);
  PG_FUNCTION_INFO_V1(spheretrans_type);
  PG_FUNCTION_INFO_V1(spheretrans_zxz);
  PG_FUNCTION_INFO_V1(spheretrans);
  PG_FUNCTION_INFO_V1(spheretrans_invert);
  PG_FUNCTION_INFO_V1(spheretrans_trans);
  PG_FUNCTION_INFO_V1(spheretrans_trans_inv);
  PG_FUNCTION_INFO_V1(spheretrans_point);
  PG_FUNCTION_INFO_V1(spheretrans_point_inverse);
#endif


  /*!
    \brief  calculate the matrix product m*v
    \param  velem count of cells within matrix
    \param  v pointer to second matrix
    \param  m pointer to first matrix
    \param  vret pointer to result matrix
    \return pointer to result matrix
  */
  static float8 *  matrix_vector_mul ( float8 * vret , const float8 * m , const float8 * v , int velem )
  {
     int i,k;
     // velem*i+k i..row k..column
     for ( i=0; i<velem ; i++ ){
       vret[i] = 0.0;
       for ( k=0; k<velem ; k++ ){
        vret[i] += m[velem*i+k]*v[k];
       }
     }
     return vret;
  }


  Datum  spheretrans_in(PG_FUNCTION_ARGS)
  {
    SEuler   * se   = ( SEuler * ) MALLOC ( sizeof ( SEuler ) ) ;
    char      *  c  = PG_GETARG_CSTRING(0);
    unsigned char etype[3];
    SPoint     sp[3];
    int            i;

    void sphere_yyparse( void );

    init_buffer ( c );
    sphere_yyparse();

    sp[0].lat = sp[1].lat = sp[2].lat = 0.0;

    if ( get_euler ( &sp[0].lng, &sp[1].lng, &sp[2].lng, etype ) ){

      for ( i=0; i<3; i++ ){
        switch ( i ){
          case 0:  se->phi_a   = etype[i] ; break;
          case 1:  se->theta_a = etype[i] ; break;
          case 2:  se->psi_a   = etype[i] ; break;
        }
      }

      spoint_check ( &sp[0] );
      spoint_check ( &sp[1] );
      spoint_check ( &sp[2] );
      se->phi       = sp[0].lng ;
      se->theta     = sp[1].lng ;
      se->psi       = sp[2].lng ;
    } else {
      reset_buffer();
      FREE( se );
      se = NULL;
      elog ( ERROR , "spheretrans_in: parse error" );
    }
    reset_buffer();
    PG_RETURN_POINTER( se );
  }

  
  Datum  spheretrans_from_float8 (PG_FUNCTION_ARGS)
  {
    SEuler  * se  = ( SEuler * ) MALLOC ( sizeof ( SEuler ) ) ;
    static SPoint  sp[3];
    sp[0].lat = sp[1].lat = sp[2].lat = 0.0;
    sp[0].lng   =  PG_GETARG_FLOAT8( 0 );
    sp[1].lng   =  PG_GETARG_FLOAT8( 1 );
    sp[2].lng   =  PG_GETARG_FLOAT8( 2 );
    spoint_check ( &sp[0] );
    spoint_check ( &sp[1] );
    spoint_check ( &sp[2] );
    se->phi       = sp[0].lng ;
    se->theta     = sp[1].lng ;
    se->psi       = sp[2].lng ;
    seuler_set_zxz( se );
    PG_RETURN_POINTER( se );
  }

  
  Datum  spheretrans_from_float8_and_type (PG_FUNCTION_ARGS)
  {
    SEuler  * se;
    Datum  d[3];
    int       i;
    char    * c = PG_GETARG_CSTRING(3);
    unsigned char t = 0;

    d[0]   =  PG_GETARG_DATUM( 0 );
    d[1]   =  PG_GETARG_DATUM( 1 );
    d[2]   =  PG_GETARG_DATUM( 2 );
    se = ( SEuler * ) DatumGetPointer( DirectFunctionCall3 ( 
           spheretrans_from_float8 , d[0], d[1], d[2] ) ); 

    for ( i=0; i<3; i++ ){
      switch ( c[i] ){
        case 'x': 
        case 'X': t = EULER_AXIS_X; break;
        case 'y':
        case 'Y': t = EULER_AXIS_Y; break;
        case 'z':
        case 'Z': t = EULER_AXIS_Z; break;
        default : t = 0;
      }

      if ( t == 0 ){
        FREE ( se );
        elog ( ERROR , "invalid axis format" );
      }
      switch ( i ){
        case 0:  se->phi_a   = t; break;
        case 1:  se->theta_a = t; break;
        case 2:  se->psi_a   = t; break;
      }
    }  
    PG_RETURN_POINTER ( se );    

  }

  void seuler_set_zxz ( SEuler * se )
  {
    se->phi_a   = EULER_AXIS_Z;
    se->theta_a = EULER_AXIS_X;
    se->psi_a   = EULER_AXIS_Z;
  }


  bool strans_eq ( const SEuler * e1, const SEuler * e2 )
  {

    static SPoint in[2], p[4] ;

    in[0].lng  = 0.0;
    in[0].lat  = 0.0;
    in[1].lng  = PIH;
    in[1].lat  = 0.0;

    euler_spoint_trans ( &p[0] , &in[0] , e1 );
    euler_spoint_trans ( &p[1] , &in[1] , e1 );
    euler_spoint_trans ( &p[2] , &in[0] , e2 );
    euler_spoint_trans ( &p[3] , &in[1] , e2 );

    return ( spoint_eq ( &p[0], &p[2] ) && spoint_eq ( &p[1], &p[3] ) );
  
  }


  
  Datum  spheretrans_equal(PG_FUNCTION_ARGS)
  {
    SEuler  * e1 =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    SEuler  * e2 =  ( SEuler * )  PG_GETARG_POINTER ( 1 ) ;
    PG_RETURN_BOOL ( strans_eq ( e1, e2 ) );
  }

  
  Datum  spheretrans_not_equal(PG_FUNCTION_ARGS)
  {
    SEuler  * e1 =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    SEuler  * e2 =  ( SEuler * )  PG_GETARG_POINTER ( 1 ) ;
    bool b = DatumGetBool(
      DirectFunctionCall2 (
        spheretrans_equal,
        PointerGetDatum ( e1 ),
        PointerGetDatum ( e2 )
      ) );
    PG_RETURN_BOOL ( ! b );
  }
  

  
  Datum  spheretrans_phi(PG_FUNCTION_ARGS)
  {
    SEuler  * se =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    PG_RETURN_POINTER ( &se->phi );
  }

  
  Datum  spheretrans_theta(PG_FUNCTION_ARGS)
  {
    SEuler  * se =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    PG_RETURN_POINTER ( &se->theta );
  }
  
  
  Datum  spheretrans_psi(PG_FUNCTION_ARGS)
  {
    SEuler  * se =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    PG_RETURN_POINTER ( &se->psi );
  }
  
  
  Datum  spheretrans_type(PG_FUNCTION_ARGS)
  {
    SEuler     * se =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    BpChar  *result =  ( BpChar * )  MALLOC ( 3 + VARHDRSZ );
    char     ret[4] ;
    int           i ;
    unsigned char t = 0;

    for ( i=0; i<3; i++ ){
      switch ( i ){
        case 0: t = se->phi_a  ; break;
        case 1: t = se->theta_a; break;
        case 2: t = se->psi_a  ; break;
      }
      switch ( t ){
        case EULER_AXIS_X : ret[i]='X'; break;
        case EULER_AXIS_Y : ret[i]='Y'; break;
        case EULER_AXIS_Z : ret[i]='Z'; break;
      }
    }
    ret[3] = '\0';
    VARATT_SIZEP(result) = 3 + VARHDRSZ;
    memcpy(VARDATA(result), &ret[0], 3 );

    PG_RETURN_BPCHAR_P(result);

  }

  SEuler * spheretrans_inv ( SEuler * se )
  {

    SPoint sp[3];
    const unsigned char c = se->phi_a ;

    sp[2].lng = - se->phi;
    sp[1].lng = - se->theta;
    sp[0].lng = - se->psi;
    sp[0].lat = sp[1].lat = sp[2].lat = 0;
    spoint_check(&sp[0]);
    spoint_check(&sp[1]);
    spoint_check(&sp[2]);
    se->phi     = sp[0].lng;
    se->theta   = sp[1].lng;
    se->psi     = sp[2].lng;
    se->phi_a   = se->psi_a ;
    se->psi_a   = c ;
    return se;

  }

  SEuler * spheretrans_inverse ( SEuler * se_out , const SEuler * se_in )
  {
    memcpy ( (void*) se_out, (void*) se_in, sizeof( SEuler) );
    spheretrans_inv( se_out );
    return se_out;
  }


  SEuler * strans_zxz( SEuler * ret , const SEuler * se )
  {
    if (
         se->phi_a   == EULER_AXIS_Z &&
         se->theta_a == EULER_AXIS_X &&
         se->psi_a   == EULER_AXIS_Z
    ){
      memcpy( (void*) ret, (void*) se, sizeof( SEuler ) );
    } else {
      static SEuler tmp;
      tmp.psi   = 0.0;
      tmp.theta = 0.0;
      tmp.phi   = 0.0;
      seuler_set_zxz( &tmp );
      seuler_trans_zxz ( ret, se, &tmp );
    }
    return  ret;
  }

  
  Datum  spheretrans_zxz(PG_FUNCTION_ARGS)
  {
    SEuler  * si  =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    SEuler  * ret =  ( SEuler * ) MALLOC ( sizeof ( SEuler ) );
    strans_zxz( ret , si );
    PG_RETURN_POINTER ( ret );
  }

  
  Datum  spheretrans(PG_FUNCTION_ARGS)
  {
    Datum d  =  PG_GETARG_DATUM ( 0 ) ;
    PG_RETURN_DATUM ( d );
  }

  
  Datum  spheretrans_invert(PG_FUNCTION_ARGS)
  {
    SEuler  * se  =  ( SEuler * )  PG_GETARG_POINTER ( 0 ) ;
    SEuler  * ret =  ( SEuler * ) MALLOC ( sizeof ( SEuler ) );
    spheretrans_inverse ( ret , se );
    PG_RETURN_POINTER ( ret );
  }
  

  SEuler * seuler_trans_zxz ( SEuler * out , const SEuler * in , const SEuler * se )
  {

    static SPoint sp[4] ;
    sp[0].lng = 0.0;
    sp[0].lat = 0.0;
    sp[1].lng = PIH;
    sp[1].lat = 0.0;
    euler_spoint_trans ( &sp[2], &sp[0], in );
    euler_spoint_trans ( &sp[3], &sp[1], in );
    euler_spoint_trans ( &sp[0], &sp[2], se );
    euler_spoint_trans ( &sp[1], &sp[3], se );
    spherevector_to_euler ( out , &sp[0], &sp[1] );
    

    return out;
  }


  
  Datum  spheretrans_trans(PG_FUNCTION_ARGS)
  {

    SEuler  * se1  =  ( SEuler  * ) PG_GETARG_POINTER ( 0 ) ;
    SEuler  * se2  =  ( SEuler  * ) PG_GETARG_POINTER ( 1 ) ;
    SEuler  * out  =  ( SEuler  * ) MALLOC ( sizeof( SEuler ) );
    seuler_trans_zxz  ( out , se1, se2 );
    PG_RETURN_POINTER ( out );
  }

  
  Datum  spheretrans_trans_inv(PG_FUNCTION_ARGS)
  {

    SEuler  * se1  =  ( SEuler  * ) PG_GETARG_POINTER ( 0 ) ;
    SEuler  * se2  =  ( SEuler  * ) PG_GETARG_POINTER ( 1 ) ;
    SEuler  * out  =  ( SEuler  * ) MALLOC ( sizeof( SEuler ) );
    SEuler    tmp  ;
    spheretrans_inverse ( &tmp , se2 );
    seuler_trans_zxz  ( out , se1, &tmp );
    PG_RETURN_POINTER ( out );
  }


  SPoint  * euler_spoint_trans ( SPoint * out , const SPoint  * in , const SEuler * se )
  {
    Vector3D v,o ;
    spoint_vector3d ( &v , in );
    euler_vector_trans ( &o , &v , se );
    vector3d_spoint ( out , &o );
    spoint_check ( out );
    return out;
  }

  Datum  spheretrans_point(PG_FUNCTION_ARGS)
  {
    SPoint *  sp =  ( SPoint * ) PG_GETARG_POINTER ( 0 ) ;
    SEuler *  se =  ( SEuler * ) PG_GETARG_POINTER ( 1 ) ;
    SPoint * out =  ( SPoint * ) MALLOC ( sizeof ( SPoint ) );
    PG_RETURN_POINTER ( euler_spoint_trans ( out , sp , se ) );
  }


  Datum  spheretrans_point_inverse (PG_FUNCTION_ARGS)
  {

   Datum sp     = PG_GETARG_DATUM ( 0 );
   SEuler *  se = ( SEuler * ) PG_GETARG_POINTER( 1 );
   SEuler tmp   ;
   Datum ret;
   
   spheretrans_inverse ( &tmp , se );
   ret =  DirectFunctionCall2(
                       spheretrans_point ,
                       sp, PointerGetDatum(&tmp) );
   PG_RETURN_DATUM( ret );
  }

  /*!
    \brief transforms a spherical vector to a inverse Euler transformation
    \param spb pointer to begin of spherical vector
    \param spe pointer to end   of spherical vector
    \param se  pointer to Euler transformation
    \return true, if calculation was successfull
    \see spherevector_to_euler ( SEuler *, SPoint *, SPoint * )
  */
  static bool spherevector_to_euler_inv ( SEuler * se, const SPoint * spb , const SPoint * spe )
  {

    if ( spoint_eq ( spb, spe ) ) { 

      return false;

    } else {
    
      static Vector3D vbeg,vend,vtmp;
      static SPoint spt[2];
      static SEuler set;

      spoint_vector3d ( &vbeg  , spb );
      spoint_vector3d ( &vend  , spe );
      vector3d_cross  ( &vtmp  , &vbeg , &vend );
      vector3d_spoint ( &spt[0], &vtmp );
      set.phi     = - spt[0].lng - PIH ; 
      set.theta   =   spt[0].lat - PIH ;
      set.psi     =   0.0 ;
      seuler_set_zxz ( &set );
      euler_spoint_trans ( &spt[1], spb, &set );
      set.psi     = - spt[1].lng;
      memcpy( (void*) se , (void*) &set, sizeof(SEuler) );
    }

    return true ;
  }


  bool spherevector_to_euler ( SEuler * se, const SPoint * spb , const SPoint * spe )
  {
    static bool ret;
    ret = spherevector_to_euler_inv ( se, spb , spe );
    spheretrans_inv ( se );
    return ret;
  }

  Vector3D  * euler_vector_trans ( Vector3D  * out , const Vector3D  * in , const SEuler * se )
  {
    static int i;
    static unsigned int t ;
    static const float8     * a ;
    static float8  m[3][3] , u[3], vr[3] ;

    t    = 0;
    a    = NULL ;
    u[0] = in->x;
    u[1] = in->y;
    u[2] = in->z;

    for ( i=0; i<3; i++ ){
      switch ( i ){
        case 0 : a = &se->phi   ; t = se->phi_a   ; break;
        case 1 : a = &se->theta ; t = se->theta_a ; break;
        case 2 : a = &se->psi   ; t = se->psi_a   ; break;
      }
      if ( FPzero( *a ) ) continue ;

      matrix_zero ( &m[0][0] , 9 ) ;

      switch ( t ){
        case EULER_AXIS_X :
              m[1][1] =  cos(*a);
              m[1][2] = -sin(*a);
              m[2][1] =  sin(*a);
              m[2][2] =  cos(*a);
              m[0][0] =  1.0;
              break;
        case EULER_AXIS_Y :
              m[0][0] =  cos(*a);
              m[0][2] =  sin(*a);
              m[2][0] = -sin(*a);
              m[2][2] =  cos(*a);
              m[1][1] =  1.0;
              break;
        case EULER_AXIS_Z :
              m[0][0] =  cos(*a);
              m[0][1] = -sin(*a);
              m[1][0] =  sin(*a);
              m[1][1] =  cos(*a);
              m[2][2] =  1.0;
              break;
      }
      matrix_vector_mul ( &vr[0], &m[0][0], &u[0], 3);
      memcpy( ( void * ) &u[0], ( void * ) &vr[0], sizeof(u) );
    }
    out->x = u[0];
    out->y = u[1];
    out->z = u[2];

    return ( out );

  }

