/* $PostgresPy: if/src/obj.c,v 1.29 2004/07/27 09:18:38 flaw Exp $
 * 
 * † Instrument:
 *     Copyright 2004, rhid development. All Rights Reserved.
 *     
 *     Usage of the works is permitted provided that this
 *     instrument is retained with the works, so that any entity
 *     that uses the works is notified of this instrument.
 *     
 *     DISCLAIMER: THE WORKS ARE WITHOUT WARRANTY.
 *     
 *     [2004, Fair License; rhid.com/fair]
 *     
 * Description:
 *    Postgres "object" interface (Datum with a TypeTuple)
 */
#include <setjmp.h>
#include <pputils.h>

#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_operator.h>
#include <nodes/params.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/catcache.h>
#include <pg.h>
#include <PGExcept.h>

#include <Python.h>
#include <structmember.h>
#include <py.h>

#include "module.h"
#include "utils.h"
#include "datum.h"
#include "tupd.h"
#include "tup.h"
#include "type.h"
#include "obj.h"

const char PyPgObject_Doc[] = "Python Interface to a Postgres Object";

#ifdef NOT_USED
/*
 * obj_instill - instill a value in the object's Datum
 *
 * Set the Datum to the new value, but not before free'ing the old
 */
static int
obj_instill(PgObj self, PyObj with)
{
	Datum nd = 0;
	
	nd = Datum_FromPyObjectAndTypeTuple(with, PgObj_FetchTypeHT(self));
	/* XXX Check NULL, raise exception */

	if (PgObj_ShouldFree(self))
		pfree(DatumGetPointer(PgObj_FetchDatum(self)));
	PgObj_FetchDatum(self) = nd;

	return(0);
}
#endif

static PyObj
obj_plus(PgObj self, PyObj with)
{
	PyObj rob;
	rob = PgObj_Operate("+", self, with);
	return(rob);
}

static PyMethodDef PyPgObject_Methods[] = {
	/* {"name", FunctionRef, flags, "docstring"}, */
	{"cat", (PyCFunction)obj_plus, METH_O, "Catentation"},
	{NULL}
};

static long
varlena_obj_length(PgObj self)
{
	return(VARSIZE(self->ob_datum));
}

static long
nullt_obj_length(PgObj self)
{
	return(strlen((char*)(self->ob_datum)));
}

static long
obj_length(PgObj self)
{
	long len = 0;
	Form_pg_type typs = PGOTYPESTRUCT(self);

	if (typs->typelem)
		len = ARR_DIMS(self->ob_datum)[0];
	else
	{
		/* Custom data types support */
		switch(PgObj_FetchTypeOid(self))
		{
			default:
				len = 0;
			break;
		}
	}
	
	return(len);
}

static PyObj
obj_repeat(PgObj self, int item)
{
	RETURN_NONE;
}

static PyObj
obj_item(PgObj self, int item)
{
	Oid typoid = PgObj_FetchTypeOid(self);
	Form_pg_type typs = PGOTYPESTRUCT(self);
	PyObj rob = NULL;

	if (typs->typelem)
	{
		ArrayType *array = DatumGetArrayTypeP(self->ob_datum);
		int ndims;
		bool isnull = false;
		int eles;
		HeapTuple ett;
		Oid elmtypoid;
		Form_pg_type etyps;
		Datum rd;
		
		eles = ARR_DIMS(array)[0];
		if (item >= eles)
		{
			PyErr_Format(PyExc_IndexError,
				"index(%d) out of range for a '%s' object with %d elements",
				item, NameStr(typs->typname), eles
			);
			return(NULL);
		}
		
		ndims = ARR_NDIM(array);
		if (ndims > 1)
		{
			/* TODO: implement MDA support */
			PyErr_SetString(PyExc_NotImplementedError,
				"subscripting multi-dimensional arrays is unimplemented");
			return(NULL);
		}
		else
		{
			ett = SearchSysCache(TYPEOID, ObjectIdGetDatum(typs->typelem),
						0, 0, 0);
			elmtypoid = HeapTuple_FetchOid(ett);
			etyps = TYPESTRUCT(ett);
		}

		++item; /* lower bounds is 1 */
		rd = array_ref(array, 1, &item,
				typs->typlen,
				etyps->typlen,
				etyps->typbyval,
				etyps->typalign,
				&isnull);
		rd = datumCopy(rd, etyps->typbyval, etyps->typlen);
		if (ett)
			ReleaseSysCache(ett);

		rob = PgObj_FromDatumAndTypeOid(rd, elmtypoid);
	}
	else
	{
		switch(typoid)
		{
			default:
				PyErr_Format(PyExc_TypeError,
					"'%s' does not support indirection",
					NameStr(typs->typname)
				);
			break;
		}
	}

	return(rob);
}

static PyObj
obj_slice(PgObj self, int fitem, int titem)
{
	RETURN_NONE;
}

static PyObj
obj_ass_item(PgObj self, int item, PyObj as)
{
	RETURN_NONE;
}

static PyObj
obj_ass_slice(PgObj self, int from, int toi, PyObj as)
{
	RETURN_NONE;
}

static PyObj
obj_contains(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_concat(PgObj self, PyObj with)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_repeat(PgObj self, int num)
{
	RETURN_NONE;
}

static PySequenceMethods PyPgObjectAsSequence = {
	(inquiry)obj_length,					/* sq_length */
	(binaryfunc)obj_plus,				/* sq_concat */
	(intargfunc)obj_repeat,				/* sq_repeat */
	(intargfunc)obj_item,				/* sq_item */
	(intintargfunc)obj_slice,			/* sq_slice */
	(intobjargproc)obj_ass_item,		/* sq_ass_item */
	(intintobjargproc)obj_ass_slice,	/* sq_ass_slice */
	(objobjproc)obj_contains,			/* sq_contains */
	(binaryfunc)obj_inplace_concat,	/* sq_inplace_concat */
	(intargfunc)obj_inplace_repeat,	/* sq_inplace_repeat */
};

#ifdef NOT_USED
static PyObj
obj_subscript(PgObj self, PyObj sub)
{
	RETURN_NONE;
}

static PyObj
obj_ass_subscript(PyObj self, PyObj sub, PyObj to)
{
	RETURN_NONE;
}

static PyMappingMethods PyPgObjectAsMapping = {
	(inquiry)obj_length,						/* mp_length */
	(binaryfunc)obj_subscript,				/* mp_subscript */
	(objobjargproc)obj_ass_subscript,	/* mp_ass_subscript */
};
#else
#define PyPgObjectAsMapping NULL
#endif

static PyObj
obj_subtract(PgObj self, PyObj with)
{
	return(PgObj_Operate("-", self, with));
}

static PyObj
obj_multiply(PgObj self, PyObj with)
{
	return(PgObj_Operate("*", self, with));
}

static PyObj
obj_divide(PgObj self, PyObj with)
{
	return(PgObj_Operate("/", self, with));
}

static PyObj
obj_remainder(PgObj self, PyObj with)
{
	return(PgObj_Operate("%", self, with));
}

static PyObj
obj_divmod(PgObj self, PyObj with)
{
	RETURN_NONE;
}

static PyObj
obj_power(PgObj self, PyObj with, PyObj andthis)
{
	RETURN_NONE;
}

static PyObj
obj_negative(PgObj self)
{
	return(PgObj_Operate("-", self, NULL));
}

static PyObj
obj_positive(PgObj self)
{
	return(PgObj_Operate("+", self, NULL));
}

static PyObj
obj_absolute(PgObj self)
{
	RETURN_NONE;
}

static int
obj_nonzero(PgObj self)
{
	return(-1);
}

static PyObj
obj_invert(PgObj self)
{
	RETURN_NONE;
}

static PyObj
obj_lshift(PgObj self, PyObj with)
{
	return(PgObj_Operate("<<", self, with));
}

static PyObj
obj_rshift(PgObj self, PyObj with)
{
	return(PgObj_Operate(">>", self, with));
}

static PyObj
obj_and(PgObj self, PyObj with)
{
	return(PgObj_Operate("&", self, with));
}

static PyObj
obj_xor(PgObj self, PyObj with)
{
	return(PgObj_Operate("^", self, with));
}

static PyObj
obj_or(PgObj self, PyObj with)
{
	return(PgObj_Operate("|", self, with));
}

static PyObj
obj_int(PgObj self)
{
	RETURN_NONE;
}

static PyObj
obj_long(PgObj self)
{
	RETURN_NONE;
}

static PyObj
obj_float(PgObj self)
{
	RETURN_NONE;
}

static PyObj
obj_oct(PgObj self)
{
	RETURN_NONE;
}

static PyObj
obj_hex(PgObj self)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_add(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_subtract(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_multiply(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_divide(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_remainder(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_power(PgObj self, PyObj ob, PyObj ex)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_lshift(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_rshift(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_and(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_xor(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_or(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_floor_divide(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_true_divide(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_floor_divide(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyObj
obj_inplace_true_divide(PgObj self, PyObj ob)
{
	RETURN_NONE;
}

static PyNumberMethods PyPgObjectAsNumber = {
	(binaryfunc)obj_plus,						/* nb_add */
	(binaryfunc)obj_subtract,					/* nb_subtract */
	(binaryfunc)obj_multiply,					/* nb_multiply */
	(binaryfunc)obj_divide,						/* nb_divide */
	(binaryfunc)obj_remainder,					/* nb_remainder */
	(binaryfunc)obj_divmod,						/* nb_divmod */
	(ternaryfunc)obj_power,						/* nb_power */
	(unaryfunc)obj_negative,					/* nb_negative */
	(unaryfunc)obj_positive,					/* nb_positive */
	(unaryfunc)obj_absolute,					/* nb_absolute */
	(inquiry)obj_nonzero,						/* nb_nonzero */
	(unaryfunc)obj_invert,						/* nb_invert */
	(binaryfunc)obj_lshift,						/* nb_lshift */
	(binaryfunc)obj_rshift,						/* nb_rshift */
	(binaryfunc)obj_and,							/* nb_and */
	(binaryfunc)obj_xor,							/* nb_xor */
	(binaryfunc)obj_or,							/* nb_or */
	(coercion)NULL,								/* nb_coerce */
	(unaryfunc)obj_int,							/* nb_int */
	(unaryfunc)obj_long,							/* nb_long */
	(unaryfunc)obj_float,						/* nb_float */
	(unaryfunc)obj_oct,							/* nb_oct */
	(unaryfunc)obj_hex,							/* nb_hex */

	(binaryfunc)obj_inplace_add,				/* nb_inplace_add */
	(binaryfunc)obj_inplace_subtract,		/* nb_inplace_subtract */
	(binaryfunc)obj_inplace_multiply,		/* nb_inplace_multiply */
	(binaryfunc)obj_inplace_divide,			/* nb_inplace_divide */
	(binaryfunc)obj_inplace_remainder,		/* nb_inplace_remainder */
	(ternaryfunc)obj_inplace_power,			/* nb_inplace_power */
	(binaryfunc)obj_inplace_lshift,			/* nb_inplace_lshift */
	(binaryfunc)obj_inplace_rshift,			/* nb_inplace_rshift */
	(binaryfunc)obj_inplace_and,				/* nb_inplace_and */
	(binaryfunc)obj_inplace_xor,				/* nb_inplace_xor */
	(binaryfunc)obj_inplace_or,				/* nb_inplace_or */

	(binaryfunc)obj_floor_divide,				/* nb_floor_divide */
	(binaryfunc)obj_true_divide,				/* nb_true_divide */
	(binaryfunc)obj_inplace_floor_divide,	/* nb_inplace_floor_divide */
	(binaryfunc)obj_inplace_true_divide,	/* nb_inplace_true_divide */
};

/*
 * 0 equal to
 * 1 greater than
 * -1 less than
 */
static int
obj_compare(PgObj self, PyObj with)
{
	int ret = -1;
	
	if (PgObj_IsNull(self))
	{
		if (with == Py_None)
			ret = 0;
		else if (PgObj_TypeCheck(with))
		{
			if (PgObj_IsNull(with))
				ret = 0;
		}
		else
			ret = -1;
	}
	else
	{
		PyObj withd;
		PgObj et, lt = NULL;

		if (with == Py_None)
			return(1);
	
		if (!PgObj_TypeCheck(with))
			withd = PgObj_FromPgTypeAndPyObject(PgObj_FetchType(self), with);
		else
			withd = with;

		if (PgObj_IsNull(withd))
			ret = 1;
		else if (PgObj(with)->ob_datum == PgObj(withd)->ob_datum &&
				PgObj_FetchTypeOid(withd) == PgObj_FetchTypeOid(with))
			ret = 0;

		et = (PgObj)PgObj_Operate("=", self, withd);
		if (et->ob_datum)
			ret = 0;
		else if ((lt = (PgObj)PgObj_Operate("<", self, withd))->ob_datum)
			ret = -1;
		else
			ret = 1;
		
		XDECREF((PyObj)et);
		XDECREF((PyObj)lt);
		if ((PyObj)withd != with)
			XDECREF((PyObj)withd);
	}

	return(ret);
}

static long
obj_hash(PgObj self)
{
	return((long)self);
}

static PyObj
obj_repr(PgObj self)
{
	PyObj rob;
	char *str;
	Form_pg_type typs = PGOTYPESTRUCT(self);

	if (PgObj_IsNull(self))
		return(PYSTR("NULL"));
	else
		str = (char *) OidFunctionCall1(typs->typoutput, self->ob_datum);

	rob = PYSTRF("'%s'::%s", str, NameStr(typs->typname));
	pfree(str);

	return(rob);
}

static PyObj
obj_str(PgObj self)
{
	PyObj rob;
	char *str;
	Form_pg_type typs;

	typs = PGOTYPESTRUCT(self);

	str = (char *)OidFunctionCall1(typs->typoutput, self->ob_datum);
	rob = PYSTR(str);
	pfree(str);

	return(rob);
}

static int
obj_richcmp(PgObj self, PyObj ob)
{
	return(-1);
}

static void
obj_dealloc(PgObj self)
{
	DECREF(PgObj_FetchType(self));
	PgObj_FetchType(self) = NULL;

	PyPgDatum_Type.tp_dealloc(PyObj(self));
}

static PyObj
obj_getattr(PgObj self, char *attr)
{
	Form_pg_type typs = PGOTYPESTRUCT(self);
	PyObj rob = NULL;
	Oid to;

	if (!strcmp(attr, "type"))
	{
		rob = self->ob_pgtype;
		INCREF(rob);
		return(rob);
	}
	else if (!strcmp(attr, "size"))
	{
		int len;

		switch (typs->typlen)
		{
			/* Varlena */
			case -1:
				len = varlena_obj_length(self);
			break;

			/* Null terminated */
			case -2:
				len = nullt_obj_length(self);
			break;

			default:
				len = typs->typlen;
			break;
		}

		return(PYINT(len));
	}
	else if (!strcmp(attr, "datum"))
	{
		char *str;
		int size;

		/* XXX: get the absolute size specified by typlen rather than
		 * sizeof(Datum)*/
		if (typs->typbyval)
		{
			str = (char *)&self->ob_datum;
			size = sizeof(Datum);
		}
		else
		{
			int2 typlen = typs->typlen;

			switch (typlen)
			{
				case -1:
					str = (char *)VARDATA(self->ob_datum);
					size = (Size)VARSIZE(self->ob_datum) - VARHDRSZ;
				break;

				case -2:
					str = DatumGetPointer(self->ob_datum);
					size = strlen(str);
				break;

				default:
					str = DatumGetPointer(self->ob_datum);
					size = typlen;
				break;
			}
		}
		rob = PyString_FromStringAndSize(str, size);
		return(rob);
	}

	to = PgObj_FetchTypeOid(self);

	switch (to)
	{
		default:
			PyErr_Format(PyExc_AttributeError, "no such attribute '%s'", attr);
		break;
	}

	return(rob);
}

static PyObj
obj_setattr(PgObj self, char *attr, PgObj ob)
{
	PyObj rob = NULL;
	Oid to;
	
	if (!strcmp(attr, "type")
		|| !strcmp(attr, "size")
		|| !strcmp(attr, "datum"))
	{
		PyErr_Format(PyExc_AttributeError,
			"'%s' attribute is immutable", attr);
		return(NULL);
	}

	to = PgObj_FetchTypeOid(self);

	switch (to)
	{
		default:
			PyErr_Format(PyExc_AttributeError, "no such attribute '%s'", attr);
		break;
	}

	return(rob);
}

/*
 * Initialize a Postgres.Object based on a PgType and a Datum source
 */
int
obj_init(PgObj self, PyObj args, PyObj kw)
{
	PyObj tob, dob; /* Postgres.Type, Datum Source Object */
	Size sz;
	char *str;
	Form_pg_type typs;
	Datum dat;

	if (PyCFunctionErr_NoKeywordsAllowed(kw))
		return(-1);

	if (!PyArg_ParseTuple(args, "OO", &tob, &dob))
		return(-1);
	
	if (!PgType_TypeCheck(tob))
	{
		PyErr_Format(PyExc_TypeError,
				"first argument must be a Postgres.Type, not a '%s'", 
				tob->ob_type->tp_name
		);
		return(-1);
	}

	if (!PyString_Check(dob))
	{
		PyErr_Format(PyExc_TypeError,
				"first argument must be a Postgres.Type, not a '%s'", 
				tob->ob_type->tp_name
		);
		return(-1);
	}

	sz = PyObject_Length(dob);
	str = PYASSTR(dob);
	typs = TOTYPESTRUCT(tob);

	if (typs->typbyval)
	{
		dat = *((Datum *)str);
	}
	else
	{
		int2 typlen = typs->typlen;

		switch (typlen)
		{
			case -1:
			{
				struct varattrib *varl;
				Size tsz = sz + VARHDRSZ;

				varl = palloc(tsz);
				VARATT_SIZEP(varl) = tsz;
				memcpy(VARDATA(varl), str, sz);

				dat = PointerGetDatum(varl);
			}
			break;

			case -2:
				dat = PointerGetDatum(pstrdup(str));
			break;

			default:
			{
				char *dst;
				if (sz != typlen)
				{
					PyErr_Format(PyExc_OverflowError,
						"the type's length is %d bytes, "
						"but the given data is %d bytes", typlen, sz
					);
					return(-1);
				}

				dst = palloc(typlen);
				memcpy(dst, str, typlen);
				dat = PointerGetDatum(dst);
			}
			break;
		}
	}

	PyPgObject_Init(PyObj(self), dat, tob);
	return(0);
}

PyTypeObject PyPgObject_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Object",				/* tp_name */
	sizeof(struct PyPgObject),		/* tp_basicsize */
	0,										/* tp_itemsize */
	(destructor)obj_dealloc,		/* tp_dealloc */
	NULL,									/* tp_print */
	(getattrfunc)obj_getattr,		/* tp_getattr */
	(setattrfunc)obj_setattr,		/* tp_setattr */
	(cmpfunc)obj_compare,			/* tp_compare */
	(reprfunc)obj_repr,				/* tp_repr */
	&PyPgObjectAsNumber,				/* tp_as_number */
	&PyPgObjectAsSequence,			/* tp_as_sequence */
	PyPgObjectAsMapping,				/* tp_as_mapping */
	(hashfunc)obj_hash,				/* tp_hash */
	(ternaryfunc)NULL,				/* tp_call */
	(reprfunc)obj_str,				/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char*)PyPgObject_Doc,			/* tp_doc */
	(traverseproc)NULL,				/* tp_traverse */
	(inquiry)NULL,						/* tp_clear */
	(richcmpfunc)obj_richcmp,		/* tp_richcompare */
	(long)0,								/* tp_weaklistoffset */
	(getiterfunc)PySeqIter_New,	/* tp_iter */
	(iternextfunc)NULL,				/* tp_iternext */
	PyPgObject_Methods,				/* tp_methods */
	NULL,									/* tp_members */
	NULL,									/* tp_getset */
	&PyPgDatum_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	NULL,									/* tp_dictoffset */
	(initproc)obj_init,				/* tp_init */
	NULL,									/* tp_alloc */
	PyType_GenericNew,				/* tp_new */
};

void
PyPgObject_Init(PyObj self, Datum dat, PyObj typobj)
{
	PgObj pgo = PgObj(self);

	INCREF(typobj);
	pgo->ob_pgtype = typobj;
	pgo->ob_datum = dat;

	if (!TOTYPESTRUCT(typobj)->typbyval)
		PgObjFlags(pgo) |= PgObjFlag_Free;
}

PyObj
PyPgObject_New(Datum dat, PyObj typ)
{
	PyObj rob;
	Assert(typ != NULL && PgType_TypeCheckExact(typ));

	rob = PgObj_NEW();
	if (rob)
		PyPgObject_Init(rob, dat, typ);

	return(rob);
}

#define fetch_opOid(oprname, oprleft, oprright) \
 LookupOperName(stringToQualifiedNameList(oprname, NULL), oprleft, oprright, 1)

PyObj
PgObj_Operate(char *op, PgObj self, PyObj with)
{
	Oid selftypoid;
	PyObj rob;
	Datum rd;

	Datum withd = 0;
	Oid withtypoid;
	
	Oid opOid = 0;
	HeapTuple opTup = NULL;
	Form_pg_operator opStruct = NULL;

	selftypoid = PgObj_FetchTypeOid(self);

	if (with == NULL)
		withtypoid = 0;
	else if (PgObj_TypeCheck(with))
	{
		withd = PgObj(with)->ob_datum;
		withtypoid = PgObj_FetchTypeOid(with);
	}
	else
	{
		withd = Datum_FromPyObjectAndTypeTuple(with, PgObj_FetchTypeHT(self));
		withtypoid = selftypoid;
	}

	opOid = fetch_opOid(op, selftypoid, withtypoid);
	opTup = SearchSysCache(OPEROID, opOid, 0, 0, 0);
	if (opTup == NULL)
	{
		PyErr_Format(PyExc_PgErr, "No operator(%s).", op); /* XXX */
		return(NULL);
	}
	opStruct = OPERSTRUCT(opTup);

	rd = OidFunctionCall2(opStruct->oprcode, self->ob_datum, withd);
	rob = PgObj_FromDatumAndTypeOid(rd, opStruct->oprresult);

	ReleaseSysCache(opTup);

	return(rob);
}

PyObj
PgObj_FromDatumAndTypeOid(Datum dat, Oid typoid)
{
	PyObj typ, rob;

	typ = PgType_FromTypeOid(typoid);
	rob = PyPgObject_New(dat, typ);
	DECREF(typ);

	return(rob);
}

PyObj
PgObj_FromArrayTypeTupleAndPySequence(HeapTuple ttup, PyObj seq)
{
	ArrayType *ar;
	PyObj rob;
	ar = Array_FromTypeTupleAndPySequence(ttup, seq);

	/* Use typeoid so it can use the cached PgType */
	rob = PgObj_FromDatumAndTypeOid(
				PointerGetDatum(ar),
				HeapTuple_FetchOid(ttup)
			);

	return(rob);
}
