/* Module:          SQLDescribeCol.c
 *
 * Description:     Returns the result descriptor column name, type, 
 *					column size, decimal digits, and nullability for 
 *					one column in the result set.
 *
 * Classes:         
 *
 * API functions:   SQLDescribeCol
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLDescribeCol(
								 SQLHSTMT		hDrvStmt,
								 SQLUSMALLINT	nCol,
								 SQLCHAR		*szColName,
								 SQLSMALLINT	nColNameMax,
								 SQLSMALLINT	*pnColNameLength,
								 SQLSMALLINT	*pnSQLDataType,
								 SQLUINTEGER	*pnColSize,
								 SQLSMALLINT	*pnDecDigits,
								 SQLSMALLINT	*pnNullable
								 )
{
	static char *func="SQLDescribeCol";
	/* gets all the information about a specific column */
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	QResultClass *res;
	char *col_name = NULL;
	Int4 fieldtype = 0;
	int precision = 0;
	ConnInfo *ci;
	char parse_ok;
	char buf[255];
	int len = 0;
	SQLRETURN result;


	mylog("%s: entering...\n", func);

    if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
        return SQL_INVALID_HANDLE;
	}

	ci = &(stmt->hdbc->connInfo);

    SC_clear_error(stmt);

	/*	Dont check for bookmark column.  This is the responsibility
		of the driver manager.  
	*/

	nCol--;		/* use zero based column numbers */


	parse_ok = FALSE;
	if (globals.parse && stmt->statement_type == STMT_TYPE_SELECT)
	{

		if (stmt->parse_status == STMT_PARSE_NONE)
		{
			mylog("SQLDescribeCol: calling parse_statement on stmt=%u\n", stmt);
			parse_statement(stmt);
		}


		mylog("PARSE: DescribeCol: nCol=%d, stmt=%u, stmt->nfld=%d, stmt->fi=%u\n", nCol, stmt, stmt->nfld, stmt->fi);

		if (stmt->parse_status != STMT_PARSE_FATAL && stmt->fi && stmt->fi[nCol])
		{

			if (nCol >= stmt->nfld)
			{
				stmt->errornumber = STMT_INVALID_COLUMN_NUMBER_ERROR;
				stmt->errormsg = "Invalid column number in DescribeCol.";
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}
			mylog("DescribeCol: getting info for nCol=%d\n", nCol);

			fieldtype = stmt->fi[nCol]->type;
			col_name = stmt->fi[nCol]->name;
			precision = stmt->fi[nCol]->precision;

			mylog("PARSE: fieldtype=%d, col_name='%s', precision=%d\n", fieldtype, col_name, precision);
			if (fieldtype > 0)
			{
				parse_ok = TRUE;
			}
		}
	}


	/*	If couldn't parse it OR the field being described was not parsed (i.e., because
		it was a function or expression, etc, then do it the old fashioned way.
	*/
	if ( ! parse_ok)
	{
		SC_pre_execute(stmt);
	
		res = SC_get_Result(stmt);

		mylog("**** SQLDescribeCol: res = %u, stmt->status = %d, !finished=%d, !premature=%d\n", res, stmt->status, stmt->status != STMT_FINISHED, stmt->status != STMT_PREMATURE);
		if ( (NULL == res) || ((stmt->status != STMT_FINISHED) && (stmt->status != STMT_PREMATURE)))
		{
			/* no query has been executed on this statement */
			stmt->errornumber = STMT_SEQUENCE_ERROR;
			stmt->errormsg = "No query has been assigned to this statement.";
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		if (nCol >= QR_NumResultCols(res))
		{
			stmt->errornumber = STMT_INVALID_COLUMN_NUMBER_ERROR;
			stmt->errormsg = "Invalid column number in DescribeCol.";
			sprintf(buf, "Col#=%d, #Cols=%d", nCol, QR_NumResultCols(res));
			SC_log_error(func, buf, stmt);
			return SQL_ERROR;
		}

		col_name = QR_get_fieldname(res, nCol);
        fieldtype = QR_get_field_type(res, nCol);

		precision = pgtype_precision(stmt, fieldtype, nCol, globals.unknown_sizes);  /* atoi(ci->unknown_sizes) */
	}

	mylog("describeCol: col %d fieldname = '%s'\n", nCol, col_name);
	mylog("describeCol: col %d fieldtype = %d\n", nCol, fieldtype);
	mylog("describeCol: col %d precision = %d\n", nCol, precision);


	result = SQL_SUCCESS;

	/************************/
	/*		COLUMN NAME     */
	/************************/
	len = strlen(col_name);

	if (pnColNameLength)
	{
		*pnColNameLength = len;
	}

	if (szColName)
	{
		strncpy_null(szColName, col_name, nColNameMax);

		if (len >= nColNameMax)
		{
			result = SQL_SUCCESS_WITH_INFO;
			stmt->errornumber = STMT_TRUNCATED;
			stmt->errormsg = "The buffer was too small for the result.";
		}
    }


	/************************/
	/*		SQL TYPE        */
	/************************/
    if (pnSQLDataType)
	{
        *pnSQLDataType = pgtype_to_sqltype(stmt, fieldtype);

		mylog("describeCol: col %d *pnSQLDataType = %d\n", nCol, *pnSQLDataType);
	}

	/************************/
	/*		PRECISION       */
	/************************/
    if (pnColSize)
	{

		if ( precision < 0)
		{
			precision = 0;		/* "I dont know" */
		}

		*pnColSize = precision;

		mylog("describeCol: col %d  *pnColSize = %d\n", nCol, *pnColSize);
	}

	/************************/
	/*		SCALE           */
	/************************/
    if (pnDecDigits)
	{
        Int2 scale;
        scale = pgtype_scale(stmt, fieldtype, nCol);
        if(scale == -1)
		{ 
			scale = 0;
		}
        
        *pnDecDigits = scale;
		mylog("describeCol: col %d  *pnDecDigits = %d\n", nCol, *pnDecDigits);
    }

	/************************/
	/*		NULLABILITY     */
	/************************/
    if (pnNullable)
	{
		*pnNullable = (parse_ok) ? stmt->fi[nCol]->nullable : pgtype_nullable(stmt, fieldtype);

		mylog("describeCol: col %d  *pnNullable = %d\n", nCol, *pnNullable);
    }

    return result;
}
