/* Module:          SQLGetData.c
 *
 * Description:     Retrieves data for a single column in the result set. 
 *
 * Classes:         
 *
 * API functions:   SQLGetData
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

/* SQLRETURN SQLGetData( SQLHSTMT      hDrvStmt,
					  SQLUSMALLINT  nCol,
					  SQLSMALLINT   nTargetType,				 C DATA TYPE 
					  SQLPOINTER    pTarget,
					  SQLINTEGER    nTargetLength,
					  SQLINTEGER    *pnLengthOrIndicator )
*/

SQLRETURN SQL_API SQLGetData(
							 SQLHSTMT		hDrvStmt,
							 SQLUSMALLINT	nCol,
							 SQLSMALLINT	nTargetType,
							 SQLPOINTER		pTarget,
							 SQLINTEGER		nTargetLength,
							 SQLINTEGER		*pnLengthOrIndicator
							 )
{
	static char *func="SQLGetData";
	QResultClass *res;
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	int num_cols, num_rows;
	Int4 field_type;
	void *value = NULL;
	int result;
	char get_bookmark = FALSE;
	
	mylog("SQLGetData: enter, stmt=%u\n", stmt);

    if( ! stmt)
	{
		SC_log_error(func, "", NULL);
        return SQL_INVALID_HANDLE;
    }
	res = stmt->result;

    if (STMT_EXECUTING == stmt->status)
	{
        stmt->errormsg = "Can't get data while statement is still executing.";
        stmt->errornumber = STMT_SEQUENCE_ERROR;
		SC_log_error(func, "", stmt);
        return SQL_ERROR;
    }

    if (stmt->status != STMT_FINISHED)
	{
        stmt->errornumber = STMT_STATUS_ERROR;
        stmt->errormsg = "GetData can only be called after the successful execution on a SQL statement";
		SC_log_error(func, "", stmt);
        return SQL_ERROR;
    }

    if (nCol == 0)
	{

		if (stmt->options.use_bookmarks == SQL_UB_OFF)
		{
			stmt->errornumber = STMT_COLNUM_ERROR;
			stmt->errormsg = "Attempt to retrieve bookmark with bookmark usage disabled";
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		/*	Make sure it is the bookmark data type */
		if (nTargetType != SQL_C_BOOKMARK)
		{
			stmt->errormsg = "Column 0 is not of type SQL_C_BOOKMARK";
			stmt->errornumber = STMT_PROGRAM_TYPE_OUT_OF_RANGE;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}
		
		get_bookmark = TRUE;

    }

	else 
	{

		/* use zero-based column numbers */
		nCol--;

		/* make sure the column number is valid */
		num_cols = QR_NumResultCols(res);
		if (nCol >= num_cols)
		{
			stmt->errormsg = "Invalid column number.";
			stmt->errornumber = STMT_INVALID_COLUMN_NUMBER_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}
	}

	if ( stmt->manual_result || ! globals.use_declarefetch)
	{
		/* make sure we're positioned on a valid row */
		num_rows = QR_get_num_tuples(res);
		if((stmt->currTuple < 0) ||
		   (stmt->currTuple >= num_rows)) 
		{
			stmt->errormsg = "Not positioned on a valid row for GetData.";
			stmt->errornumber = STMT_INVALID_CURSOR_STATE_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}
		mylog("     num_rows = %d\n", num_rows);

		if ( ! get_bookmark)
		{
			if ( stmt->manual_result)
			{
				value = QR_get_value_manual(res, stmt->currTuple, nCol);
			}
			else 
			{
				value = QR_get_value_backend_row(res, stmt->currTuple, nCol);
			}
			mylog("     value = '%s'\n", value);
		}
	}
	else 
	{ /* it's a SOCKET result (backend data) */
		if (stmt->currTuple == -1 || ! res || ! res->tupleField)
		{
			stmt->errormsg = "Not positioned on a valid row for GetData.";
			stmt->errornumber = STMT_INVALID_CURSOR_STATE_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		if ( ! get_bookmark)
		{
			value = QR_get_value_backend(res, nCol);
		}

		mylog("  socket: value = '%s'\n", value);
	}

	if ( get_bookmark) 
	{
		*((UDWORD *) pTarget) = SC_get_bookmark(stmt);

		if (pnLengthOrIndicator)
		{
			*pnLengthOrIndicator = 4;
		}

		return SQL_SUCCESS;
	}

	field_type = QR_get_field_type(res, nCol);

	mylog("**** SQLGetData: nCol = %d, nTargetType = %d, field_type = %d, value = '%s'\n", nCol, nTargetType, field_type, value);

	stmt->current_col = nCol;

    result = copy_and_convert_field(stmt, field_type, value, 
                                    nTargetType, pTarget, nTargetLength, pnLengthOrIndicator);

	stmt->current_col = -1;

	switch(result)
	{
		case COPY_OK:
			return SQL_SUCCESS;

		case COPY_UNSUPPORTED_TYPE:
			stmt->errormsg = "Received an unsupported type from Postgres.";
			stmt->errornumber = STMT_RESTRICTED_DATA_TYPE_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;

		case COPY_UNSUPPORTED_CONVERSION:
			stmt->errormsg = "Couldn't handle the necessary data type conversion.";
			stmt->errornumber = STMT_RESTRICTED_DATA_TYPE_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;

		case COPY_RESULT_TRUNCATED:
			stmt->errornumber = STMT_TRUNCATED;
			stmt->errormsg = "The buffer was too small for the result.";
			return SQL_SUCCESS_WITH_INFO;

		case COPY_GENERAL_ERROR:	/* error msg already filled in */
			SC_log_error(func, "", stmt);
			return SQL_ERROR;

		case COPY_NO_DATA_FOUND:
			/* SC_log_error(func, "no data found", stmt); */
			return SQL_NO_DATA_FOUND;

		default:
			stmt->errormsg = "Unrecognized return value from copy_and_convert_field.";
			stmt->errornumber = STMT_INTERNAL_ERROR;
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
    }
}

