/*
 * PAM authentication module for PostgreSQL
 * 
 * Based in part on pam_unix.c of FreeBSD. See debian/copyright
 * for licensing details.
 *
 * David D.W. Downey ("pgpkeys") <david-downey@codecastle.com>
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <syslog.h>
#include <ctype.h>
#include <mhash.h>
#include <time.h>
#include <libpq-fe.h>
#include <crypt.h>
#include <unistd.h>

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT
#define PAM_SM_PASSWORD
#include <security/pam_modules.h>
#include "pam_mod_misc.h"

#define PASSWORD_PROMPT         "Password: "
#define PASSWORD_PROMPT_NEW	    "New password: "
#define PASSWORD_PROMPT_CONFIRM "Confirm new password: "
#define CONF                    "/etc/pam_pgsql.conf"

#define DBGLOG(x...)  if(options->debug) {                          \
                          openlog("PAM_pgsql", LOG_PID, LOG_AUTH);  \
                          syslog(LOG_DEBUG, ##x);                   \
                          closelog();                               \
                      }
#define SYSLOG(x...)  do {                                          \
                          openlog("PAM_pgsql", LOG_PID, LOG_AUTH);  \
                          syslog(LOG_INFO, ##x);                    \
                          closelog();                               \
                      } while(0);

typedef enum {
    PW_CLEAR = 1,
    PW_MD5,
    PW_CRYPT,
} pw_scheme;

struct module_options {
	/* Only below here for new config */
    pw_scheme pw_type;
    int debug;
	char *connectionstring;
	char *getpassword;
	char *isexpired;
	char *newpassrequired;
	char *changepw;
};

/* private: parse and set the specified string option */
static void
set_module_option(const char *option, struct module_options *options)
{
    char *buf, *eq;
    char *val, *end;

    if(!option || !*option)
        return;

    buf = strdup(option);

    if((eq = strchr(buf, '='))) {
        end = eq - 1;
        val = eq + 1;
        if(end <= buf || !*val)
            return;
        while(end > buf && isspace(*end))
            end--;
        end++;
        *end = '\0';
        while(*val && isspace(*val))
            val++;
    } else {
        val = NULL;
    }

    DBGLOG("setting option: %s=>%s\n", buf, val);

	if(!strcmp(buf, "connectionstring")) {
		options->connectionstring = strdup(val);
	} else if(!strcmp(buf, "getpassword")) {
		options->getpassword = strdup(val);
	} else if(!strcmp(buf, "isexpired")) {
		options->isexpired = strdup(val);
	} else if(!strcmp(buf, "newpassrequired")) {
		options->newpassrequired = strdup(val);
	} else if(!strcmp(buf, "changepw")) {
		options->changepw = strdup(val);
    } else if(!strcmp(buf, "pw_type")) {
        options->pw_type = PW_CLEAR;
        if(!strcmp(val, "md5")) {
            options->pw_type = PW_MD5;
        } else if(!strcmp(val, "crypt")) {
            options->pw_type = PW_CRYPT;
        }
    } else if(!strcmp(buf, "debug")) {
        options->debug = 1;
    }

    free(buf);
}

/* private: read module options from file or commandline */
static int 
get_module_options(int argc, const char **argv, struct module_options **options)
{
    int i, rc;
    FILE *fp;
    struct module_options *opts;

    opts = (struct module_options *)malloc(sizeof *opts);
    bzero(opts, sizeof(*opts));
    opts->pw_type = PW_CLEAR;
    rc = 0;

    if((fp = fopen(CONF, "r"))) {
        char line[1024];
        char *str, *end;

        while(fgets(line, sizeof(line), fp)) {
			/* String off all leading and trailing spaces */
            str = line;
            end = line + strlen(line) - 1;
            while(*str && isspace(*str))
                str++;
            while(end > str && isspace(*end))
                end--;
            end++;
            *end = '\0';
            set_module_option(str, opts);
        }
        
        fclose(fp);
    }

    for(i = 0; i < argc; i++) {
        if(pam_std_option(&rc, argv[i]) == 0)
            continue;
        set_module_option(argv[i], opts);
    }
    *options = opts;

    return rc;
}

/* private: free module options returned by get_module_options() */
static void
free_module_options(struct module_options *options)
{
	if(options->connectionstring)
		free(options->connectionstring);
	if(options->getpassword)
		free(options->getpassword);
	if(options->isexpired)
		free(options->isexpired);
	if(options->newpassrequired)
		free(options->newpassrequired);
	if(options->changepw)
		free(options->changepw);
							
    bzero(options, sizeof(*options));
    free(options);
}

/* private: make sure required options are present (in cmdline or conf file) */
static int
options_valid(struct module_options *options)
{
    if(options->connectionstring == 0 ||
	   options->getpassword == 0 ||
	   options->changepw == 0) 
    {
        SYSLOG("the database, table and user_column options are required.");
        return -1;
    }
    return 0;
}

/* private: open connection to PostgreSQL */
static PGconn *
pg_connect(struct module_options *options)
{
    PGconn *conn;

    conn = PQconnectdb(options->connectionstring);

    if(PQstatus(conn) != CONNECTION_OK) {
        SYSLOG("PostgreSQL connection failed: '%s'", PQerrorMessage(conn));
        return NULL;
    }

    return conn;
}

/* private: execute query */
static int
pg_res_ok(PGresult *res)
{
    if(PQresultStatus(res) != PGRES_COMMAND_OK && 
       PQresultStatus(res) != PGRES_TUPLES_OK) {
        SYSLOG("PostgreSQL query failed: '%s'", PQresultErrorMessage(res));
        return 0;
    }

    return 1;
}

/* private: convert an integer to a radix 64 character */
static int
i64c(int i)
{
	if (i <= 0)
		return ('.');

	if (i == 1)
		return ('/');

	if (i >= 2 && i < 12)
		return ('0' - 2 + i);

	if (i >= 12 && i < 38)
		return ('A' - 12 + i);

	if (i >= 38 && i < 63)
		return ('a' - 38 + i);

	return ('z');
}

/* private: generate random salt character */
static char *
crypt_make_salt(void)
{
	time_t now;
	static unsigned long x;
	static char result[3];

	time(&now);
	x += now + getpid() + clock();
	result[0] = i64c(((x >> 18) ^ (x >> 6)) & 077);
	result[1] = i64c(((x >> 12) ^ x) & 077);
	result[2] = '\0';
	return result;
}

/* private: encrypt password using the preferred encryption scheme */
static char *
encrypt_password(struct module_options *options, const char *pass)
{
    char *s = NULL;

    switch(options->pw_type) {
        case PW_CRYPT:
            s = strdup(crypt(pass, crypt_make_salt()));
            break;
        case PW_MD5: {
                char *buf;
                int buf_size;
                MHASH handle;
                unsigned char *hash;

                handle = mhash_init(MHASH_MD5);

                if(handle == MHASH_FAILED) {
                    SYSLOG("could not initialize mhash library!");
                } else {
                    int i; 

                    mhash(handle, pass, strlen(pass));
                    hash = mhash_end(handle);

                    buf_size = (mhash_get_block_size(MHASH_MD5) * 2)+1;
                    buf = (char *)malloc(buf_size);
                    bzero(buf, buf_size);

                    for(i = 0; i < mhash_get_block_size(MHASH_MD5); i++) {
                        /* should be safe */
                        sprintf(&buf[i * 2], "%.2x", hash[i]);
                    }
                    s = buf;
                }
            }
            break;
        case PW_CLEAR:
        default:
            s = strdup(pass);
    }
    return s;
}

/* private: authenticate user and passwd against database */
static int
auth_verify_password(const char *user, const char *passwd, 
                     struct module_options *options)
{
    PGresult *res;
    PGconn *conn;
    int rc;
    const char *params[1];
#define CRYPT_LEN 13

    if(!(conn = pg_connect(options)))
        return PAM_AUTH_ERR;

    params[0] = user;

	DBGLOG(params[0]);

    //DBGLOG("query: SELECT %s FROM %s WHERE %s='%s'", options->pwd_column, options->table, options->user_column, user);
    res = PQexecParams(conn, options->getpassword, 1, NULL, params, NULL, NULL, 0);
	if (!pg_res_ok(res)) {
		PQclear(res);
		PQfinish(conn);
		return PAM_AUTH_ERR;
	}

    if(PQntuples(res) == 0) {
        rc = PAM_USER_UNKNOWN;
    } else {
        char *stored_pw = PQgetvalue(res, 0, 0);
        rc = PAM_AUTH_ERR;
        switch(options->pw_type) {
            case PW_CLEAR:
                if(strcmp(passwd, stored_pw) == 0)
                    rc = PAM_SUCCESS;
                break;
            case PW_CRYPT: 
                if(strcmp(crypt(passwd, stored_pw), stored_pw) == 0)
                    rc = PAM_SUCCESS;
                break;
            case PW_MD5: {
                char *buf;
                int buf_size;
                MHASH handle;
                unsigned char *hash;

                handle = mhash_init(MHASH_MD5);

                if(handle == MHASH_FAILED) {
                    SYSLOG("could not initialize mhash library!");
                } else {
                    int i; 

                    mhash(handle, passwd, strlen(passwd));
                    hash = mhash_end(handle);

                    buf_size = (mhash_get_block_size(MHASH_MD5) * 2)+1;
                    buf = (char *)malloc(buf_size);
                    bzero(buf, buf_size);

                    for(i = 0; i < mhash_get_block_size(MHASH_MD5); i++) {
                        sprintf(&buf[i * 2], "%.2x", hash[i]);
                    }

                    if(strcmp(buf, stored_pw) == 0)
                        rc = PAM_SUCCESS;
                    free(buf);
                }
            }
            break;
        }
    }

    PQclear(res);
    PQfinish(conn);
    return rc;
}

/* public: authenticate user */
PAM_EXTERN int
pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
    struct module_options *options;
    const char *user, *password;
    int rc, std_flags;

    if((rc = pam_get_user(pamh, &user, NULL)) != PAM_SUCCESS)
        return rc;

    std_flags = get_module_options(argc, argv, &options);
    if(options_valid(options) != 0) {
        free_module_options(options);
        return PAM_AUTH_ERR;
    }

    DBGLOG("attempting to authenticate: %s", user);

    if((rc = pam_get_pass(pamh, PAM_AUTHTOK, &password, PASSWORD_PROMPT, std_flags) 
        != PAM_SUCCESS)) {
        free_module_options(options);
        return rc;
    }

    if((rc = auth_verify_password(user, password, options)) != PAM_SUCCESS) {
        free_module_options(options);
        return rc;
    }

    SYSLOG("(%s) user %s authenticated.", pam_get_service(pamh), user);
    free_module_options(options);

    return PAM_SUCCESS;
}

/* public: check if account has expired, or needs new password */
PAM_EXTERN int
pam_sm_acct_mgmt(pam_handle_t *pamh, int flags, int argc,
                            const char **argv)
{
    struct module_options *options;
    const char *user;
    const char *params[1];
    int rc;
    PGconn *conn;
    PGresult *res;

    get_module_options(argc, argv, &options);
    if(options_valid(options) != 0) {
        free_module_options(options);
        return PAM_AUTH_ERR;
    }

    /* both not specified, just succeed. */
    if(options->isexpired == 0 && options->newpassrequired == 0) {
        free_module_options(options);
        return PAM_SUCCESS;
    }

    if((rc = pam_get_user(pamh, &user, NULL)) != PAM_SUCCESS) {
        SYSLOG("could not retrieve user");
        free_module_options(options);
        return rc;
    }

    if(!(conn = pg_connect(options))) {
        free_module_options(options);
        return PAM_AUTH_ERR;
    }

    params[0] = user;

    /* if account has expired then returns at least 1 row */
	if(options->isexpired) {
        //DBGLOG("query: SELECT 1 FROM %s WHERE %s='%s' AND %s='y' OR %s='1'", options->table, options->user_column, user, options->expired_column, options->expired_column);
		res = PQexecParams(conn, options->isexpired, 1, NULL, params, NULL, NULL, 0);
		if (!pg_res_ok(res)) {
			PQclear(res);
			PQfinish(conn);
        	free_module_options(options);
			return PAM_AUTH_ERR;
		}

        if(PQntuples(res) > 0) {
            PQclear(res);
            PQfinish(conn);
            free_module_options(options);
            return PAM_ACCT_EXPIRED;
        }
        PQclear(res);
    }

    /* if new password is required then newtok_column = 'y' or '1' */
	if(options->newpassrequired) {
        //DBGLOG("query: SELECT 1 FROM %s WHERE %s='%s' AND %s='y' OR %s='1'", options->table, options->user_column, user, options->newtok_column, options->newtok_column);
		res = PQexecParams(conn, options->newpassrequired, 1, NULL, params, NULL, NULL, 0);
		if (!pg_res_ok(res)) {
			PQclear(res);
			PQfinish(conn);
			free_module_options(options);
			return PAM_AUTH_ERR;
		}

        if(PQntuples(res) > 0) {
            PQclear(res);
            PQfinish(conn);
            free_module_options(options);
            return PAM_NEW_AUTHTOK_REQD;
        }
        PQclear(res);
    }

    PQfinish(conn);
    return PAM_SUCCESS;
}

/* public: change password */
PAM_EXTERN int
pam_sm_chauthtok(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
    struct module_options *options;
    int rc, std_flags;
    const char *user, *pass, *newpass;
    char *newpass_crypt;
	const char *params[2];
    PGconn *conn;
    PGresult *res;

    std_flags = get_module_options(argc, argv, &options);
    if(options_valid(options) != 0) {
        free_module_options(options);
        return PAM_AUTH_ERR;
    }

    if((rc = pam_get_user(pamh, &user, NULL)) != PAM_SUCCESS) {
        free_module_options(options);
        return rc;
    }

    if(!(conn = pg_connect(options))) {
        free_module_options(options);
        return PAM_AUTH_ERR;
    }

    if(flags & PAM_PRELIM_CHECK) {
        return PAM_SUCCESS;
    } else if(flags & PAM_UPDATE_AUTHTOK) {
	/* only try to check old password if user is not root */
	if(getuid() != 0) {
    	    if((rc = pam_get_pass(pamh, PAM_OLDAUTHTOK, &pass, PASSWORD_PROMPT, std_flags)) == PAM_SUCCESS) {
        	if((rc = auth_verify_password(user, pass, options)) == PAM_SUCCESS) {
            	    rc = pam_set_item(pamh, PAM_OLDAUTHTOK, (const void *)pass);
            	    if(rc != PAM_SUCCESS) {
                	SYSLOG("failed to set PAM_OLDAUTHTOK!");
            	        free_module_options(options);
            	        return rc;
            	    }
        	} else {
            	    DBGLOG("password verification failed for '%s'", user);
            	    return rc;
        	}
    	    } else {
        	SYSLOG("could not retrieve password from '%s'", user);
        	return PAM_AUTH_ERR;
    	    }
	}

        /* get and confirm the new passwords */
        rc = pam_get_confirm_pass(pamh, &newpass, PASSWORD_PROMPT_NEW, PASSWORD_PROMPT_CONFIRM, std_flags);
        if(rc != PAM_SUCCESS) {
            SYSLOG("could not retrieve new authentication tokens");
            free_module_options(options);
            return rc;
        }
      
       /* save the new password for subsequently stacked modules */
       rc = pam_set_item(pamh, PAM_AUTHTOK, (const void *)newpass);
       if(rc != PAM_SUCCESS) {
           SYSLOG("failed to set PAM_AUTHTOK!");
           free_module_options(options);
           return rc;
       }

        /* update the database */
        if(!(newpass_crypt = encrypt_password(options, newpass))) {
            free_module_options(options);
            return PAM_BUF_ERR;
        }
        if(!(conn = pg_connect(options))) {
            free_module_options(options);
            return PAM_AUTHINFO_UNAVAIL;
        }

		params[0] = user;
		params[1] = newpass_crypt;

        //DBGLOG("query: UPDATE %s SET %s='%s' WHERE %s='%s'", options->table, options->pwd_column, "******", options->user_column, user);
		res = PQexecParams(conn, options->changepw, 2, NULL, params, NULL, NULL, 0);
		if(!pg_res_ok(res)) {
            free(newpass_crypt);
            free_module_options(options);
			PQclear(res);
            PQfinish(conn);
            return PAM_AUTH_ERR;
        }

        /* if we get here, we must have succeeded (pg_exec checks for success) */
        free(newpass_crypt);
        PQclear(res);
        PQfinish(conn);
    }

    free_module_options(options);
    SYSLOG("(%s) password for '%s' was changed.", pam_get_service(pamh), user);
    return PAM_SUCCESS;
}

/* public: just succeed. */
PAM_EXTERN int
pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return PAM_SUCCESS;
}
