/*
*   Author: Scott Bailey
*   License: BSD
*   
*   Purpose: Creates period type and base functions.
*
*   This file provides compatability with the temporal project written by Jeff Davis.
*   http://pgfoundry.org/projects/temporal/
*   The temporal project uses a custom type written in C and is generally a bit faster
*   than the composite type that this file provides. The temporal project requires
*   all users to compile from source, which can be a pain on some platforms (Windows).
*
*/

CREATE TYPE period AS (
	start_time		timestampTz,
	end_time    	timestampTz
);


/*****************************************************************
*                          Constructors
*****************************************************************/


/* 
parse string in one of the following formats:
 [start, next) (default)
 [start, last]
 (prev,  next)
 (prev,  last]
*/
CREATE OR REPLACE FUNCTION period(
	varchar
) RETURNS period AS
$$
	SELECT CASE WHEN array_upper(matches, 1) = 4 THEN
	(
		CASE matches[1] WHEN '[' THEN 
		  matches[2]::timestampTz
		WHEN '('  THEN 
		  next(matches[2]::timestampTz) END,
		CASE matches[4] WHEN ']'THEN 
		  next(matches[3]::timestampTz)
		WHEN ')' THEN 
		  matches[3]::timestampTz END
	)::period END
	FROM (
		SELECT regexp_matches($1, E'^(\\(|\\[)(.*), (.*)(\\)|\\])$') AS matches
	) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;

-- constructor [start, next)
CREATE OR REPLACE FUNCTION period(
	timestampTz,
	timestampTz
) RETURNS period AS
$$
	SELECT CASE WHEN $1 <= $2
    THEN ($1, $2)::period
    ELSE ($2, $1)::period END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


/*****************************************************************
*                         Fact Functions
*****************************************************************/

-- first inclusive value in period
CREATE OR REPLACE FUNCTION first(period)
RETURNS timestampTz AS
$$
    SELECT $1.start_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- last inclusive value in period
CREATE OR REPLACE FUNCTION last(period)
RETURNS timestampTz AS
$$
	SELECT $1.end_time - period_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- last exclusive value before the start of period
CREATE OR REPLACE FUNCTION prior(period)
RETURNS timestampTz AS
$$
	SELECT $1.start_time - period_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- first exclusive value after end of period
CREATE OR REPLACE FUNCTION next(period)
RETURNS timestampTz AS
$$
    SELECT $1.end_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- 
CREATE OR REPLACE FUNCTION length(period)
RETURNS interval AS
$$
    SELECT $1.end_time - $1.start_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


/*****************************************************************
*                      Boolean Functions
*****************************************************************/


-- p1 = p2
CREATE OR REPLACE FUNCTION equals(period, period)
RETURNS boolean AS
$$
	SELECT $1.start_time = $2.start_time AND $1.end_time = $2.end_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- p1 != p2
CREATE OR REPLACE FUNCTION nequals(period, period)
RETURNS boolean AS
$$
	SELECT $1.start_time != $2.start_time OR $1.end_time != $2.end_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- p1.last < p2.first
CREATE OR REPLACE FUNCTION before(period, period)
RETURNS boolean AS
$$
    SELECT last($1) < first($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- p1.first > p2.last
CREATE OR REPLACE FUNCTION after(period, period)
RETURNS boolean AS
$$
    SELECT first($1) > last($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;

-- p1 overlaps p2
CREATE OR REPLACE FUNCTION overlaps(period, period)
RETURNS boolean AS
$$
    SELECT ($1).start_time < ($2).end_time 
    AND ($1).end_time > ($2).start_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- p1.next = p2.first OR p1.first = p2.next
CREATE OR REPLACE FUNCTION adjacent(period, period)
RETURNS boolean AS
$$
    SELECT first($1) = next($2)
      OR next($1) = first($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- p1 contains all values of p2
CREATE OR REPLACE FUNCTION contains(period, period)
RETURNS boolean AS
$$
    SELECT $1.start_time <= $2.start_time AND $1.end_time >= $2.end_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- ts1 between p1.first and p1.last
CREATE OR REPLACE FUNCTION contains(period, timestampTz)
RETURNS boolean AS
$$
    SELECT $2 BETWEEN $1.start_time AND last($1);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- all values of p1 are contained by p2
CREATE OR REPLACE FUNCTION contained_by(period, period)
RETURNS boolean AS
$$
    SELECT $2.start_time <= $1.start_time AND $2.end_time >= $1.end_time;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- ts contained by p2
CREATE OR REPLACE FUNCTION contained_by(timestampTz, period)
RETURNS boolean AS
$$
    SELECT $1 BETWEEN $2.start_time AND last($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/*****************************************************************
*                         Set Functions
*****************************************************************/

-- Returns union of p1 and p2 if they overlap or are adjacent
-- otherwise null is returned.
-- Note: temporal raises an exception in above case
CREATE OR REPLACE FUNCTION period_union(period, period)
RETURNS period AS
$$
  SELECT CASE WHEN overlaps($1, $2) OR adjacent($1, $2)
  THEN (LEAST($1.start_time, $2.start_time),
  GREATEST($1.end_time, $2.end_time))::period END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;

-- Returns all values in p1 that are also in p2
CREATE OR REPLACE FUNCTION period_intersect(period, period)
RETURNS period AS
$$
    SELECT CASE WHEN overlaps($1, $2)
    THEN (GREATEST($1.start_time, $2.start_time),
    LEAST($1.end_time, $2.end_time))::period END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;

-- Remove any values from p1 that are contained in p2
-- if p1 contains p2 null is returned.
-- note: temporal raises an exception in above case
-- The period_except() function returns a period array and
-- will return both pieces if p1 contains p2
CREATE OR REPLACE FUNCTION period_minus(period, period)
RETURNS period AS
$$ 
    SELECT CASE WHEN contains($1, $2)
    THEN NULL
    WHEN $1.start_time < $2.start_time
    THEN ($1.start_time, LEAST($1.end_time, $2.start_time))::period
    ELSE (GREATEST($1.end_time, $2.start_time), $1.end_time)::period END;
$$ LANGUAGE 'sql' IMMUTABLE;

/*****************************************************************
*                          Operators
*****************************************************************/

CREATE OPERATOR &&(
  PROCEDURE = overlaps,
  LEFTARG = period,
  RIGHTARG = period
);
  
CREATE OPERATOR @>(
  PROCEDURE = contains,
  LEFTARG = period,
  RIGHTARG = timestampTz
);

CREATE OPERATOR @>(
  PROCEDURE = contains,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR <@(
  PROCEDURE = contained_by,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR <@(
  PROCEDURE = contained_by,
  LEFTARG = timestampTz,
  RIGHTARG = period
);

CREATE OPERATOR +(
  PROCEDURE = period_union,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR <<(
  PROCEDURE = before,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR >>(
  PROCEDURE = after,
  LEFTARG = period,
  RIGHTARG = period
);
