/*
 * Copyright (c) 2004-2025 Tada AB and other contributors, as listed below.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the The BSD 3-Clause License
 * which accompanies this distribution, and is available at
 * http://opensource.org/licenses/BSD-3-Clause
 *
 * Contributors:
 *   Tada AB
 *   Purdue University
 */
package org.postgresql.pljava.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation that supplies verbatim commands to be copied into the
 * deployment descriptor.
 *<p>
 * Strings supplied to {@link #install install} or {@link #remove remove} within
 * a single {@code SQLAction} annotation become code snippets emitted into the
 * deployment descriptor's {@code INSTALL} or {@code REMOVE} section,
 * respectively, in the order supplied.
 *<p>
 * Snippets from different {@code SQLAction} annotations,
 * and snippets generated by annotations on functions, types, and such, will be
 * assembled in  an order that can be influenced by {@link #provides provides}
 * and {@link #requires requires} labels. No snippet X will be emitted as an
 * {@code INSTALL} action ahead of any snippets that provide what X requires.
 * The sense of that dependency is reversed when ordering {@code REMOVE}
 * snippets.
 *<h2>Conditional execution</h2>
 *<p>
 * An {@code SQLAction} may supply an {@code install} snippet that tests some
 * condition at the time of deployment and adjusts the
 * {@code pljava.implementors} setting to include or not include a specific
 * {@code <implementor name>}, controlling whether actions later in
 * the deployment descriptor that are annotated with that
 * {@code <implementor name>} will be executed. The {@code SQLAction} that
 * controls whether an {@code <implementor name>} will be recognized should use
 * {@link #provides provides} with exactly that name, which is implicitly
 * 'required' by statements that use that name as
 * {@link #implementor implementor}. For details on this usage, which involves
 * a different ordering rule, see "conditional execution" in
 * {@link org.postgresql.pljava.annotation the package documentation}.
 * @author Thomas Hallgren - pre-Java6 version
 * @author Chapman Flack (Purdue Mathematics) - updated to Java6,
 * added SQLAction
 */
@Documented
@Target({ElementType.PACKAGE,ElementType.TYPE})
@Repeatable(SQLActions.class)
@Retention(RetentionPolicy.CLASS)
public @interface SQLAction
{
	/**
	 * SQL commands to be included in a BEGIN INSTALL group (each string
	 * considered a separate command to which a semicolon will be added).
	 */
	String[] install() default {};

	/**
	 * SQL commands to be included in a BEGIN REMOVE group (each string
	 * considered a separate command to which a semicolon will be added).
	 */
	String[] remove() default {};

	/**
	 * One or more arbitrary labels that will be considered 'provided' by the
	 * object carrying this annotation. The deployment descriptor will be
	 * generated in such an order that other objects that 'require' labels
	 * 'provided' by this come later in the output for install actions, and
	 * earlier for remove actions.
	 *<p>
	 * For use of this element on an {@code SQLAction} that tests a condition
	 * to control conditional execution, see "conditional execution" in
	 * {@link SQLAction the class description}.
	 */
	String[] provides() default {};

	/**
	 * One or more arbitrary labels that will be considered 'required' by the
	 * object carrying this annotation. The deployment descriptor will be
	 * generated in such an order that other objects that 'provide' labels
	 * 'required' by this come earlier in the output for install actions, and
	 * later for remove actions.
	 */
	String[] requires() default {};

	/**
	 * The {@code <implementor name>} to be used around SQL code generated
	 * here. Defaults to {@code PostgreSQL}. Set explicitly to {@code ""} to
	 * emit code not wrapped in an {@code <implementor block>}.
	 */
	String implementor() default "";
}
