/*****************************************************************************/

/*
 *      misc.c  --  Miscellaneous routines for the soundmodem.
 *
 *      Copyright (C) 1996  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *        Swiss Federal Institute of Technology (ETH), Electronics Lab
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *  This is the Linux realtime sound output driver
 */

/*****************************************************************************/

#define _GNU_SOURCE
#include <stdio.h>

#include "sm.h"

/* --------------------------------------------------------------------- */

int snprintpkt(char *buf, size_t sz, const u_int8_t *pkt, unsigned len)
{
	u_int8_t v1=1 , cmd=0;
	u_int8_t i, j;
	int ret;
	size_t cursz = sz;

	if (sz < 1)
		return 0;
	*buf = 0;
	if (!pkt || len < 8)
		return 0;
	if (pkt[1] & 1) {
		/*
		 * FlexNet Header Compression
		 */
		v1 = 0;
		cmd = (pkt[1] & 2) != 0;
		if ((ret = snprintf(buf, cursz, "fm ? to ")) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
		i = (pkt[2] >> 2) & 0x3f;
		if (i) {
			if ((ret = snprintf(buf, cursz, "%c",i+0x20)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		i = ((pkt[2] << 4) | ((pkt[3] >> 4) & 0xf)) & 0x3f;
		if (i) {
			if ((ret = snprintf(buf, cursz, "%c", i+0x20)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		i = ((pkt[3] << 2) | ((pkt[4] >> 6) & 3)) & 0x3f;
		if (i) {
			if ((ret = snprintf(buf, cursz, "%c", i+0x20)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		i = pkt[4] & 0x3f;
		if (i) {
			if ((ret = snprintf(buf, cursz, "%c", i+0x20)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		i = (pkt[5] >> 2) & 0x3f;
		if (i) {
			if ((ret = snprintf(buf, cursz, "%c", i+0x20)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		i = ((pkt[5] << 4) | ((pkt[6] >> 4) & 0xf)) & 0x3f;
		if (i) {
			if ((ret = snprintf(buf, cursz, "%c", i+0x20)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		if ((ret = snprintf(buf, cursz, "-%u QSO Nr %u", pkt[6] & 0xf, 
				    (pkt[0] << 6) | (pkt[1] >> 2))) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
		pkt += 7;
		len -= 7;
	} else {
		/*
		 * normal header
		 */
		if (len < 15)
			return 0;
		if ((pkt[6] & 0x80) != (pkt[13] & 0x80)) {
			v1 = 0;
			cmd = (pkt[6] & 0x80);
		}
		if ((ret = snprintf(buf, cursz, "fm ")) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
		for(i = 7; i < 13; i++) 
			if ((pkt[i] &0xfe) != 0x40) {
				if ((ret = snprintf(buf, cursz, "%c", pkt[i] >> 1)) == -1)
					return -1;
				buf += ret;
				cursz -= ret;
			}
		if ((ret = snprintf(buf, cursz, "-%u to ", (pkt[13] >> 1) & 0xf)) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
		for(i = 0; i < 6; i++) 
			if ((pkt[i] &0xfe) != 0x40) {
				if ((ret = snprintf(buf, cursz, "%c", pkt[i] >> 1)) == -1)
					return -1;
				buf += ret;
				cursz -= ret;
			}
		if ((ret = snprintf(buf, cursz, "-%u", (pkt[6] >> 1) & 0xf)) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
		pkt += 14;
		len -= 14;
		if ((!(pkt[-1] & 1)) && (len >= 7)) {
			if ((ret = snprintf(buf, cursz, " via ")) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		while ((!(pkt[-1] & 1)) && (len >= 7)) {
			for(i = 0; i < 6; i++) 
				if ((pkt[i] &0xfe) != 0x40) {
					if ((ret = snprintf(buf, cursz, "%c", pkt[i] >> 1)) == -1)
						return -1;
					buf += ret;
					cursz -= ret;
				}
			if ((ret = snprintf(buf, cursz, "-%u", (pkt[6] >> 1) & 0xf)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			pkt += 7;
			len -= 7;
			if ((!(pkt[-1] & 1)) && (len >= 7)) {
				if ((ret = snprintf(buf, cursz, ",")) == -1)
					return -1;
				buf += ret;
				cursz -= ret;
			}
		}
	}
	if(!len) 
		return sz - cursz;
	i = *pkt++;
	len--;
	j = v1 ? ((i & 0x10) ? '!' : ' ') : 
		((i & 0x10) ? (cmd ? '+' : '-') : (cmd ? '^' : 'v'));
	if (!(i & 1)) {
		/*
		 * Info frame
		 */
		if ((ret = snprintf(buf, cursz, " I%u%u%c", (i >> 5) & 7, (i >> 1) & 7, j)) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
	} else if (i & 2) {
		/*
		 * U frame
		 */
		switch (i & (~0x10)) {
		case 0x03:
			if ((ret = snprintf(buf, cursz, " UI%c", j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x2f:
			if ((ret = snprintf(buf, cursz, " SABM%c", j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x43:
			if ((ret = snprintf(buf, cursz, " DISC%c", j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x0f:
			if ((ret = snprintf(buf, cursz, " DM%c", j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x63:
			if ((ret = snprintf(buf, cursz, " UA%c", j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x87:
			if ((ret = snprintf(buf, cursz, " FRMR%c", j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		default:
			if ((ret = snprintf(buf, cursz, " unknown U (0x%x)%c", i & (~0x10), j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		}
	} else {
		/*
		 * supervisory
		 */
		switch (i & 0xf) {
		case 0x1:
			if ((ret = snprintf(buf, cursz, " RR%u%c", (i >> 5) & 7, j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x5:
			if ((ret = snprintf(buf, cursz, " RNR%u%c", (i >> 5) & 7, j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		case 0x9:
			if ((ret = snprintf(buf, cursz, " REJ%u%c", (i >> 5) & 7, j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		default:
			if ((ret = snprintf(buf, cursz, " unknown S (0x%x)%u%c", i & 0xf, 
					    (i >> 5) & 7, j)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
			break;
		}
	}
	if (!len) {
		if ((ret = snprintf(buf, cursz, "\n")) == -1)
			return -1;
		return sz - cursz + ret;
	}
	if ((ret = snprintf(buf, cursz, " pid=%02X\n", *pkt++)) == -1)
		return -1;
	buf += ret;
	cursz -= ret;
	len--;
	j = 0;
	while (len) {
		i = *pkt++;
		if ((i >= 32) && (i < 128)) {
			if ((ret = snprintf(buf, cursz, "%c", i)) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		else if (i == 13) {
			if (j) {
				if ((ret = snprintf(buf, cursz, "\n")) == -1)
					return -1;
				buf += ret;
				cursz -= ret;
			}
			j = 0;
		} else {
			if ((ret = snprintf(buf, cursz, ".")) == -1)
				return -1;
			buf += ret;
			cursz -= ret;
		}
		if (i >= 32) 
			j = 1;
		len--;
	}
	if (j) {
		if ((ret = snprintf(buf, cursz, "\n")) == -1)
			return -1;
		buf += ret;
		cursz -= ret;
	}
	return sz - cursz;
}

