/*****************************************************************************/

/*
 *      util.c  --  utility functions.
 *
 *      Copyright (C) 1996  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *        Swiss Federal Institute of Technology (ETH), Electronics Lab
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

/*****************************************************************************/
 
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>

#include "util.h"

/* --------------------------------------------------------------------- */

#ifndef HAVE_ANON_MMAP

static int zero_fd = -1;

extern void *mmap_anon(void *start, size_t length, int prot, int flags)
{
	if (zero_fd == -1) {
		zero_fd = open("/dev/zero", O_RDWR);
		if (zero_fd == -1) {
			perror("open: /dev/zero");
			exit(1);
		}
	}
	return mmap(start, length, prot, flags, zero_fd, 0);
}
#endif /* HAVE_ANON_MMAP */

/* --------------------------------------------------------------------- */

#ifndef HAVE_SNPRINTF
int snprintf(char *str, size_t n, const char *format, ...)
{
	va_list ap;
	int ret;

	va_start(ap, format);
	ret = vsnprintf(str, n, format, ap);
	va_end(ap);
	return ret;
}
#endif

#ifndef HAVE_VSNPRINTF
int vsnprintf(char *str, size_t n, const char *format, va_list ap)
{
	static void *xbuf = MAP_FAILED;
	static unsigned sz = 0, pgsz = 0;
	int ret;

	if (xbuf == MAP_FAILED) {
		pgsz = getpagesize();
		sz = pgsz * 2;
		xbuf = mmap_anon(NULL, sz+pgsz, PROT_READ | PROT_WRITE, MAP_PRIVATE);
		if (xbuf == MAP_FAILED) {
			perror("mmap_anon");
			exit(1);
		}
		munmap((u_int8_t)xbuf + sz, pgsz);
	}
	if (n > sz) {
		fprintf(stderr, "vsnprintf emulation: size too large, %u > %u\n", 
			n, sz);
		exit(1);
	}
	ret = vsprintf(xbuf, format, ap);
	if (ret >= n)
		return -1;
	if (ret < 0)
		return ret;
	memcpy(str, xbuf, ret+1);
	return ret;
}
#endif

/* --------------------------------------------------------------------- */
