/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

// keyboard.cpp

#include "keyboard.h"
#include "download.h"

MainScreen* Keyboard_input_with_data::mainscreen_ptr;
SendWin* Keyboard_input_with_data::sendwin_ptr;
ReceiveWin* Keyboard_input_with_data::receivewin_ptr;
DList<Screen>* Keyboard_input_with_data::screenform_ptr;
Transmit_buffer* Keyboard_input_with_data::tr_buffer_ptr;
Tnc* Keyboard_input_with_data::tnc_ptr;
BufferList* Keyboard_input_with_data::buffer_list_ptr;
Keyboard_input* Keyboard_input_with_data::keyboard_ptr;
Keyboard_input* Keyboard_input_with_data::default_input_ptr;
Keyboard_funcs Keyboard_input_with_data::keyboard_funcs;

// define the helper class

// the following matrix contains the assigned key representing numbers, for 3 series
// of keys F1 to F4 (one for linux console, one for xterm and one for any user keycode
// supplied with F1_KEYCODE (passed by the Makefile))

int Keyboard_input_with_data::F_KEYmatrix[3][7];

// now we need to define some key codes for F1 to F4 in case terminfo does not define them
// properly

// check first that ncurses is of a sufficiently recent version
#if NCURSES_VERSION_PATCH < 970816
#error :  ncurses 4.2 or above must be installed.
#endif

#define ESC_STR "\033"
#define ESC_CHAR '\033'

// PRINT_MARK is the mark which will be printed to screen when a print mark is set
#define PRINT_MARK 167

const int ADDED_FKEY0 = 1000;
inline int ADDED_FKEY(unsigned char n) {
    if (n > 0 && n < 5) return (n - 1);
    else if (n > 9 && n < 13) return (n - 6);
    else return 0;
}

const char LINUX_F1[] = ESC_STR"[[A";
const char XTERM_F1[] = ESC_STR"[11~";
const char F10[] = ESC_STR"[21~";
const char F11[] = ESC_STR"[23~";
const char F12[] = ESC_STR"[24~";

// the following function is passed to the constructor of the first object defined and deriving from
// the Screen class in screen.h

void curses_init(void) {
// set up basic keyboard functions
    nodelay(stdscr, TRUE);
    noecho();
    cbreak();
    raw();
    nl();
    keypad(stdscr, TRUE);

// now make sure the Alt (meta) key generates a Esc character and doesn't set the 8th bit
    system("setmetamode esc < /dev/tty");

    int index;
    char keycode[10];

// initialise F_KEYmatrix
    int keynumber = ADDED_FKEY0 + 1;
    int index2;
    for (index = 0; index < 3; index++) {
        for (index2 = 0; index2 < 7; index2++) {
	    Keyboard_input_with_data::F_KEYmatrix[index][index2] = keynumber;
	    keynumber++;
	}
    }

// now define the keycodes for F1 - F4
    strcpy(keycode, LINUX_F1);
    for (index = 0; index < 4; index++, keycode[3]++) {
        define_key(keycode, Keyboard_input_with_data::F_KEYmatrix[0][index]);
    }
    strcpy(keycode, XTERM_F1);
    for (index = 0; index < 4; index++, keycode[3]++) {
        define_key(keycode, Keyboard_input_with_data::F_KEYmatrix[1][index]);
    }
    if (strlen(F1_USERCODE) && strlen(F1_USERCODE) < 8) {
        strcpy(keycode, ESC_STR);
	strcat(keycode, F1_USERCODE);
// find the last alphanumeric
	char* letter = keycode + strlen(keycode);
	while (!isalnum((unsigned char)*letter) && letter > keycode) letter--;
	if (letter > keycode) {
	    for (index = 0; index < 4; index++, (*letter)++) {
	        define_key(keycode, Keyboard_input_with_data::F_KEYmatrix[2][index]);
	    }
	}
    }
// now define default keycodes for Linux console and xterm for F10 - F12
    define_key((char*)F10, Keyboard_input_with_data::F_KEYmatrix[0][ADDED_FKEY(10)]);
    define_key((char*)F11, Keyboard_input_with_data::F_KEYmatrix[0][ADDED_FKEY(11)]);
    define_key((char*)F12, Keyboard_input_with_data::F_KEYmatrix[0][ADDED_FKEY(12)]);
}

int Keyboard_input::input_confirm(void) {
  // this returns -1 if there is not valid input,
  // or 1 if the answer is yes, or 0 if the answer is no
    int result = -1;
    int letter = getch();
    if (letter == 'y' || letter == 'Y' || letter == '\n') result = 1;
    if (letter == 'n' || letter == 'N') result = 0;
    return result;
}

int Keyboard_input::action_pending(fd_set* fds_ptr) const {
    return FD_ISSET(0, fds_ptr);
}

void Keyboard_input_with_data::set_data(MainScreen* a, SendWin* b, ReceiveWin* c, DList<Screen>* d,
			      Transmit_buffer* e, Tnc* f, BufferList* g, Keyboard_input* h) {
    keyboard_funcs.mainscreen_ptr = mainscreen_ptr = a;
    keyboard_funcs.sendwin_ptr = sendwin_ptr = b;
    keyboard_funcs.receivewin_ptr = receivewin_ptr = c;
    keyboard_funcs.screenform_ptr = screenform_ptr = d;
    keyboard_funcs.tr_buffer_ptr = tr_buffer_ptr = e;
    keyboard_funcs.tnc_ptr = tnc_ptr = f;
    keyboard_funcs.buffer_list_ptr = buffer_list_ptr = g;
    default_input_ptr = h;
}

int Keyboard_input_with_data::action_pending(fd_set* fds_ptr) const {
    if (FD_ISSET(0, fds_ptr)
	   || keyboard_funcs.connect_script_flag == Keyboard_funcs::running
	   || keyboard_funcs.send_parms_flag == Keyboard_funcs::running) 
        return TRUE;
    else return FALSE;
}

void Exit_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) prog_func.exitflag = TRUE;
    }
}

void Exit_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in exit_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Close down (Y/n)?", 1, 11);
    popupwin_ptr->winrefresh();
}


void Help_file::setup(void) {
    receivewin_ptr->display_helpfile();
    keyboard_ptr = this;
}

void Help_file::action(void) {
    run_funcs();
    int letter = getch();
    if (letter == KEY_F(1)  || letter == F_KEYmatrix[0][ADDED_FKEY(1)]
		    || letter == F_KEYmatrix[1][ADDED_FKEY(1)]
		    || letter == F_KEYmatrix[2][ADDED_FKEY(1)]) {
        keyboard_ptr = default_input_ptr;
	receivewin_ptr->end_helpfile();
	refresh_winlist(*screenform_ptr);
	receivewin_ptr->redisplay();
    }
    else if (letter ==  KEY_UP) {
        receivewin_ptr->scrollup_helpwin();
    }

    else if (letter == KEY_DOWN) {
        receivewin_ptr->scrolldown_helpwin();
    }
}

void Openfile::setup(Mode mode_) {
    mode = mode_;
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin(6, 52, 10, Screen::get_ncols()/2 - 26);
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in Openfile::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();

    if (buffer_list_ptr->get_upload_status(tnc_ptr->tnc_func.active_stream(),
	                              tnc_ptr->tnc_func.active_port) == BufferList::file) {
	stop_sending_query_flag = TRUE;
        popupwin_ptr->write("Stop sending the file? (Y/n)", 1, 4);
    }
    else {
	stop_sending_query_flag = FALSE;
        popupwin_ptr->set_cursor_status(Screen::on);
	popupwin_ptr->write("File to be sent? (^Q to quit)", 1, 4);
	popupwin_ptr->write("", 2, 4);
	index = 0;
	if (strlen(prog_func.filedir) < 254) {
	    for (char* temp_ptr = prog_func.filedir; *temp_ptr != 0 && index < 255; temp_ptr++, index++) {
	      filename[index] = *temp_ptr;
	      popupwin_ptr->write(filename[index]);
	    }
	    filename[index] = '/';
	    popupwin_ptr->write('/');
	    index++;
	}
    }
    popupwin_ptr->winrefresh();
}

void Openfile::action(void) {
    run_funcs();

    if (stop_sending_query_flag) {
        if (buffer_list_ptr->get_upload_status(tnc_ptr->tnc_func.active_stream(),
	                                          tnc_ptr->tnc_func.active_port) == BufferList::file) {
	    int result = input_confirm();
	    if (result != -1) {
	        keyboard_ptr = default_input_ptr;
		popupwin_ptr->unshow();
		delete popupwin_ptr;
		refresh_winlist(*screenform_ptr);
		if (result == 1) { // we need to find the file buffer sending the
	                           // message on the active stream and port
		    buffer_list_ptr->reset(DList_enum::bottom_end);
		    FileBuffer* file_buffer_ptr;
		    while ((file_buffer_ptr = (FileBuffer*)buffer_list_ptr->inspect(DList_enum::up)) != 0
			   && !(file_buffer_ptr->get_stream() == tnc_ptr->tnc_func.active_stream()
				&& file_buffer_ptr->get_port() == tnc_ptr->tnc_func.active_port)) {}
		    if (file_buffer_ptr) file_buffer_ptr->end_loading();
		    else {
		        receivewin_ptr->write("\nOops - can't find the file buffer to delete\n");
			receivewin_ptr->winrefresh();
			beep();
			doupdate();
		    }
		}
	    }
	}
	else { // the file must completed uploading by itself while we were waiting for an answer to the prompt!
	    keyboard_ptr = default_input_ptr;
	    popupwin_ptr->unshow();
	    delete popupwin_ptr;
	    refresh_winlist(*screenform_ptr);
	}
    }

    else {
        int letter = getch();
	if (letter > 44 && letter < 123) {
	    filename[index++] = letter;
	    popupwin_ptr->write(letter);
	    popupwin_ptr->winrefresh();
	}
	else if (letter == KEY_BACKSPACE && index) {
	    index--;
	    popupwin_ptr->write(8);
	    popupwin_ptr->write(32);
	    popupwin_ptr->write(8);
	    popupwin_ptr->winrefresh();
	}
	if ((letter == '\n' && index > 0) || index == 255) {
	    filename[index] = 0;
	    keyboard_ptr = default_input_ptr;
	    popupwin_ptr->unshow();
	    delete popupwin_ptr;
	    refresh_winlist(*screenform_ptr);
	    FileBuffer::Buffer_mode b_mode;
	    if (mode == Openfile::text) b_mode = FileBuffer::text;
	    else if (mode == Openfile::binary) b_mode = FileBuffer::binary;
	    else b_mode = FileBuffer::s_plus;
	    FilesendBuffer* filesend_ptr = new FilesendBuffer(b_mode,
		tnc_ptr->tnc_func.active_stream(), tnc_ptr->tnc_func.active_port, tnc_ptr,
		      sendwin_ptr, receivewin_ptr, screenform_ptr);
	    if (!filesend_ptr) {
	        cerr << "Memory allocation error in Openfile::action()" << endl;
		exit(MEM_ERROR);
	    }
	    if (filesend_ptr->open_file(filename, FALSE)) buffer_list_ptr->add(filesend_ptr);
	    else delete filesend_ptr;
	}
	if (letter == 17) {
	    keyboard_ptr = default_input_ptr;
	    popupwin_ptr->unshow();
	    delete popupwin_ptr;
	    refresh_winlist(*screenform_ptr);
	}
    }
}

void Input_command::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin(6, 52, 10, Screen::get_ncols()/2 - 26);
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in input_command::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->set_cursor_status(Screen::on);
    popupwin_ptr->write("Command? (^Q to quit)", 1, 2);
    popupwin_ptr->write("(Press F5 again to enter a configuration script)", 2, 2);
    popupwin_ptr->write("", 3, 2);
    popupwin_ptr->winrefresh();
    index = 0;
    get_filename_flag = FALSE;
}

void Input_command::action(void) {
    run_funcs();
    int letter = getch();
    if ((letter > 044 && letter < 127)
	|| (!get_filename_flag && letter == ' ')) {
        buffer[index++] = letter;
	popupwin_ptr->write(letter);
	popupwin_ptr->winrefresh();
    }
    else if (letter == KEY_BACKSPACE && index){
        index--;
	popupwin_ptr->write(8);
	popupwin_ptr->write(32);
	popupwin_ptr->write(8);
	popupwin_ptr->winrefresh();
    }

    else if (!get_filename_flag && letter == KEY_F(5)) {
	get_filename_flag = TRUE;
        popupwin_ptr->unshow();

	popupwin_ptr->show();
        popupwin_ptr->set_cursor_status(Screen::on);
	popupwin_ptr->write("Configuration to be set? (^Q to quit)", 1, 2);
	popupwin_ptr->write("", 2, 2);
	popupwin_ptr->winrefresh();
	index = 0;
    }

    if ((letter == '\n' && index > 0) || index == 254) {
	buffer[index] = 0;
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (!get_filename_flag) tnc_ptr->send_kamcommand(buffer);
	else keyboard_funcs.send_parms(buffer);
    }
    
    if (letter == 17) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
}

void Callsign::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in callsign::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->set_cursor_status(Screen::on);
    popupwin_ptr->write("Callsign ? (^Q to quit)", 1, 9);
    popupwin_ptr->write("", 2, 9);
    popupwin_ptr->winrefresh();
    index = 0;
}

void Callsign::action(void) {
    run_funcs();
    int letter = getch();
    if (letter > 96 && letter < 123) letter &= 0xdf;        // convert lower to upper case
    if (letter == '-' || (letter > 46 && letter < 58)
	  || (letter > 64 && letter < 91)) {
        buffer[index] = letter;
	index++;
	popupwin_ptr->write(letter);
	popupwin_ptr->winrefresh();
    }
    else if (letter == KEY_BACKSPACE && index){
        index--;
	popupwin_ptr->write(8);
	popupwin_ptr->write(32);
	popupwin_ptr->write(8);
	popupwin_ptr->winrefresh();
    }
    if ((letter == '\n' && index > 2) || index == hisCall_SIZE) {
	buffer[index] = 0;
	strcpy(tnc_ptr->tnc_func.hisCall[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port], buffer);
	if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	    tnc_ptr->tnc_func.hisCall_lock = Tnc_func::on;
	    if (tnc_ptr->is_validcall(buffer)) tnc_ptr->make_selcall(buffer, tnc_ptr->tnc_func.selCall);
	}
	mainscreen_ptr->display_callsign();
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
    
    if (letter == 17) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
}

void Send_parms_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    keyboard_funcs.send_parms("tnc.parms");
	}
    }
}

void Send_parms_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in send_parms_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Send parameters to TNC (Y/n)?", 1, 6);
    popupwin_ptr->winrefresh();
}

void Connect::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in connect::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->set_cursor_status(Screen::on);
    popupwin_ptr->write("Connect with ? (^Q to quit)", 1, 8);
    popupwin_ptr->write("", 2, 8);
    char* temp_ptr;
    if (tnc_ptr->tnc_func.active_port && (tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
      || tnc_ptr->tnc_func.hfmode == Tnc_func::lamtor
      || tnc_ptr->tnc_func.hfmode == Tnc_func::fec)) {
        temp_ptr = tnc_ptr->tnc_func.selCall;
	max_size = selCall_SIZE;
    }
    else {
        temp_ptr = tnc_ptr->tnc_func.hisCall[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port];
	if (!tnc_ptr->tnc_func.active_port || tnc_ptr->tnc_func.hfmode == Tnc_func::packet) max_size = CONNECT_SIZE;
	else max_size = hisCall_SIZE;
    }

    index = 0;
    for (; *temp_ptr != 0; temp_ptr++) {
        buffer[index++] = *temp_ptr;
	popupwin_ptr->write(*temp_ptr);
    }
    popupwin_ptr->winrefresh();
}

void Connect::action(void) {
    run_funcs();
    int letter = getch();
    if (letter > 96 && letter < 123) letter &= 0xdf;        // convert lower to upper case
    if (letter == '-' || (letter > 46 && letter < 58)
	  || (letter > 64 && letter < 91)
	  || (letter == '!' && !index && tnc_ptr->tnc_func.active_port && 
	        tnc_ptr->tnc_func.hfmode == Tnc_func::pactor)
	  || ((letter == ',' || letter == ' ') && index && 
	        (!tnc_ptr->tnc_func.active_port || tnc_ptr->tnc_func.hfmode == Tnc_func::packet))) {
        buffer[index++] = letter;
	popupwin_ptr->write(letter);
	popupwin_ptr->winrefresh();
    }
    else if (letter == KEY_BACKSPACE && index){
        index--;
	popupwin_ptr->write(8);
	popupwin_ptr->write(32);
	popupwin_ptr->write(8);
	popupwin_ptr->winrefresh();
    }
    if ((letter == '\n' && index > 2) || index == max_size) {
        tr_buffer_ptr->reset();
	buffer[index] = 0;  // null terminate the end of the string in buffer

	if (!tnc_ptr->tnc_func.active_port
	      || tnc_ptr->tnc_func.hfmode == Tnc_func::packet) { // check to see if we need to run a connect script
	    char* filename = new char[strlen(prog_func.filedir) + strlen(buffer) + 2];
	    if (!filename) {
	        cerr << "Memory allocation error in Connect::action()" << endl;
		exit(MEM_ERROR);
	    }
	    strcpy(filename, prog_func.filedir);
	    strcat(filename, "/");
	    strcat(filename, buffer);

	    if (!access(filename, F_OK)) keyboard_funcs.run_connect_script(filename);
	    delete[] filename;
	}

	if (keyboard_funcs.connect_script_flag == Keyboard_funcs::not_running) {
	    if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    char* commandmessage;
	    if ((commandmessage = new char[max_size + 10]) == 0) {
	        cerr << "Memory allocation error in connect::action()" << endl;
		exit(MEM_ERROR);
	    }

	    if (!tnc_ptr->tnc_func.active_port || tnc_ptr->tnc_func.hfmode == Tnc_func::packet) {
	        strcpy (commandmessage, packet_connectCMD);
	    }

	    else if (tnc_ptr->tnc_func.hfmode == Tnc_func::amtor ||
		     tnc_ptr->tnc_func.hfmode == Tnc_func::lamtor || 
		     tnc_ptr->tnc_func.hfmode == Tnc_func::fec) {
	        strcpy (commandmessage, amtor_connectCMD);
		tnc_ptr->tnc_func.hfmode = Tnc_func::amtor;
		mainscreen_ptr->display_mode();
	    }

	    else if (tnc_ptr->tnc_func.hfmode == Tnc_func::gtor ||
		     tnc_ptr->tnc_func.hfmode == Tnc_func::gmon) {
	        strcpy (commandmessage, gtor_connectCMD);
		tnc_ptr->tnc_func.hfmode = Tnc_func::gtor;
		mainscreen_ptr->display_mode();
	    }

	    else if (tnc_ptr->tnc_func.hfmode == Tnc_func::pactor) {
	        strcpy (commandmessage, pactor_connectCMD);
	    }

	    strcat(commandmessage, buffer);
	    tnc_ptr->send_kamcommand(commandmessage);

	    if (!tnc_ptr->tnc_func.active_port || (tnc_ptr->tnc_func.hfmode != Tnc_func::amtor
		   && tnc_ptr->tnc_func.hfmode != Tnc_func::lamtor
	           && tnc_ptr->tnc_func.hfmode != Tnc_func::fec)) {
	        char* temp = buffer;
		if (*temp == '!') temp++; // remove leading '!' if a pactor connect on long path
		char* temp2;
		int count;
		for (count = 0, temp2 = temp; count < hisCall_SIZE
		       && *temp2 && *temp2 != ' '; temp2++, count++) {}
		*temp2 = 0;    // terminate string at first space
		strcpy(tnc_ptr->tnc_func.hisCall[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port], temp);
		if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
		    tnc_ptr->tnc_func.hisCall_lock = Tnc_func::on;
		    if (tnc_ptr->is_validcall(temp)) tnc_ptr->make_selcall(temp, tnc_ptr->tnc_func.selCall);
		}
		mainscreen_ptr->display_callsign();
	    }
	    else if (tnc_ptr->tnc_func.active_port) strcpy(tnc_ptr->tnc_func.selCall, buffer);
	    delete[] commandmessage;
	}
	receivewin_ptr->redisplay(); // go to end of current active window
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
    
    if (letter == 17) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
}


void Pactor_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(pactorCMD);

	    tnc_ptr->tnc_func.hfmode = Tnc_func::pactor;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Pactor_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in pactor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Pactor (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Amtor_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(amtorCMD);
    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::amtor;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Amtor_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in amtor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Amtor (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}


void Ascii_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(asciiCMD);

	    tnc_ptr->tnc_func.hfmode = Tnc_func::ascii;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Ascii_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in ascii_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Ascii (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Fec_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(fecCMD);
   
	    tnc_ptr->tnc_func.hfmode = Tnc_func::fec;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Fec_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in fec_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Amtor FEC (Y/n)?", 1, 4);
    popupwin_ptr->winrefresh();
}

void Gmon_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(gmonCMD);
	    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::gmon;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Gmon_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in gmon_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Gmon (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Gtor_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(gtorCMD);
	    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::gtor;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Gtor_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in gtor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Gtor (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}


void Lamtor_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(lamtorCMD);
	    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::lamtor;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Lamtor_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in lamtor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Amtor-Listen (Y/n)?", 1, 4);
    popupwin_ptr->winrefresh();
}

void Packet_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    receivewin_ptr->redisplay(); // switch to end of current hf window

	    if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Tor mode, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_specialcommand(packetCMD);
	    tnc_ptr->tnc_func.hfmode = Tnc_func::packet;
	    tnc_ptr->tnc_func.hisCall_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_callsign();
	    mainscreen_ptr->display_capture_status();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Packet_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in packet_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to Packet (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Rtty_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(rttyCMD);
	    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::rtty;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Rtty_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in rtty_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to RTTY (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Tor_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(torCMD);
	    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::tor;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
}

void Tor_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in tor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Change to TOR (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Cw_query::action(void) {
    run_funcs();
    int exit_flag = FALSE;
    if (tnc_ptr->tnc_func.hfmode == Tnc_func::cw) {
        int letter = getch();
	if (index < 2 && letter > 47 && letter < 58) {
	    buffer[index++] = letter;
	    popupwin_ptr->write(letter);
	    popupwin_ptr->winrefresh();
	}
	else if (letter == KEY_BACKSPACE && index){
	    index--;
	    popupwin_ptr->write(8);
	    popupwin_ptr->write(32);
	    popupwin_ptr->write(8);
	    popupwin_ptr->winrefresh();
	}
	else if (letter == 17) exit_flag = TRUE;
	else if (letter == '\n' && index) {
	    buffer[index] = 0;
	    int speed = (atoi(buffer) + 2)/5;
	    if (speed && speed < 11) {
	        if (speed == 10) tnc_ptr->send_specialcommand('0');
		else tnc_ptr->send_specialcommand(speed | 0x30);
		exit_flag = TRUE;
	    }
	    else {
	        beep();
		doupdate();
	    }
	}
    }
    else {
        int result = input_confirm();
	if (result != -1) exit_flag = TRUE;
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->tnc_func.active_port = 1;
	    tnc_ptr->tnc_func.active_hf_stream = 0;
	    receivewin_ptr->redisplay(); // switch current active window to hf port

	    if (tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        tnc_ptr->send_specialcommand(packetCMD);
		usleep(200000);
	    }

	    else if (tnc_ptr->tnc_func.capture_stream.port   // if file capture was set to a HF Packet, cancel
		&& tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
		mainscreen_ptr->display_capture_status();
	    }

	    tnc_ptr->send_kamcommand(cwCMD);
	    
	    tnc_ptr->tnc_func.hfmode = Tnc_func::cw;
	    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
	    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
	    tnc_ptr->tnc_func.sending_cw_flag = FALSE;
	    mainscreen_ptr->display_mode();
	    mainscreen_ptr->display_current_stream();
	    mainscreen_ptr->display_connected_status();
	    mainscreen_ptr->make_torline();
	    mainscreen_ptr->display_freebytes();
	}
    }
    if (exit_flag) {
        keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
}

void Cw_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in tor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    if (tnc_ptr->tnc_func.hfmode == Tnc_func::cw) {
        popupwin_ptr->set_cursor_status(Screen::on);
        popupwin_ptr->write("New speed (5 - 50 WPM)? (^Q to quit)", 1, 2);
	popupwin_ptr->write("", 2, 2);
    }
    else popupwin_ptr->write("Change to CW (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
    index = 0;
}

void Disconnect_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();

	    if (!tnc_ptr->tnc_func.active_port || tnc_ptr->tnc_func.hfmode == Tnc_func::packet) {
	        tnc_ptr->send_kamcommand(packet_disconnectCMD);
	    }
	   
	    else tnc_ptr->send_specialcommand(tor_disconnectCMD);
	}
    }
}

void Disconnect_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in Disconnect_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Disconnect (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Abort_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    tr_buffer_ptr->reset();
	    tnc_ptr->send_specialcommand(aborttxCMD);
	}
    }
}

void Abort_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in Disconnect_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Abort (Y/n)?", 1, 12);
    popupwin_ptr->winrefresh();
}

void Send_message_query::setup(char msg_num) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in Send_message_file_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Send message number ", 1, 6);
    popupwin_ptr->write(msg_num);
    popupwin_ptr->write(" (Y/n)?");
    popupwin_ptr->winrefresh();
    message_num = msg_num;
}

void Send_message_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) keyboard_funcs.send_message(message_num);
    }
}

void Autocq_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in tor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    if (!prog_func.sending_autocq) {
        popupwin_ptr->write("Send auto-cq (Y/n)?", 1, 8);
    }
    else popupwin_ptr->write("End auto-cq (Y/n)?", 1, 8);
    popupwin_ptr->winrefresh();
}

void Autocq_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    if (!prog_func.sending_autocq) {
	        CqsendBuffer* cqsend_ptr = new CqsendBuffer(tnc_ptr, mainscreen_ptr, sendwin_ptr,
							    receivewin_ptr, screenform_ptr);
		if (!cqsend_ptr) {
		    cerr << "Memory allocation error in Keyboard_controller::send_message()" << endl;
		    exit(MEM_ERROR);
		}
		if (cqsend_ptr->load_buffer()) buffer_list_ptr->add(cqsend_ptr);
		else delete cqsend_ptr;
	    }
	    else {       // we need to find the file buffer sending the cq message
	        buffer_list_ptr->reset(DList_enum::bottom_end);
		FileBuffer* file_buffer_ptr;
		while ((file_buffer_ptr = (FileBuffer*)buffer_list_ptr->inspect(DList_enum::up)) != 0
		       && !(file_buffer_ptr->get_stream() == 0
			    && file_buffer_ptr->get_port() == 1)) {}
		if (file_buffer_ptr) file_buffer_ptr->end_loading();
		else {
		    receivewin_ptr->write("\nOops - can't find the cq file buffer to delete\n");
		    receivewin_ptr->winrefresh();
		    beep();
		    doupdate();
		    prog_func.sending_autocq = FALSE;
		    mainscreen_ptr->display_connected_status();
		}
	    }
	}
    }
}

void Enter_rst::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in Enter_rst::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->set_cursor_status(Screen::on);
    popupwin_ptr->write("RST? (^Q to quit):  ", 1, 4);
    for (index = 0; prog_func.rst[index] && index < 3; index++) {
        popupwin_ptr->write(prog_func.rst[index]);
    }
    popupwin_ptr->winrefresh();
}

void Enter_rst::action(void) {
    run_funcs();
    int letter = getch();
    if (letter > 47 && letter < 58 && index < 3) {
        prog_func.rst[index] = letter;
	index++;
	popupwin_ptr->write(letter);
	popupwin_ptr->winrefresh();
    }
    else if (letter == KEY_BACKSPACE && index) {
        index--;
	popupwin_ptr->write(8);
	popupwin_ptr->write(32);
	popupwin_ptr->write(8);
	popupwin_ptr->winrefresh();
    }
    if ((letter == '\n' && index == 3) || letter == 17) {
        keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
    }
}

void Download_query::setup(Mode mode_) {
    mode = mode_;
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in Downmload_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    if (tnc_ptr->tnc_func.download_list_ptr->get_download_status(tnc_ptr->tnc_func.active_stream(),
	                              tnc_ptr->tnc_func.active_port) == DownloadList::on) {
        popupwin_ptr->write("Stop downloading the file? (Y/n)", 1, 4);
    }
    else {
        popupwin_ptr->set_cursor_status(Screen::on);
	popupwin_ptr->write("Name to save file as? (^Q to quit)", 1, 2);
	popupwin_ptr->write("", 2, 2);
	index = 0;
	if (strlen(prog_func.filedir) < 254) {
	    for (char* temp_ptr = prog_func.filedir; *temp_ptr != 0 && index < 255; temp_ptr++, index++) {
	      filename[index] = *temp_ptr;
	      popupwin_ptr->write(filename[index]);
	    }
	    filename[index] = '/';
	    popupwin_ptr->write('/');
	    index++;
	}
    }
    popupwin_ptr->winrefresh();
}

void Download_query::action(void) {
    run_funcs();
    DownloadList* download_list_ptr = tnc_ptr->tnc_func.download_list_ptr;
    if (download_list_ptr->get_download_status(tnc_ptr->tnc_func.active_stream(),
	                              tnc_ptr->tnc_func.active_port) == DownloadList::on) {
        int result = input_confirm();
	if (result != -1) {
	    keyboard_ptr = default_input_ptr;
	    popupwin_ptr->unshow();
	    delete popupwin_ptr;
	    refresh_winlist(*screenform_ptr);
	    if (result == 1) { // we need to find the download file object receiving the
	                       // file on the active stream and port
	        download_list_ptr->reset(DList_enum::bottom_end);
		DownloadFile* dload_file_ptr;
		while ((dload_file_ptr = (DownloadFile*)download_list_ptr->inspect(DList_enum::up)) != 0
		       && !(dload_file_ptr->get_stream() == tnc_ptr->tnc_func.active_stream()
			    && dload_file_ptr->get_port() == tnc_ptr->tnc_func.active_port)) {}
		if (dload_file_ptr) {
		    download_list_ptr->extract();
		    delete dload_file_ptr;
		    mainscreen_ptr->display_connected_status();
		}
		else {
		    receivewin_ptr->write("\nOops - can't find the download file object to delete\n");
		    receivewin_ptr->winrefresh();
		    beep();
		    doupdate();
		}
	    }
	}
    }

    else {
        int letter = getch();
	if (letter > 44 && letter < 123 && index < 256) {
	    filename[index++] = letter;
	    popupwin_ptr->write(letter);
	    popupwin_ptr->winrefresh();
	}
	else if (letter == KEY_BACKSPACE && index) {
	    index--;
	    popupwin_ptr->write(8);
	    popupwin_ptr->write(32);
	    popupwin_ptr->write(8);
	    popupwin_ptr->winrefresh();
	}
	if (letter == '\n' && index > 0) {
	    filename[index] = 0;
	    keyboard_ptr = default_input_ptr;
	    popupwin_ptr->unshow();
	    delete popupwin_ptr;
	    refresh_winlist(*screenform_ptr);
	    DownloadFile::Download_mode b_mode;
	    if (mode == Download_query::binary) b_mode = DownloadFile::binary;
	    else b_mode = DownloadFile::s_plus;
	    DownloadFile* download_file_ptr = new DownloadFile(b_mode, tnc_ptr->tnc_func.active_stream(),
							       tnc_ptr->tnc_func.active_port);
	    if (!download_file_ptr) {
	        cerr << "Memory allocation error in Download_query::action()" << endl;
		exit(MEM_ERROR);
	    }
	    if (download_file_ptr->open_file(filename)) {
	        tnc_ptr->tnc_func.download_list_ptr->add(download_file_ptr);
		mainscreen_ptr->display_connected_status();
	    }
	    else {
	        delete download_file_ptr;
	        popupwin_ptr = new PopupWin;
		if (!popupwin_ptr) {
		    cerr << "Memory allocation error in Download_query::action()" << endl;
		    exit(MEM_ERROR);
		}
		popupwin_ptr->show();
		popupwin_ptr->write(filename, 1, 1);
		popupwin_ptr->write("already exists or cannot be written to", 3, 1);
		popupwin_ptr->winrefresh();
		sleep(3);
		popupwin_ptr->unshow();
		delete popupwin_ptr;
		refresh_winlist(*screenform_ptr);
	    }
	}
	if (letter == 17) {
	    keyboard_ptr = default_input_ptr;
	    popupwin_ptr->unshow();
	    delete popupwin_ptr;
	    refresh_winlist(*screenform_ptr);
	}
    }
}

void Printmark_query::setup(void) {
    if (tnc_ptr->tnc_func.print_list_ptr->get_print_status(tnc_ptr->tnc_func.active_stream(),
	  tnc_ptr->tnc_func.active_port) ==  PrintList::on) {
        mode = print;
    }
    else mode = set_mark;

    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in tor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    if (mode == set_mark) {
        popupwin_ptr->write("Set print mark (Y/n)?", 1, 8);
    }
    else {
        popupwin_ptr->write("Print from mark (Y/n/c)?", 1, 7);
	popupwin_ptr->write("(Press C to cancel print job)", 2, 5);
    }
    popupwin_ptr->winrefresh();
}

void Printmark_query::action(void) {
    run_funcs();
    int result = -1;
    int letter = getch();
    if (letter == 'y' || letter == 'Y' || letter == '\n') result = 1;
    if (letter == 'n' || letter == 'N') result = 0;
    if ((letter == 'c' || letter == 'C') && mode == print) result = 2;
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result > 0) {
	    PrintList* print_list_p = tnc_ptr->tnc_func.print_list_ptr;
	    if (mode == set_mark) {
		PrintFile* print_file_p = new PrintFile(tnc_ptr->tnc_func.active_stream(),
							tnc_ptr->tnc_func.active_port);
		if (!print_file_p) {
		    cerr << "Memory allocation error in Printmark_query::action()" << endl;
		    exit(MEM_ERROR);
		}
		if (print_file_p->start_store()) {
		    print_list_p->add(print_file_p);
		    receivewin_ptr->ReceiveWin::write((char)PRINT_MARK);
		    receivewin_ptr->store_letter((char)PRINT_MARK);
		    if (!receivewin_ptr->is_scrolling()) {
		        receivewin_ptr->winrefresh();
		    }
		}
		else {
		    delete print_file_p;
		    PopupWin* popupwin_ptr = new PopupWin(5, 60, 10, Screen::get_ncols()/2 - 30);
		    if (!popupwin_ptr) {
		        cerr << "Memory allocation error in Printmark_query::action()()" << endl;
			exit(MEM_ERROR);
		    }
		    popupwin_ptr->show();
		    popupwin_ptr->write("Temporary print file cannot be opened for writing", 1, 5);
		    popupwin_ptr->write("Please check permissions in /tmp directory", 2, 5);
		    popupwin_ptr->winrefresh();
		    sleep(3);
		    popupwin_ptr->unshow();
		    delete popupwin_ptr;
		    refresh_winlist(*screenform_ptr);
		}
	    }
	    else {  // printing from mark
	        print_list_p->reset(DList_enum::bottom_end);
		PrintFile* print_file_p;
		while ((print_file_p = (PrintFile*)print_list_p->inspect(DList_enum::up)) != 0
		       && !(print_file_p->get_stream() == tnc_ptr->tnc_func.active_stream()
			    && print_file_p->get_port() == tnc_ptr->tnc_func.active_port)) {}
		if (print_file_p) {
		    print_list_p->extract();
		    if (result == 1) print_file_p->print();
		    else print_file_p->cancel();
		    delete print_file_p;
		}
		else {
		    receivewin_ptr->ReceiveWin::write("\nOops - can't find the print file object to delete\n"
					              "Please report bug\n");
		    beep();
		    doupdate();
		}
	    }
	}
    }
}

void Printbuffer_query::setup(void) {
    keyboard_ptr = this;
    popupwin_ptr = new PopupWin;
    if (!popupwin_ptr) {
        cerr << "Memory allocation error in tor_query::setup()" << endl;
	exit(MEM_ERROR);
    }
    popupwin_ptr->show();
    popupwin_ptr->write("Print the scroll buffer (Y/n)?", 1, 5);
    popupwin_ptr->winrefresh();
}

void Printbuffer_query::action(void) {
    run_funcs();
    int result = input_confirm();
    if (result != -1) {
	keyboard_ptr = default_input_ptr;
	popupwin_ptr->unshow();
	delete popupwin_ptr;
	refresh_winlist(*screenform_ptr);
	if (result == 1) {
	    char* text = receivewin_ptr->text();
	    FILE* pipe_fp = popen(prog_func.print_cmd, "w");
	    fwrite(text, sizeof(char), strlen(text), pipe_fp);
	    fclose(pipe_fp);
	    delete[] text;
	}
    }
}

Keyboard_controller::Keyboard_controller(MainScreen& mainscreen, SendWin& sendwin,
		ReceiveWin& receivewin, DList<Screen>& screenform, Transmit_buffer& tr_buffer,
		Tnc* tnc_ptr, BufferList& buffer_list) {
    set_data(&mainscreen, &sendwin, &receivewin, &screenform, &tr_buffer,
	       tnc_ptr, &buffer_list, this);
    keyboard_ptr = this;
}

void Keyboard_controller::action(void) {
    run_funcs();
    int result;
    int termchar;
    if ((termchar = getch()) != ERR) {  // non-blocking getch() returns ERR if no letter available to extract
        if (keyboard_funcs.connect_script_flag == Keyboard_funcs::not_running) {
	    if (buffer_list_ptr->get_upload_status(tnc_ptr->tnc_func.active_stream(),
				   tnc_ptr->tnc_func.active_port) == BufferList::keyboard) {
		if (tnc_ptr->tnc_func.active_port && termchar > 31 && termchar < 257) {
		    if (tnc_ptr->tnc_func.hfmode == Tnc_func::rtty
			|| tnc_ptr->tnc_func.hfmode == Tnc_func::fec
			|| tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
			|| (tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
			    && tnc_ptr->tnc_func.stream_status[0][1] == Tnc_func::disconnected)
			|| (tnc_ptr->tnc_func.hfmode == Tnc_func::tor
			    && (tnc_ptr->tnc_func.stream_status[0][1] == Tnc_func::disconnected
				|| tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor))) {
		      if (!is_baudot((char)termchar)) termchar = 0;
		      else if (tnc_ptr->tnc_func.hfmode == Tnc_func::rtty) termchar = toupper(termchar);
		    }
		
		    else if (tnc_ptr->tnc_func.hfmode == Tnc_func::cw) {
		        if (!is_morse((char)termchar)) termchar = 0;
			else termchar = toupper(termchar);
		    }
		}

#if CHAR_SET==LATIN_1
		if ((termchar < 257 && termchar > 31) || termchar == '\n' || termchar == KEY_BACKSPACE) {
#elif CHAR_SET==CP437
		if ((termchar < 256 && termchar > 31) || termchar == '\n' || termchar == KEY_BACKSPACE) {
#else
		if ((termchar < 127 && termchar > 31) || termchar == '\n' || termchar == KEY_BACKSPACE) {
#endif
		    if (termchar == KEY_BACKSPACE) termchar = 8;
		    keyboard_funcs.wordwrap_buffer_add(termchar);
		}
	    }

	    if ((termchar == KEY_F(4) || termchar == F_KEYmatrix[0][ADDED_FKEY(4)]
		          || termchar == F_KEYmatrix[1][ADDED_FKEY(4)]
		          || termchar == F_KEYmatrix[2][ADDED_FKEY(4)])
		      && (!tnc_ptr->tnc_func.active_port
			  || (tnc_ptr->tnc_func.hfmode != Tnc_func::rtty &&  
			      tnc_ptr->tnc_func.hfmode != Tnc_func::ascii &&
			      tnc_ptr->tnc_func.hfmode != Tnc_func::tor && 
			      tnc_ptr->tnc_func.hfmode != Tnc_func::cw))) {
	        connect_obj.setup();
	    }
	
	    else if (termchar == KEY_F(6)      // this gets a file name to send and opens/closes it
		     && !(prog_func.sending_autocq 
			    && !tnc_ptr->tnc_func.active_stream()
			    && tnc_ptr->tnc_func.active_port)) {
	        openfile_obj.setup(Openfile::text);
	    }
	
	    else if (termchar == KEY_F(7)) {   // this displays enters a call in the callsign buffer
	        callsign_obj.setup();
	    }

	    else if (termchar == KEY_F(9)) {    // this sends the tnc the parameters
	        send_parms_query_obj.setup();   // in tnc.parms
	    }

	    else if (termchar == KEY_NPAGE && tnc_ptr->tnc_func.active_port 
		       && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) { // Page Down
	        if (tr_buffer_ptr->letters_free() > 5) {
		    if (tnc_ptr->tnc_func.stream_status[0][1] == Tnc_func::connected) {
		        if (tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor) {
			    keyboard_funcs.wordwrap_buffer_add('=');
			    keyboard_funcs.wordwrap_buffer_add('>');
			}
			else if (tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
				 || (tnc_ptr->tnc_func.hfmode == Tnc_func::tor
				     && tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor)) {
			    sendwin_ptr->write("+?");
			    sendwin_ptr->winrefresh();
			}
		    }
		    keyboard_funcs.flush_wordwrap_buffer();
		    if (tnc_ptr->tnc_func.hfmode == Tnc_func::cw) {
		        tnc_ptr->send_specialcommand(immediate_rxCMD);
			tnc_ptr->tnc_func.sending_cw_flag = FALSE;
			usleep(100000);
		    }
		    else {
		        tr_buffer_ptr->add_letter(CMDinbuffer);
			tr_buffer_ptr->add_letter(rxCMD);
			tr_buffer_ptr->add_letter(CMDinbuffer);
		    }
		}
		else {
		    beep();
		    doupdate();
		}
	    }
	
	    else if (termchar == 1) {                                              // Ctrl-A
		if (tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		     [tnc_ptr->tnc_func.active_port] == Tnc_func::connected
			&& (!tnc_ptr->tnc_func.active_port
			     || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
			     || tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor)) {
		    keyboard_funcs.flush_wordwrap_buffer();
		    tr_buffer_ptr->add_letter(1);
		    beep();
		    doupdate();
		}
	        else if (tnc_ptr->tnc_func.hfmode != Tnc_func::amtor
			   && tnc_ptr->tnc_func.active_port
			   && !prog_func.sending_autocq) {
		    amtor_query_obj.setup();
		}
	    }

	    else if (termchar == 3                                                  // Ctrl-C
		       && tnc_ptr->tnc_func.active_port
		       && !prog_func.sending_autocq) {
	        cw_query_obj.setup();
	    }

	    else if (termchar == 6 && tnc_ptr->tnc_func.hfmode != Tnc_func::fec    // Ctrl-F
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        fec_query_obj.setup();
	    }

	    else if (termchar == 7 && tnc_ptr->tnc_func.hfmode != Tnc_func::gtor // Ctrl-G
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        gtor_query_obj.setup();
	    }

	    else if (termchar == 12 && tnc_ptr->tnc_func.hfmode != Tnc_func::lamtor // Ctrl-L
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        lamtor_query_obj.setup();
	    }

	    else if (termchar == 15 && tnc_ptr->tnc_func.hfmode != Tnc_func::gmon   // Ctrl-O
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        gmon_query_obj.setup();
	    }

	    else if (termchar == 16 && tnc_ptr->tnc_func.hfmode != Tnc_func::pactor // Ctrl-P
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        pactor_query_obj.setup();
	    }

	    else if (termchar == 18 && tnc_ptr->tnc_func.hfmode != Tnc_func::rtty // Ctrl-R
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        rtty_query_obj.setup();
	    }

	    else if (termchar == 19 && tnc_ptr->tnc_func.hfmode != Tnc_func::ascii // Ctrl-S
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        ascii_query_obj.setup();
	    }

	    else if (termchar == 20) {                                             // Ctrl-T
		if (tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		     [tnc_ptr->tnc_func.active_port]  == Tnc_func::connected
		        && (!tnc_ptr->tnc_func.active_port
			     || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
			     || tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor)) {
		    keyboard_funcs.flush_wordwrap_buffer();
		    tr_buffer_ptr->add_letter(20);
		}

		else if (tnc_ptr->tnc_func.hfmode != Tnc_func::tor
		           && tnc_ptr->tnc_func.active_port
			   && !prog_func.sending_autocq) {
		    tor_query_obj.setup();
		}
	    }

	    else if (termchar == 24 && tnc_ptr->tnc_func.hfmode != Tnc_func::packet // Ctrl-X
		     && tnc_ptr->tnc_func.active_port
		     && !prog_func.sending_autocq) {
	        packet_query_obj.setup();
	    }

	    else if (termchar == 26                                                // Ctrl-Z
		       && tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		           [tnc_ptr->tnc_func.active_port]  == Tnc_func::connected
		       && (!tnc_ptr->tnc_func.active_port
			     || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
			     || tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor)) {
	        keyboard_funcs.flush_wordwrap_buffer();
  	        tr_buffer_ptr->add_letter(26);
	    }
	    else if (keyboard_funcs.send_parms_flag == Keyboard_funcs::not_running) {

	        if (termchar == KEY_F(5)) {   // send a command to the Kam
		    if (tnc_ptr->Tnc::get_active_freebytes() > 3) {
		        input_command_obj.setup();

		    }
		    else {
		        beep();
			doupdate();
		    }
		}
	
		else if (termchar == KEY_F(10)    // toggles active port between hf and vhf
			 || termchar == F_KEYmatrix[0][ADDED_FKEY(10)]) {
		    if(tnc_ptr->get_active_freebytes() > tr_buffer_ptr->letters_used() + 3) {
		        if (!tnc_ptr->tnc_func.active_port) tnc_ptr->tnc_func.active_port = 1;
			else tnc_ptr->tnc_func.active_port = 0;
			receivewin_ptr->redisplay(); // switch current active window in display
			mainscreen_ptr->display_current_stream();
			mainscreen_ptr->display_callsign();
			mainscreen_ptr->display_connected_status();
			mainscreen_ptr->make_torline();
			mainscreen_ptr->display_freebytes();
			mainscreen_ptr->update_lockinfo();
			mainscreen_ptr->update_torinfo();
			mainscreen_ptr->update_txinfo();
		                 // notify the Kam LEDs of the change
			tnc_ptr->send_kamcommand("");
		    }
		    else {
		        beep();
			doupdate();
		    }
		}

		else if (termchar == KEY_F(11)   // goes down one packet stream on active channel
			 || termchar == F_KEYmatrix[0][ADDED_FKEY(11)]) {
		    if(tnc_ptr->get_active_freebytes() > tr_buffer_ptr->letters_used() + 3) {
		        if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode == Tnc_func::packet) {
			    if (!tnc_ptr->tnc_func.active_hf_stream) tnc_ptr->tnc_func.active_hf_stream = MAXUSERS - 1;
			    else tnc_ptr->tnc_func.active_hf_stream--;
			}
			else if (!tnc_ptr->tnc_func.active_port) {
			    if (!tnc_ptr->tnc_func.active_vhf_stream) tnc_ptr->tnc_func.active_vhf_stream = MAXUSERS - 1;
			    else tnc_ptr->tnc_func.active_vhf_stream--;
			}
			else return;

			receivewin_ptr->redisplay(); // switch current active window in display
			mainscreen_ptr->display_current_stream();
			mainscreen_ptr->display_callsign();
			mainscreen_ptr->display_connected_status();

                               // notify the Kam LEDs of the change
			tnc_ptr->send_kamcommand("");
		    }
		    else {
		        beep();
			doupdate();
		    }
		}

	    
		else if (termchar == KEY_F(12)   // goes up one packet stream on active channel
			 || termchar == F_KEYmatrix[0][ADDED_FKEY(12)]) {
		    if (tnc_ptr->get_active_freebytes() > tr_buffer_ptr->letters_used() + 3) {
		        if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode == Tnc_func::packet) {
			    tnc_ptr->tnc_func.active_hf_stream++;
			    if (tnc_ptr->tnc_func.active_hf_stream == MAXUSERS) tnc_ptr->tnc_func.active_hf_stream = 0;
			}
			else if (!tnc_ptr->tnc_func.active_port) {
			    tnc_ptr->tnc_func.active_vhf_stream++;
			    if (tnc_ptr->tnc_func.active_vhf_stream == MAXUSERS) tnc_ptr->tnc_func.active_vhf_stream = 0;
			}
			else return;
		
			receivewin_ptr->redisplay(); // switch current active window in display
			mainscreen_ptr->display_current_stream();
			mainscreen_ptr->display_callsign();
			mainscreen_ptr->display_connected_status();

                               // notify the Kam LEDs of the change
			tnc_ptr->send_kamcommand("");
		    }	    
		    else {
		        beep();
			doupdate();
		    }
		}
	    }
	}

	if (termchar == 4 && (!tnc_ptr->tnc_func.active_port              // Ctrl-D
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::tor)) {
	    disconnect_query_obj.setup();
	}

	else if (termchar == 17) {      // Ctrl-Q
	        exit_query_obj.setup();
	    }

	else if (termchar == KEY_UP) {
	    receivewin_ptr->scrollup();
	}

	else if (termchar == KEY_DOWN) {
	    receivewin_ptr->scrolldown();
	}

	else if (termchar == KEY_RIGHT) {
	    receivewin_ptr->scrollout();
	}

	else if (termchar == KEY_PPAGE && tnc_ptr->tnc_func.active_port 
		   && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) { // Page Up
	    tnc_ptr->send_specialcommand(txCMD);
	    if (tnc_ptr->tnc_func.hfmode == Tnc_func::cw) tnc_ptr->tnc_func.sending_cw_flag = TRUE;
	    usleep(100000);
	}
	    
	else if (termchar == KEY_F(1)  || termchar == F_KEYmatrix[0][ADDED_FKEY(1)]
		   || termchar == F_KEYmatrix[1][ADDED_FKEY(1)]
		   || termchar == F_KEYmatrix[2][ADDED_FKEY(1)]) {   // this displays the help file
	    help_obj.setup();
	}
	
	else if (termchar == KEY_F(2) || termchar == F_KEYmatrix[0][ADDED_FKEY(2)]
		   || termchar == F_KEYmatrix[1][ADDED_FKEY(2)]
		   || termchar == F_KEYmatrix[2][ADDED_FKEY(2)]) {   // this toggles the file capture status
	    if (tnc_ptr->tnc_func.capturefile_flag) {
	        tnc_ptr->tnc_func.capturefile_flag = FALSE;
		tnc_ptr->tnc_func.capturefile.close();
	    }

	    else {
	        tnc_ptr->tnc_func.capturefile_flag = TRUE;
		tnc_ptr->tnc_func.capture_stream.port = tnc_ptr->tnc_func.active_port;
		tnc_ptr->tnc_func.capture_stream.stream = tnc_ptr->tnc_func.active_stream();
		char* filename = new char[strlen(prog_func.filedir) + 13];
		if (!filename) {
		    cerr << "Memory allocation error in Keyboard_controller.action()" << endl;
		    exit(MEM_ERROR);
		}
		strcpy(filename, prog_func.filedir);
		strcat(filename, "/capture.txt");
		    
		tnc_ptr->tnc_func.capturefile.open(filename, ios::out | ios::app);
		if (!tnc_ptr->tnc_func.capturefile) {
		    popupwin_ptr = new PopupWin;
		    if (!popupwin_ptr) {
		        cerr << "Memory allocation error in Keyboard_controller.action()" << endl;
			exit(MEM_ERROR);
		    }
		    popupwin_ptr->show();
		    popupwin_ptr->write("File capture.txt could not be opened", 2, 1);
		    popupwin_ptr->winrefresh();
		    sleep(3);
		    popupwin_ptr->unshow();
		    delete popupwin_ptr;
		    refresh_winlist(*screenform_ptr);
		    tnc_ptr->tnc_func.capturefile_flag = FALSE;
		}
		delete[] filename;
	    }
	    mainscreen_ptr->display_capture_status();
	}

	else if (termchar == KEY_F(3) || termchar == F_KEYmatrix[0][ADDED_FKEY(3)]
		   || termchar == F_KEYmatrix[1][ADDED_FKEY(3)]
		   || termchar == F_KEYmatrix[2][ADDED_FKEY(3)]) {   // this toggles the line/guard/word sending status
	    if (prog_func.send_mode == Prog_func::line) prog_func.send_mode = Prog_func::word;
	    else if (prog_func.send_mode == Prog_func::word) prog_func.send_mode = Prog_func::guard;
	    else prog_func.send_mode = Prog_func::line;
	    mainscreen_ptr->display_send_mode_status();
	}
	
	else if (termchar == KEY_F(8) && tnc_ptr->tnc_func.active_port // this toggles the call lock status
		   && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	    if (tnc_ptr->tnc_func.hisCall_lock == Tnc_func::off) tnc_ptr->tnc_func.hisCall_lock = Tnc_func::on;
	    else tnc_ptr->tnc_func.hisCall_lock = Tnc_func::off;
	    mainscreen_ptr->display_callsign();
	}

	else if (termchar == ESC_CHAR) {   // Alt key?
	    termchar = getch();

	    if (termchar == 4                    // Ctrl-Alt-D
		   && tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		                   [tnc_ptr->tnc_func.active_port] == Tnc_func::connected
		   && (!tnc_ptr->tnc_func.active_port
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || (tnc_ptr->tnc_func.hfmode == Tnc_func::tor
			   && tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor))) {
	        download_query_obj.setup(Download_query::binary);
	    }

	    else if (termchar == 21              // Ctrl-Alt-U
		   && tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		                   [tnc_ptr->tnc_func.active_port] == Tnc_func::connected
		   && (!tnc_ptr->tnc_func.active_port
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || (tnc_ptr->tnc_func.hfmode == Tnc_func::tor
			   && tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor))) {
	        openfile_obj.setup(Openfile::binary);
	    }

	    else if ((termchar == 'B' || termchar == 'b')
		     && receivewin_ptr->is_text()) {
	        printbuffer_query_obj.setup();
	    }

	    else if ((termchar == 'C' || termchar == 'c')
		   && tnc_ptr->tnc_func.active_port
		   && (tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::tor)
		   && (prog_func.sending_autocq
			|| buffer_list_ptr->get_upload_status(0, 1) == BufferList::keyboard)) {
	        prog_func.tor_autocq_mode = Prog_func::amtor;
	        autocq_query_obj.setup();
	    }

	    else if ((termchar == 'D' || termchar == 'd')
		   && tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		                   [tnc_ptr->tnc_func.active_port] == Tnc_func::connected
		   && (!tnc_ptr->tnc_func.active_port
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || (tnc_ptr->tnc_func.hfmode == Tnc_func::tor
			   && tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor))) {
	        download_query_obj.setup(Download_query::s_plus);
	    }

	    else if ((termchar == 'I' || termchar == 'i')
		        && tnc_ptr->tnc_func.active_port
		        && tnc_ptr->tnc_func.hfmode != Tnc_func::packet
		        && keyboard_funcs.connect_script_flag == Keyboard_funcs::not_running) {
	        char* message = new char[strlen(prog_func.myCall) + hisCall_SIZE + 5]; // must be large enough for
		if (!message) {
		    cerr << "Memory allocation error in Keyboard_controller::action()" << endl;
		    exit(MEM_ERROR);
		}
	        strcpy(message, tnc_ptr->tnc_func.hisCall[0][1]);     // myCall and hisCall and " de "
		if (tnc_ptr->tnc_func.hfmode == Tnc_func::rtty
		     || tnc_ptr->tnc_func.hfmode == Tnc_func::cw) {
		    strcat(message, " DE ");
		}
		else strcat(message, " de ");
		strcat(message, prog_func.myCall);
		strcat(message, " ");

		int index;
		for (index = result = 0; result != -1 && message[index]; index++) {
		    result = keyboard_funcs.wordwrap_buffer_add(message[index]);
		}
		sendwin_ptr->winrefresh();
		delete[] message;
	    }

	    else if ((termchar == 'L' || termchar == 'l') 
		  && tnc_ptr->tnc_func.active_port
		  && (tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		    || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		    || tnc_ptr->tnc_func.hfmode == Tnc_func::tor)) {
	        if (tnc_ptr->tnc_func.speed_lock == Tnc_func::off) {
		    tnc_ptr->tnc_func.speed_lock = Tnc_func::on;
		    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::yes;
		    tnc_ptr->send_specialcommand('1');
		}
		else {
		    tnc_ptr->tnc_func.speed_lock = Tnc_func::off;
		    tnc_ptr->tnc_func.have_sent_lock = Tnc_func::no;
		    tnc_ptr->send_specialcommand('0');
		}
		mainscreen_ptr->update_lockinfo();
	    }

	    else if (termchar == 'P' || termchar == 'p') {
	        printmark_query_obj.setup();
	    }

	    else if ((termchar == 'R' || termchar == 'r')
		     && tnc_ptr->tnc_func.active_port
		     && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
	        enter_rst_obj.setup();
	    }

	    else if ((termchar == 'S' || termchar == 's')
		        && tnc_ptr->tnc_func.active_port) {
	        if (tnc_ptr->tnc_func.hfmode == Tnc_func::cw) {
		    tnc_ptr->send_specialcommand('L');
		    usleep(200000);
		    tnc_ptr->send_specialcommand('U');
		}
		else if (tnc_ptr->tnc_func.hfmode == Tnc_func::lamtor
			  || tnc_ptr->tnc_func.hfmode == Tnc_func::fec
			  || (tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
			       && tnc_ptr->tnc_func.stream_status[0][1] == Tnc_func::disconnected)) {
		    tnc_ptr->send_specialcommand(immediate_rxCMD);
		    usleep(100000);
		}
	    }

	    else if ((termchar == 'T' || termchar == 't')
		   && tnc_ptr->tnc_func.active_port
		   && tnc_ptr->tnc_func.hfmode == Tnc_func::tor
		   && (prog_func.sending_autocq
			|| buffer_list_ptr->get_upload_status(0, 1) == BufferList::keyboard)) {
	        prog_func.tor_autocq_mode = Prog_func::pactor;
	        autocq_query_obj.setup();
	    }

	    else if ((termchar == 'U' || termchar == 'u')
		   && tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()]
		                   [tnc_ptr->tnc_func.active_port] == Tnc_func::connected
		   && (!tnc_ptr->tnc_func.active_port
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::packet
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || (tnc_ptr->tnc_func.hfmode == Tnc_func::tor
			   && tnc_ptr->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor))) {
	        openfile_obj.setup(Openfile::s_plus);
	    }

	    else if ((termchar == 'X' || termchar == 'x')
		   && tnc_ptr->tnc_func.active_port
		   && (tnc_ptr->tnc_func.hfmode == Tnc_func::pactor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::amtor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::gtor
		       || tnc_ptr->tnc_func.hfmode == Tnc_func::tor)) {
	        abort_query_obj.setup();
	    }

	    else if (termchar >= '0' && termchar <= '9') {
	        if (keyboard_funcs.connect_script_flag == Keyboard_funcs::not_running) {
		    if (prog_func.noprompt_flag == FALSE) send_message_query_obj.setup((char)termchar);
		    else keyboard_funcs.send_message((char)termchar);
		}
		else {
		    beep();
		    doupdate();
		}
	    }
	}
    }
}

Keyboard_setup_mode::Keyboard_setup_mode(MainScreen& mainscreen, SendWin& sendwin, ReceiveWin& receivewin,
                                           DList<Screen>& screenform, Transmit_buffer& tr_buffer, Tnc* tnc_ptr) {
    set_data(&mainscreen, &sendwin, &receivewin, &screenform, &tr_buffer, tnc_ptr, 0, this);
    keyboard_ptr = this;
}

void Keyboard_setup_mode::action(void) {
    int termchar;
    int result;

    if ((termchar = getch()) != ERR) {  // non-blocking getch() returns ERR if no letter available to extract

        if ((termchar < 127 && termchar > 31) || termchar == '\n' || termchar == 3) {
	    result = tr_buffer_ptr->add_letter((char)termchar);
	    if (result == -1) beep();
	    else if (termchar != 3) {
	        sendwin_ptr->write((char)termchar);
		sendwin_ptr->winrefresh();
	    }
	}
	
	else if (termchar == KEY_BACKSPACE) {
	    result = tr_buffer_ptr->erase_letter();
	    if (!result) {
	        beep();
		doupdate();
	    }
	    else {
	        if (result == -1) {  // if last letter already extracted, and we are in pactor or gtor
		                     // we need to send value 8 to the Kam
		    tr_buffer_ptr->add_letter(8);
		}
		sendwin_ptr->write(8);        // now erase letter from screen
		sendwin_ptr->write(32);
		sendwin_ptr->write(8);
		sendwin_ptr->winrefresh();
	    }
	}
	else if (termchar == 17) {      // Ctrl-Q
	    exit_query_obj.setup();
	}
    }
}

void Gpl_query::action(void) {
    int letter = getch();
    if (letter == 'Y' || letter == 'y') {
        prog_func.GPL_flag = TRUE;
// GPL terms accepted - put a marker in the home directory
	char* gpl_file = new char[strlen(prog_func.homedir) + 9];
	if (!gpl_file) {
	    cerr << "Memory allocation error in Gpl_query::action()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(gpl_file, prog_func.homedir);
	strcat(gpl_file, "/");
	strcat(gpl_file, ".kamgpl");
	fileout.open(gpl_file, ios::out);
    }
    if (letter == 'N' || letter == 'n') {
        prog_func.exitflag = TRUE;
	prog_func.GPL_flag = TRUE;
    }
    else if (letter ==  KEY_UP) copyright_screen.scrollup();
    else if (letter == KEY_DOWN) copyright_screen.scrolldown();
}

