
/* mktext.c - generate text version from radio data file.

   Copyright (c) 1997 Riku Kalinen, OH2LWO, oh2lwo@sral.fi

   RCS $Id: ft50mktext.c,v 1.2 1997/12/11 15:01:52 riku Exp $ */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "yaesu.h"

/* Print lots of hexdump.  Controlled by -d - option. */

int debug = 0;

/* Dump contents of one DTMF memory. */

static void dtmfmemdump (unsigned char *b)
{
  printf ("%1x%02x\n", *b, b[1]);
}

/* Dump flags for 16 channels. */

static void flagdump (unsigned char *b, int inx)
{
  int i;
  
  printf ("# %d-:\t", inx);
  for (i = 0; i < 16; i ++) {
    printf ("%s", flag_string (b[i]));
    if (i < 15) printf (" "); else printf ("\n");
  }
}

/* Dump whole flag block */

static void flagblockdump (unsigned char *b, int s)
{
  int i;

  if (debug) {
    for (i = 0; i < s; i += 16) {
      printf ("b3 XXXX:  "); wide_hexdump (b + i, 16);
    }
  }

  for (i = 0; i < s; i += 16) {
    flagdump (b + i, i + 1);
  }

}

/* Dump DTMF memories. */

static void dtmfdump (unsigned char *b, unsigned char flags)
{
  unsigned char bit;
  int i;

  for (i = 0, bit = 0x01; i < 8; i++, bit <<= 1) {
    printf ("%s: ", dtmfmem[i]);
    if (flags & bit)
      printf ("D ");
    else
      printf ("- ");
    dtmfmemdump (b + i * 2);
  }
}

/* Dump AD memory. */

static void addump (unsigned char *b, int i)
{
  int no = b[3];
  unsigned char *cp;
  char c;
  int j, limit;

  if ((b[0] == 0x0d) && (b[1] == 0x16) && (b[2] == 0x24)
      && ((no >= 0) && (no <= 9))) {
    printf ("%s: ", admem[no - 1]);
    limit = 16;
    if (no == 9) limit = 32;
    for (j = 0, cp = b + 4; j < limit; j ++, cp ++) {
      if (*cp == 0xff)
	break;
      switch (*cp) {
      case 0x00: c = '0'; break;
      case 0x01: c = '1'; break;
      case 0x02: c = '2'; break;
      case 0x03: c = '3'; break;
      case 0x04: c = '4'; break;
      case 0x05: c = '5'; break;
      case 0x06: c = '6'; break;
      case 0x07: c = '7'; break;
      case 0x08: c = '8'; break;
      case 0x09: c = '9'; break;
      case 0x0a: c = 'A'; break;
      case 0x0b: c = 'B'; break;
      case 0x0c: c = 'C'; break;
      case 0x0d: c = 'D'; break;
      case 0x0e: c = '*'; break;
      case 0x0f: c = '#'; break;
      default:   c = '?'; break;
      }
      putchar (c);
    }
    putchar ('\n');
  } else {
    printf ("  D?:\t"); hexdump (b, 20);
  }

}

/* Dump one channel slot. */

static void chandump (unsigned char *b, unsigned char c, char *name)
{
  int i, isvalid = 0;
  
  printf ("%4.4s: %s ", name, flag_string (c));

  /* If channel slot is filled with 0xff, just print "XX" to all fields.
     Heuristics is this: If any byte in channel data is something else than
     0xff, we assume that channel data is entirely valid. */
  for (i = 0; i < 16; i ++) {
    if (b[i] != 0xff) {
      isvalid = 1;
      break;
    }
  }

  if (isvalid) {
    printf ("%s %s %s %s %s %s %s %s %s ",
	    txpo_string ((b[1] & 0xf0) >> 4),
	    step_string (b[1] & 0x0f),
	    code_string ((b[2] & 0xe0) >> 5),
	    paging_string ((b[2] & 0x0c) >> 2),
	    shift_string (b[2] & 0x03),
	    selcal_string ((b[3] & 0xc0) >> 6),
	    ctcss_string (b[3] & 0x3f),
	    dcs_string (b[4]),
	    mode_string (b[5]));

    printf ("%s ", freq_string (b + 6));
    printf ("%s", freq_string (b + 9));
    
    if (b[0] & 0x80)
      printf (" %s", text_string (b + 12));
  } else {
    printf ("XX XXXX X XX XX XXX XXXXX XXX XXX XXXXXXX XXXXXXX");
  }

  printf ("\n");
	     
}

static void dump_b1 (unsigned char *b, int s)
{
  printf ("b1 000:\t");
  hexdump (b, s);
}

static void dump_b2 (unsigned char *b, int s)
{
  printf ("b2 000:\t");
  hexdump (b, s);
}

static void dump_b3 (unsigned char *b, int s, unsigned char *b7)
{
  printf ("#b3\tChannel flags:\n");

  flagblockdump (b, s);

  if (memcmp (b, b7 + 2, s) != 0) {
    printf ("# WARNING: b3 and its duplicate in b7 differ !\n");
  }

}

static void dump_b4 (unsigned char *b, int s)
{
  printf ("# b4\tVHF Home:\n");
  printf ("# Mem  Fl Po Step C Pg Sh Sel CTCSS DCS Mod Rx      Off/Tx   Name\n");
  if (debug) { printf ("b4 %03x:\t", 0); hexdump (b, 16); }
  chandump (b, 0x03, "VHFH");
}

static void dump_b5 (unsigned char *b, int s)
{
  printf ("# b5\tUHF Home:\n");
  printf ("# Mem  Fl Po Step C Pg Sh Sel CTCSS DCS Mod Rx      Off/Tx   Name\n");
  if (debug) { printf ("b5 %03x:\t", 0); hexdump (b, 16); }
  chandump (b, 0x03, "UHFH");
}

static void dump_b6 (unsigned char *b, int s, char *b3)
{
  int inx, off;
  
  printf ("# b6\tChannel data:\n");
  printf ("# Mem  Fl Po Step C Pg Sh Sel CTCSS DCS Mod Rx      Off/Tx   Name\n");
  for (inx = 0, off = 0; off < (s - 16); off += 16, inx ++) {
    if (debug) { printf ("b6 %03x:\t", off); hexdump (b + off, 16); }
    chandump (b + off, b3[inx], channels[inx]);
  }
  if (debug) { printf ("b6 %03x:\t", off); hexdump (b + off, 16); }
}

static void dump_b7 (unsigned char *b, int s)
{
  int i, off;
  unsigned char uc;
  
  printf ("# b7\tVarious parameters, VFOs, DTMF, autodial, unknown data:\n");

  /* First dump known data. */

  /* For some reason, channel flags are duplicated here ?? */
  printf ("# Channel flags (duplicated)\n");
  flagblockdump (b + 2, 112);
  
  printf ("# Mem  Fl Po Step C Pg Sh Sel CTCSS DCS Mod Rx      Off/Tx   Name\n");

  for (i = 0; i < 10; i ++) {
    off = 116 + i * 16;
    if (debug) { printf ("b7 %03x:\t", off); hexdump (b + off, 16); }
    chandump (b + (116 + i * 16), 0x03, vfo_string(i));
  }
  /* Band edges are probably updated automatically by radio itself when band
     is changed, hence we do not try to change binary data with these.
     (Band edges are printed out as a comment). */
  if (debug) {
    printf ("b7 %03x:\t", 276); hexdump (b + 276, 3);
    printf ("b7 %03x:\t", 280); hexdump (b + 280, 3);
  }
  printf ("# Current band edges : %s - ", freq_string (b + 276)); /*XXX*/
  printf ("%s\n", freq_string (b + 280));
  if (debug) { printf ("b7 %03x:\t", 301); hexdump (b + 301, 8); }
  printf (" SUB: %s\n", sub_string (b[301]));
  printf (" APO: %s\n", apo_string (b[303]));
  printf (" TOT: %s\n", tot_string (b[304]));
  printf ("LOCK: %s\n", lock_string (b[305]));
  printf ("RSAV: %s\n", rsav_string (b[306]));
  printf ("LAMP: %s\n", lamp_string (b[307]));
  printf ("BELL: %s\n", bell_string (b[308]));
  if (debug) { printf ("b7 %03x:\t", 309); hexdump (b + 309, 16); }
  printf ("  ID: %s\n", id_string (b + 309));
  if (debug) { printf ("b7 %03x:\t", 326); hexdump (b + 326, 2); }
  printf ("ARTS: %s\n", arts_string (b[326]));
  printf ("ARTS/BEEP: %s\n", arts_beep_string (b[327]));
  for (i = 0; i < 9; i ++) {
    off = 330 + i * 20;
    if (debug) {
      printf ("b7 %03x:\t", off);
      hexdump (b + off, i == 8 ? 36 : 20);
    }
    addump (b + (330 + i * 20), i);
  }
  if (debug) { printf ("b7 %03x:\t", 526); hexdump (b + 526, 17); }
  dtmfdump (b + 526, b[526 + 16]);
  if (debug) { printf ("b7 %03x:\t", 543); hexdump (b + 543, 4); }
  printf ("PAGE/SPED: %s\n", page_sped_string (b[543]));
  printf ("PAGE/DLAY: %s\n", page_dlay_string (b[544]));
  printf ("PAGE/BELL: %s\n", bell_string (b[545]));
  printf ("PAGE/ASBK: %s\n", page_asbk_string (b[546]));
  if (debug) { printf ("b7 %03x:\t", 1660); hexdump (b + 1660, 19); }
  printf ("CURR/VFOA: %s\n", vfo_string (b[1660]));
  printf ("CURR/VFOB: %s\n", vfo_string (b[1661] + 5));
#if 0
  printf ("CURR/VFO: ");
  switch (b[1662]) {
  case 0x00: printf ("B\n"); break;
  case 0x06: printf ("A\n"); break;
  default: printf ("?%02x\n", b[1662]); break;
  }
#endif
  printf ("# Mem  Fl Po Step C Pg Sh Sel CTCSS DCS Mod Rx      Off/Tx\n");
  printf ("CURR:  U %s %s %s %s %s %s %s %s %s %s ",
	  txpo_string (b[1663]),
	  step_string (b[1664]),
	  code_string (b[1665]),
	  paging_string (b[1666]),
	  shift_string (b[1667]),
	  selcal_string (b[1668]),
	  ctcss_string (b[1669]),
	  dcs_string (b[1670]),
	  mode_string (b[1671]),
	  freq_string (b + 1672));
  printf ("%s\n", freq_string (b + 1676));
  if (debug) { printf ("b7 %03x:\t", 1686); hexdump (b + 1686, 5); }
  printf (" SQL: %s\n", sql_string (b[1686]));
  printf ("WSQL: %s\n", wsql_string (b[1687]));
  uc = b[1688];
  printf ("RPTL: "); if (uc & 0x80) printf ("on\n"); else printf ("off\n");
  printf ("AMOD: "); if (uc & 0x40) printf ("on\n"); else printf ("off\n");
  printf ("SCNL: "); if (uc & 0x20) printf ("on\n"); else printf ("off\n");
  printf ("RESM: "); if (uc & 0x10) printf ("carr\n"); else printf ("5sec\n");
  printf (" ARS: "); if (uc & 0x08) printf ("on\n"); else printf ("off\n");
  printf ("BEEP: "); if (uc & 0x04) printf ("on\n"); else printf ("off\n");
  printf (" LCK: "); if (uc & 0x02) printf ("on\n"); else printf ("off\n");
  uc = b[1689];
  printf (" LGT: "); if (uc & 0x80) printf ("on\n"); else printf ("off\n");
  printf ("PAGE/AMSG: ");
  if (uc & 0x40) printf ("on\n"); else printf ("off\n");
  printf ("BCLO: "); if (uc & 0x10) printf ("on\n"); else printf ("off\n");
  printf ("CWID: "); if (uc & 0x02) printf ("on\n"); else printf ("off\n");
  printf ("TSAV: "); if (uc & 0x01) printf ("on\n"); else printf ("off\n");
  uc = b[1690];
  printf ("ARTS/SPED: ");
  if (uc & 0x08) printf ("25\n"); else printf ("15\n");
  printf ("RVHM: "); if (uc & 0x02) printf ("rev\n"); else printf ("home\n");
  printf (" MON: "); if (uc & 0x01) printf ("tcal\n"); else printf ("mon\n");

  if (debug) {
    /* Then unknown. */
    printf ("b7 %03x:\t", 0); hexdump (b, 2);
    printf ("b7 %03x:\t", 279); hexdump (b + 279, 1);
    printf ("b7 %03x:\t", 283); hexdump (b + 283, 5);
    printf ("b7 %03x:\t", 288); hexdump (b + 288, 13);
    printf ("b7 %03x:\t", 325); hexdump (b + 325, 1);
    printf ("b7 %03x:\t", 328); hexdump (b + 328, 2);
    printf ("b7 %03x:\t", 547); hexdump (b + 547, 13);
    for (i = 35; i < 103; i ++) {
      printf ("b7 %03x:\t", i * 16); hexdump (b + i * 16, 16);
    }
    printf ("b7 %03x:\t", 1648); hexdump (b + 1648, 12);
    printf ("b7 %03x:\t", 1679); hexdump (b + 1679, 1);
    printf ("b7 %03x:\t", 1680); hexdump (b + 1680, 6);
    printf ("b7 %03x:\t", 1691); hexdump (b + 1691, 5);
    for (i = 106; i < 111; i ++) {
      printf ("b7 %03x:\t", i * 16); hexdump (b + i * 16, 16);
    }
  }
}

static void dump_b8 (unsigned char *b, int s, unsigned char cksum)
{
  printf ("# b8\tComputed checksum = %02x ", cksum);
  if (*b == cksum)
    printf ("(match).\n");
  else
    printf ("FAIL!\n");
  printf ("b8 000:\t"); hexdump (b, s);
}

int main (int argc, char *argv[])
{
  unsigned char b1[10], b2[16], b3[112], b4[16], b5[16], b6[1776], b7[1776],
  b8[1];
  int ret, fd, c;
  unsigned char cksum;

  while ((c = getopt (argc, argv, "d")) != -1)
    switch (c) {
    case 'd': debug ++; break;
    case '?': printf ("Usage: mktext [-d] <infile>\n"); exit (1); break;
    }
 
  if ((argc - optind) != 1) {
    printf ("Usage: mktext [-d] <infile>\n");
    exit (1);
  }

  fd = open (argv[optind], O_RDONLY);
  if (fd < 0) {
    perror (argv[optind]);
    exit (1);
  }

  ret = read_save_file (fd, b1, b2, b3, b4, b5, b6, b7, b8);
  if (ret != 0) {
    perror ("read_save_file");
    exit (1);
  }

  close (fd);

  cksum = calculate_cksum (b1, b2, b3, b4, b5, b6, b7);

  printf ("# Dump of file %s\n\n", argv[optind]);

  dump_b1 (b1, sizeof (b1));
  dump_b2 (b2, sizeof (b2));
  dump_b3 (b3, sizeof (b3), b7);
  dump_b4 (b4, sizeof (b4));
  dump_b5 (b5, sizeof (b5));
  dump_b6 (b6, sizeof (b6), b3);
  dump_b7 (b7, sizeof (b7));
  dump_b8 (b8, sizeof (b8), cksum);

  exit (0);
  
}
