
/* mkbin.c - generate binary version from radio text file and template.

   Copyright (c) 1997 Riku Kalinen, OH2LWO, oh2lwo@sral.fi

   RCS $Header: /home/mole/riku/CVS/yaesu/vx1mkbin.c,v 1.2 1997/12/11 15:02:00 riku Exp $ */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "yaesu.h"

/* Generate debug output? */

int debug = 1;

/* Data buffers. */

unsigned char b1[4818];

/* Set channel name. */

static int nameset (unsigned char *base, char *name)
{
  int i, j;
  
  if (strlen (name) > 6)
    return -1;

  for (j = 0; j < 6; j++) {
    if (name[j] == 0)
      break;
    i = vx1_char_bin (name[j]);
    if (i < 0)
      return -1;
    base[j] = (unsigned char) (i & 0xff);
  }

  return 0;
}

static int power_clksft_step (char *power_s, char *clksft_s, char *step_s)
{
  int i, ret = 0;

  i = vx1_txpwr_bin (power_s);
  if (i < 0) return -11;
  ret |= i << 4;

  i = vx1_clksft_bin (clksft_s);
  if (i < 0) return -12;
  ret |= i << 3;

  i = vx1_step_bin (step_s);
  if (i < 0) return -13;
  ret |= i;

  return ret;
}

static int skip_freqalpha_mode_repeater_selcal (char *skip_s, char
						*freqalpha_s, char *mode_s,
						char *repeater_s,
						char *selcal_s) 
{
  int i, ret = 0;

  i = vx1_skip_bin (skip_s);
  if (i < 0) return -11;
  ret |= i << 7;

  i = vx1_nameflag_bin (freqalpha_s);
  if (i < 0) return -12;
  ret |= i << 6;

  i = vx1_mode_bin (mode_s);
  if (i < 0) return -13;
  ret |= i << 4;

  i = shift_bin (repeater_s);
  if (i < 0) return -14;
  ret |= i << 2;

  i = selcal_bin (selcal_s);
  if (i < 0) return -15;
  ret |= i;

  return ret;
}

/* Set one 8-byte channel slot up. */

static int chanset8 (unsigned char *base, char *rx_s, char *name_s,
		     unsigned char u1)
{
  int ret, i;

  memset (base, 0x00, 8);
  
  memset (base + 2, 0x3f, 6);

  if (strcasecmp (rx_s, "XX") == 0) {
    base[0] = 0xff;
    return 0;
  }
  
  if (name_s) {
    ret = nameset (base + 2, name_s);
    if (ret < 0) return -12;
  } 

  i = vx1_bcfreq_bin (rx_s);
  if (i < 0) return -11;
  base[0] = (unsigned char) (i & 0xff);

  base[1] = u1;
  
  return 0;
}

/* Set one 12-byte channel slot up. */

static int chanset12 (unsigned char *base, char *rx_s, char *skip_s,
		      char *freqalpha_s, char *mode_s, char *repeater_s,
		      char *selcal_s, char *power_s, char *clksft_s, char
		      *step_s, char *name_s)
{
  int ret, i;
  unsigned char freq[8];
  
  memset (base + 6, 0x3f, 6);
  if (name_s) {
    ret = nameset (base + 6, name_s);
    if (ret < 0) return -21;
  } 

  if (vx1_freq_bin (freq, rx_s) != 0) return -12;
  (void) memcpy (base + 0, freq, 3);
  
  i = skip_freqalpha_mode_repeater_selcal (skip_s, freqalpha_s, mode_s,
					   repeater_s, selcal_s);
  if (i < 0) return i - 2;
  base[4] = (unsigned char) (i & 0xff);

  i = power_clksft_step (power_s, clksft_s, step_s);
  if (i < 0) return i - 7;
  base[5] = (unsigned char) (i & 0xff);

  return 0;
}

/* Set one 16-byte channel slot up. */

static int chanset16 (unsigned char *base, char *rx_s, char *txoff_s, char
		      *skip_s, char *freqalpha_s, char *mode_s, char
		      *repeater_s, char *selcal_s, char *power_s, char
		      *clksft_s, char *step_s, char *tone_s, char *dcs_s, char
		      *name_s)
{
  int ret, i;
  unsigned char freq[8];
  
  memset (base + 10, 0x3f, 6);
  if (name_s) {
    ret = nameset (base + 10, name_s);
    if (ret < 0) return -24;
  } 

  if (vx1_freq_bin (freq, rx_s) != 0) return -12;
  (void) memcpy (base + 0, freq, 3);
  
  if (vx1_freq_bin (freq, txoff_s) != 0) return -13;
  (void) memcpy (base + 3, freq, 3);

  i = skip_freqalpha_mode_repeater_selcal (skip_s, freqalpha_s, mode_s,
					   repeater_s, selcal_s);
  if (i < 0) return i - 3;
  base[6] = (unsigned char) (i & 0xff);

  i = power_clksft_step (power_s, clksft_s, step_s);
  if (i < 0) return i - 9;
  base[7] = (unsigned char) (i & 0xff);

  i = ctcss_bin (tone_s);
  if (i < 0) return -22;
  base[8] = (unsigned char) (i & 0xff);

  i = dcs_bin (dcs_s);
  if (i < 0) return -23;
  base[9] = (unsigned char) (i & 0xff);

  return 0;
}

/* Error codes for hook functions:
   -1: General error
   -2: Argument count error
   -10: Tag error
   -11: Parameter 1 error
   ...
   -23: Parameter 13 error
   -24: Parameter 14 error */

/* Set one channel slot. */

#define DEFUN(NAME) int NAME (char *tag, char *d1, char *d2, char *d3, \
			      char *d4, char *d5, char *d6, char *d7, \
			      char *d8, char *d9, char *d10, char *d11, \
			      char *d12, char *d13, char *d14)

#define DEBUG(NAME) if (debug) \
printf ("%s: %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s\n", NAME, tag, \
	d1 ? d1 : "NULL", d2 ? d2 : "NULL", d3 ? d3 : "NULL", \
	d4 ? d4 : "NULL", d5 ? d5 : "NULL", d6 ? d6 : "NULL", \
	d7 ? d7 : "NULL", d8 ? d8 : "NULL", d9 ? d9 : "NULL", \
	d10 ? d10 : "NULL", d11 ? d11 : "NULL", d12 ? d12 : "NULL", \
	d13 ? d13 : "NULL", d14 ? d14 : "NULL")

static DEFUN(set_band)
{
  int band;
  
  DEBUG("set_band");

  if (d1 == NULL) return -2;

  band = vx1_band_bin (d1);
  if (band < 0)
    return -11;

  if (debug)
    printf ("set_band: before band %d 0x0002, 0x0302 = %02x, %02x\n",
	    band, b1[0x0002], b1[0x0302]);

  b1[0x0002] &= 0xf0;
  b1[0x0002] |= band;
  b1[0x0302] &= 0xf0;
  b1[0x0302] |= band;

  if (debug)
    printf ("set_band: after band %d 0x0002, 0x0302 = %02x, %02x\n",
	    band, b1[0x0002], b1[0x0302]);

  return 0;
}

static DEFUN(set_bsyled)
{
  int onoff;
  
  DEBUG("set_bsyled");

  if (d1 == NULL) return -2;

  onoff = onoff_bin (d1);
  if (onoff < 0)
    return -11;

  setbit (b1 + 0x0007, 6, onoff);
  setbit (b1 + 0x0307, 6, onoff);

  return 0;
}

static DEFUN(set_bclo)
{
  int onoff;
  
  DEBUG("set_bclo");

  if (d1 == NULL) return -2;

  onoff = onoff_bin (d1);
  if (onoff < 0)
    return -11;

  setbit (b1 + 0x0007, 5, onoff);
  setbit (b1 + 0x0307, 5, onoff);

  return 0;
}

static DEFUN(set_beep)
{
  int onoff;
  
  DEBUG("set_beep");

  if (d1 == NULL) return -2;

  onoff = onoff_bin (d1);
  if (onoff < 0)
    return -11;

  setbit (b1 + 0x0007, 4, onoff);
  setbit (b1 + 0x0307, 4, onoff);

  return 0;
}

static DEFUN(set_scnl)
{
  int onoff;
  
  DEBUG("set_scnl");

  if (d1 == NULL) return -2;

  onoff = onoff_bin (d1);
  if (onoff < 0)
    return -11;

  setbit (b1 + 0x0007, 3, onoff);
  setbit (b1 + 0x0307, 3, onoff);

  return 0;
}

static DEFUN(set_resume)
{
  int resume;
  
  DEBUG("set_resume");

  if (d1 == NULL) return -2;

  resume = vx1_resume_bin (d1);
  if (resume < 0)
    return -11;

  setbit (b1 + 0x0007, 2, resume);
  setbit (b1 + 0x0307, 2, resume);

  return 0;
}

static DEFUN(set_atmd)
{
  int atmd;
  
  DEBUG("set_atmd");

  if (d1 == NULL) return -2;

  atmd = vx1_atmd_bin (d1);
  if (atmd < 0)
    return -11;

  setbit (b1 + 0x0007, 1, atmd);
  setbit (b1 + 0x0307, 1, atmd);

  return 0;
}

static DEFUN(set_ars)
{
  int val;
  
  DEBUG("set_ars");

  if (d1 == NULL) return -2;

  val = onoff_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x0007, 0, val);
  setbit (b1 + 0x0307, 0, val);

  return 0;
}

static DEFUN(set_lamp)
{
  int val;
  
  DEBUG("set_lamp");

  if (d1 == NULL) return -2;

  val = vx1_lamp_bin (d1);
  if (val < 0)
    return -11;

  b1[0x0008] &= 0x3f;
  b1[0x0008] |= val << 6;
  b1[0x0308] &= 0x3f;
  b1[0x0308] |= val << 6;

  return 0;
}

static DEFUN(set_apo)
{
  int val;
  
  DEBUG("set_apo");

  if (d1 == NULL) return -2;

  val = apo_bin (d1);
  if (val < 0)
    return -11;

  b1[0x0008] &= 0xc7;
  b1[0x0008] |= val << 3;
  b1[0x0308] &= 0xc7;
  b1[0x0308] |= val << 3;

  return 0;
}

static DEFUN(set_rxsave)
{
  int val;
  
  DEBUG("set_rxsave");

  if (d1 == NULL) return -2;

  val = rsav_bin (d1);
  if (val < 0)
    return -11;

  b1[0x0008] &= 0xf8;
  b1[0x0008] |= val;
  b1[0x0308] &= 0xf8;
  b1[0x0308] |= val;

  return 0;
}

static DEFUN(set_lk)
{
  int val;
  
  DEBUG("set_lk");

  if (d1 == NULL) return -2;

  val = onoff_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x0009, 7, val);
  setbit (b1 + 0x0309, 7, val);

  return 0;
}

static DEFUN(set_lock)
{
  int val;
  
  DEBUG("set_lock");

  if (d1 == NULL) return -2;

  val = lock_bin (d1);
  if (val < 0)
    return -11;

  b1[0x0009] &= 0x8f;
  b1[0x0009] |= val << 4;
  b1[0x0309] &= 0x8f;
  b1[0x0309] |= val << 4;

  return 0;
}

static DEFUN(set_bell)
{
  int val;
  
  DEBUG("set_bell");

  if (d1 == NULL) return -2;

  val = bell_bin (d1);
  if (val < 0)
    return -11;

  b1[0x0009] &= 0xfc;
  b1[0x0009] |= val;
  b1[0x0309] &= 0xfc;
  b1[0x0309] |= val;

  return 0;
}

static DEFUN(set_artsbp)
{
  int val;
  
  DEBUG("set_artsbp");

  if (d1 == NULL) return -2;

  val = arts_beep_bin (d1);
  if (val < 0)
    return -11;

  b1[0x000a] &= 0x3f;
  b1[0x000a] |= val << 6;
  b1[0x030a] &= 0x3f;
  b1[0x030a] |= val << 6;

  return 0;
}

static DEFUN(set_smtmd)
{
  int val;
  
  DEBUG("set_smtmd");

  if (d1 == NULL) return -2;

  val = vx1_smtmd_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x000a, 5, val);
  setbit (b1 + 0x030a, 5, val);

  return 0;
}

static DEFUN(set_tot)
{
  int val;
  
  DEBUG("set_tot");

  if (d1 == NULL) return -2;

  val = vx1_tot_bin (d1);
  if (val < 0)
    return -11;

  b1[0x000a] &= 0xe3;
  b1[0x000a] |= val << 2;
  b1[0x030a] &= 0xe3;
  b1[0x030a] |= val << 2;

  return 0;
}

static DEFUN(set_dialm)
{
  int val;
  
  DEBUG("set_dialm");

  if (d1 == NULL) return -2;

  val = vx1_dialm_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x000a, 1, val);
  setbit (b1 + 0x030a, 1, val);

  return 0;
}

static DEFUN(set_montc)
{
  int val;
  
  DEBUG("set_montc");

  if (d1 == NULL) return -2;

  val = vx1_montc_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x000a, 0, val);
  setbit (b1 + 0x030a, 0, val);

  return 0;
}

static DEFUN(set_grp)
{
  int val;
  
  DEBUG("set_grp");

  if (d1 == NULL) return -2;

  val = vx1_group_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x000b, 2, val);
  setbit (b1 + 0x030b, 2, val);

  return 0;
}

static DEFUN(set_cwid)
{
  int val;
  
  DEBUG("set_cwid");

  if (d1 == NULL) return -2;

  val = onoff_bin (d1);
  if (val < 0)
    return -11;

  setbit (b1 + 0x000b, 1, val);
  setbit (b1 + 0x030b, 1, val);

  return 0;
}

static DEFUN(set_dtmfm)
{
  int val;
  
  DEBUG("set_dtmfm");

  if (d1 == NULL) return -2;

  val = vx1_dtmfm_bin (d1);
  if (val < 0)
    return -11;

  b1[0x000e] &= 0xf0;
  b1[0x000e] |= val;
  b1[0x030e] &= 0xf0;
  b1[0x030e] |= val;

  return 0;
}

static DEFUN(set_vol)
{
  int val;
  
  DEBUG("set_vol");

  if (d1 == NULL) return -2;

  val = vx1_vol_bin (d1);
  if (val < 0)
    return -11;

  b1[0x000f] = val;
  b1[0x030f] = val;

  return 0;
}

static DEFUN(set_sql)
{
  int val;
  
  DEBUG("set_sql");

  if (d1 == NULL) return -2;

  val = vx1_sql_bin (d1);
  if (val < 0)
    return -11;

  b1[0x0010] = val;
  b1[0x0310] = val;

  return 0;
}

static DEFUN(set_curr)
{
  DEBUG("set_curr");

  if (d13 == NULL) return -2;

  return chanset16 (b1 + 0x0011, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11,
		    d12, d13, d14);
}

/* set_ss: tag "SS/<no>:", -15 <= no <= 15 */

static DEFUN(set_ss)
{
  int no;
  int off;
  
  DEBUG("set_ss");

  no = atoi (tag + 3);
  off = ((no + 15) * 16) + 0x00f1;
  
  if (d13 == NULL) return -2;

  return chanset16 (b1 + off, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11,
		    d12, d13, d14);
}

static DEFUN(set_id)
{
  int i, j, ret = 0;
  unsigned char *base = b1 + 0x02f1;
  char newid[32];
  
  DEBUG("set_id");

  if (d1 == NULL) {
    strcpy (newid, "");
  } else {
    if (strlen (d1) > 8) return -11;
    strcpy (newid, d1);
  }

  memset (base, 0x00, 9);

  for (j = 0; j < 9; j ++) {
    if (newid[j] == 0) {
      base[j] = 0x3d;
      break;
    }
    i = vx1_char_bin (newid[j]);
    if (i < 0) ret = -11;
    base[j] = (unsigned char) (i & 0xff);
  }

  return ret;
}
 
static DEFUN(set_vfo_bcband)
{
  DEBUG("set_vfo_bcband");

  if (d1 == NULL) return -2;

  return chanset8 (b1 + 0x0311, d1, d2, 0x00);
}

/* set_vfo: tag "VFO/<band>:" */

static DEFUN(set_vfo)
{
  int no;
  int off;
  char *cp;
  
  DEBUG("set_vfo");

  for (cp = tag; *cp; cp ++)
    if (*cp == ':') {
      *cp = 0;
      break;
    }
  no = vx1_band_bin (tag + 4) - 1;
  off = (no * 16) + 0x0321;

  if (debug)
    printf ("set_vfo: no = %d, off = %04x\n", no, off);
  
  if (d13 == NULL) return -2;

  return chanset16 (b1 + off, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11,
		    d12, d13, d14);
}

static DEFUN(set_pri)
{
  int no;
  
  DEBUG("set_pri");

  if (d1 == NULL) return -11;

  no = atoi (d1);
  if ((no < 0) || (no > 142)) return -11;

  b1[0x03a1] = (unsigned char) ((no + 7) & 0xff);

  return 0;
}

static int fixbits (unsigned baseoff, char *d1, int no)
{
  int i, off, bitoff, bit, band;
  int masked = 0;
  
  if ((strcasecmp (d1, "Masked") == 0) ||
      (strcasecmp (d1, "XX") == 0)) masked = 1;
  bitoff = (no - 8) / 8;
  bit = (no - 8) % 8;

  for (i = 0; i < 8; i ++) {
    off = baseoff + i * 32 + 1 + bitoff;
    setbit (b1 + off, bit, 0);
  }
  
  if (! masked) {
    band = vx1_band_bin (d1);
    if (band < 0) return -11;
    band --;
    off = baseoff + band * 32 + 1 + bitoff;
    setbit (b1 + off, bit, 1);
  }

  return 0;

}

/* set_mem: tag "CG<cg>/<memory>:" */

static DEFUN(set_mem)
{
  int no, cg, i, off;
  char *cp;
  
  DEBUG("set_mem");

  for (cp = tag; *cp; cp ++)
    if (*cp == ':') {
      *cp = 0;
      break;
    }
  tag[3] = 0;
  cg = atoi (tag + 2);

  if (cg == 1) {
    no = vx1_chan_cg1_bin (tag + 4);
    if (no > 7) {
      i = fixbits (0x10e7, d1, no);
      if (i < 0) return -11;
    }
    off = 0x0bd1 + no * 16;
    if (strcasecmp (d1, "XX") == 0) {
      memset (b1 + off, 0xff, 16);
      return 0;
    }
    if (d13 == NULL) return -2;
    return chanset16 (b1 + off, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11,
		      d12, d13, d14);
  } else {
    no = vx1_chan_cg2_bin (tag + 4);
    if (no > 7) {
      i = fixbits (0x10d1, d1, no);
      if (i < 0) return -11;
    }
    off = 0x03d1 + no * 12;
    if (strcasecmp (d1, "XX") == 0) {
      memset (b1 + off, 0xff, 12);
      return 0;
    }
    if (d10 == NULL) return -2;
    return chanset12 (b1 + off, d2, d3, d4, d5, d6, d7, d8, d9, d10, d11);
  }

  return 0;

}

/* set_bc_mem: tag "BCBAND/<ch>:" */

static DEFUN(set_bc_mem)
{
  int no, off;
  char *cp;
  
  DEBUG("set_bc_mem");

  if (d1 == NULL) return -2;

  for (cp = tag; *cp; cp ++)
    if (*cp == ':') {
      *cp = 0;
      break;
    }
  no = vx1_chan_bc_bin (tag + 7);

  off = 0x11f1 + no * 8;

  return chanset8 (b1 + off, d1, d2, 0x40);
}

/* set_ad: tag "dmtf-<no>:" */

static DEFUN(set_ad)
{
  unsigned char *base;
  int off, no, i, j, ret = 0, limit = 16;
  char newad[64];
    
  DEBUG("set_ad");

  if (d1 == NULL) {
    strcpy (newad, "");
  } else {
    if (strlen (d1) > 15) return -11;
    strcpy (newad, d1);
  }

  no = atoi (tag + 5);
  off = 0x1251 + no * 16;

  base = b1 + off;
  memset (base, 0x00, limit);
  
  for (j = 0; j < limit; j ++) {
    if (newad[j] == 0) {
      base[j] = 0x3d;
      break;
    }
    i = vx1_dtmf_bin (newad[j]);
    if (i < 0) ret = -11;
    base[j] = (unsigned char) (i & 0xff);
  }

  return ret;
}
 
/* Tag table. This binds line tags and corresponding function together. */

typedef struct tagfunc {
  char *tag;
  int (*func)(char *, char *, char *, char *, char *, char *, char *, char *,
	      char *, char *, char *, char *, char *, char *, char *);
} Tag;

Tag tags[] = {
  /* Misc. data */
  { "BAND:", set_band }, { "BSYLED:", set_bsyled }, { "BCLO:", set_bclo },
  { "BEEP:", set_beep }, { "SCNL:", set_scnl }, { "RESUME:", set_resume },
  { "ATMD:", set_atmd }, { "ARS:", set_ars }, { "LAMP:", set_lamp },
  { "APO:", set_apo }, { "RXSAVE:", set_rxsave }, { "LK:", set_lk },
  { "LOCK:", set_lock }, { "BELL:", set_bell }, { "ARTSBP:", set_artsbp },
  { "SMTMD:", set_smtmd }, { "TOT:", set_tot }, { "DIALM:", set_dialm },
  { "MONTC:", set_montc }, { "GRP:", set_grp }, { "CWID:", set_cwid },
  { "DTMFM:", set_dtmfm }, { "VOL:", set_vol }, { "SQL:", set_sql },
  /* Current settings */
  { "CURR:", set_curr },
  /* Smart search memories */
  { "SS/-15:", set_ss }, { "SS/-14:", set_ss }, { "SS/-13:", set_ss },
  { "SS/-12:", set_ss }, { "SS/-11:", set_ss }, { "SS/-10:", set_ss },
  { "SS/-9:", set_ss }, { "SS/-8:", set_ss }, { "SS/-7:", set_ss },
  { "SS/-6:", set_ss }, { "SS/-5:", set_ss }, { "SS/-4:", set_ss },
  { "SS/-3:", set_ss }, { "SS/-2:", set_ss }, { "SS/-1:", set_ss },
  { "SS/0:", set_ss }, { "SS/1:", set_ss }, { "SS/2:", set_ss },
  { "SS/3:", set_ss }, { "SS/4:", set_ss }, { "SS/5:", set_ss },
  { "SS/6:", set_ss }, { "SS/7:", set_ss }, { "SS/8:", set_ss },
  { "SS/9:", set_ss }, { "SS/10:", set_ss }, { "SS/11:", set_ss },
  { "SS/12:", set_ss }, { "SS/13:", set_ss }, { "SS/14:", set_ss },
  { "SS/15:", set_ss },
  /* Call sign */
  { "ID:", set_id },
  /* BC VFO */
  { "VFO/BCBAND:", set_vfo_bcband },
  /* VFOs */
  { "VFO/FM:", set_vfo }, { "VFO/AIR:", set_vfo }, { "VFO/V-HAM:", set_vfo },
  { "VFO/VHF-TV:", set_vfo }, { "VFO/ACT1:", set_vfo },
  { "VFO/U-HAM:", set_vfo }, { "VFO/UHF-TV:", set_vfo },
  { "VFO/ACT2:", set_vfo },
  /* Priority channel */
  { "PRI:", set_pri },
  /* CG 2 */
  { "CG2/H-FM:", set_mem }, { "CG2/H-AIR:", set_mem },
  { "CG2/H-V-HA:", set_mem }, { "CG2/H-V-TV:", set_mem },
  { "CG2/H-ACT1:", set_mem }, { "CG2/H-U-HA:", set_mem }, 
  { "CG2/H-U-TV:", set_mem }, { "CG2/H-ACT2:", set_mem },
  { "CG2/1:", set_mem }, 
  { "CG2/2:", set_mem }, { "CG2/3:", set_mem }, { "CG2/4:", set_mem }, 
  { "CG2/5:", set_mem }, { "CG2/6:", set_mem }, { "CG2/7:", set_mem }, 
  { "CG2/8:", set_mem }, { "CG2/9:", set_mem }, { "CG2/10:", set_mem }, 
  { "CG2/11:", set_mem }, { "CG2/12:", set_mem }, { "CG2/13:", set_mem }, 
  { "CG2/14:", set_mem }, { "CG2/15:", set_mem }, { "CG2/16:", set_mem }, 
  { "CG2/17:", set_mem }, { "CG2/18:", set_mem }, { "CG2/19:", set_mem }, 
  { "CG2/20:", set_mem }, { "CG2/21:", set_mem }, { "CG2/22:", set_mem }, 
  { "CG2/23:", set_mem }, { "CG2/24:", set_mem }, { "CG2/25:", set_mem }, 
  { "CG2/26:", set_mem }, { "CG2/27:", set_mem }, { "CG2/28:", set_mem }, 
  { "CG2/29:", set_mem }, { "CG2/30:", set_mem }, { "CG2/31:", set_mem }, 
  { "CG2/32:", set_mem }, { "CG2/33:", set_mem }, { "CG2/34:", set_mem }, 
  { "CG2/35:", set_mem }, { "CG2/36:", set_mem }, { "CG2/37:", set_mem }, 
  { "CG2/38:", set_mem }, { "CG2/39:", set_mem }, { "CG2/40:", set_mem }, 
  { "CG2/41:", set_mem }, { "CG2/42:", set_mem }, { "CG2/43:", set_mem }, 
  { "CG2/44:", set_mem }, { "CG2/45:", set_mem }, { "CG2/46:", set_mem }, 
  { "CG2/47:", set_mem }, { "CG2/48:", set_mem }, { "CG2/49:", set_mem }, 
  { "CG2/50:", set_mem }, { "CG2/51:", set_mem }, { "CG2/52:", set_mem }, 
  { "CG2/53:", set_mem }, { "CG2/54:", set_mem }, { "CG2/55:", set_mem }, 
  { "CG2/56:", set_mem }, { "CG2/57:", set_mem }, { "CG2/58:", set_mem }, 
  { "CG2/59:", set_mem }, { "CG2/60:", set_mem }, { "CG2/61:", set_mem }, 
  { "CG2/62:", set_mem }, { "CG2/63:", set_mem }, { "CG2/64:", set_mem }, 
  { "CG2/65:", set_mem }, { "CG2/66:", set_mem }, { "CG2/67:", set_mem }, 
  { "CG2/68:", set_mem }, { "CG2/69:", set_mem }, { "CG2/70:", set_mem }, 
  { "CG2/71:", set_mem }, { "CG2/72:", set_mem }, { "CG2/73:", set_mem }, 
  { "CG2/74:", set_mem }, { "CG2/75:", set_mem }, { "CG2/76:", set_mem }, 
  { "CG2/77:", set_mem }, { "CG2/78:", set_mem }, { "CG2/79:", set_mem }, 
  { "CG2/80:", set_mem }, { "CG2/81:", set_mem }, { "CG2/82:", set_mem }, 
  { "CG2/83:", set_mem }, { "CG2/84:", set_mem }, { "CG2/85:", set_mem }, 
  { "CG2/86:", set_mem }, { "CG2/87:", set_mem }, { "CG2/88:", set_mem }, 
  { "CG2/89:", set_mem }, { "CG2/90:", set_mem }, { "CG2/91:", set_mem }, 
  { "CG2/92:", set_mem }, { "CG2/93:", set_mem }, { "CG2/94:", set_mem }, 
  { "CG2/95:", set_mem }, { "CG2/96:", set_mem }, { "CG2/97:", set_mem }, 
  { "CG2/98:", set_mem }, { "CG2/99:", set_mem }, { "CG2/100:", set_mem }, 
  { "CG2/101:", set_mem }, { "CG2/102:", set_mem }, { "CG2/103:", set_mem }, 
  { "CG2/104:", set_mem }, { "CG2/105:", set_mem }, { "CG2/106:", set_mem }, 
  { "CG2/107:", set_mem }, { "CG2/108:", set_mem }, { "CG2/109:", set_mem }, 
  { "CG2/110:", set_mem }, { "CG2/111:", set_mem }, { "CG2/112:", set_mem }, 
  { "CG2/113:", set_mem }, { "CG2/114:", set_mem }, { "CG2/115:", set_mem }, 
  { "CG2/116:", set_mem }, { "CG2/117:", set_mem }, { "CG2/118:", set_mem }, 
  { "CG2/119:", set_mem }, { "CG2/120:", set_mem }, { "CG2/121:", set_mem }, 
  { "CG2/122:", set_mem }, { "CG2/123:", set_mem }, { "CG2/124:", set_mem }, 
  { "CG2/125:", set_mem }, { "CG2/126:", set_mem }, { "CG2/127:", set_mem }, 
  { "CG2/128:", set_mem }, { "CG2/129:", set_mem }, { "CG2/130:", set_mem }, 
  { "CG2/131:", set_mem }, { "CG2/132:", set_mem }, { "CG2/133:", set_mem }, 
  { "CG2/134:", set_mem }, { "CG2/135:", set_mem }, { "CG2/136:", set_mem }, 
  { "CG2/137:", set_mem }, { "CG2/138:", set_mem }, { "CG2/139:", set_mem }, 
  { "CG2/140:", set_mem }, { "CG2/141:", set_mem }, { "CG2/142:", set_mem }, 
  { "CG2/L1:", set_mem }, { "CG2/U1:", set_mem }, { "CG2/L2:", set_mem }, 
  { "CG2/U2:", set_mem }, { "CG2/L3:", set_mem }, { "CG2/U3:", set_mem }, 
  { "CG2/L4:", set_mem }, { "CG2/U4:", set_mem }, { "CG2/L5:", set_mem }, 
  { "CG2/U5:", set_mem }, { "CG2/L6:", set_mem }, { "CG2/U6:", set_mem }, 
  { "CG2/L7:", set_mem }, { "CG2/U7:", set_mem }, { "CG2/L8:", set_mem }, 
  { "CG2/U8:", set_mem }, { "CG2/L9:", set_mem }, { "CG2/U9:", set_mem }, 
  { "CG2/L10:", set_mem }, { "CG2/U10:", set_mem }, 
  /* CG 1 */
  { "CG1/H-FM:", set_mem }, { "CG1/H-AIR:", set_mem },
  { "CG1/H-V-HA:", set_mem }, { "CG1/H-V-TV:", set_mem },
  { "CG1/H-ACT1:", set_mem }, { "CG1/H-U-HA:", set_mem }, 
  { "CG1/H-U-TV:", set_mem }, { "CG1/H-ACT2:", set_mem },
  { "CG1/1:", set_mem }, 
  { "CG1/2:", set_mem }, { "CG1/3:", set_mem }, { "CG1/4:", set_mem }, 
  { "CG1/5:", set_mem }, { "CG1/6:", set_mem }, { "CG1/7:", set_mem }, 
  { "CG1/8:", set_mem }, { "CG1/9:", set_mem }, { "CG1/10:", set_mem }, 
  { "CG1/11:", set_mem }, { "CG1/12:", set_mem }, { "CG1/13:", set_mem }, 
  { "CG1/14:", set_mem }, { "CG1/15:", set_mem }, { "CG1/16:", set_mem }, 
  { "CG1/17:", set_mem }, { "CG1/18:", set_mem }, { "CG1/19:", set_mem }, 
  { "CG1/20:", set_mem }, { "CG1/21:", set_mem }, { "CG1/22:", set_mem }, 
  { "CG1/23:", set_mem }, { "CG1/24:", set_mem }, { "CG1/25:", set_mem }, 
  { "CG1/26:", set_mem }, { "CG1/27:", set_mem }, { "CG1/28:", set_mem }, 
  { "CG1/29:", set_mem }, { "CG1/30:", set_mem }, { "CG1/31:", set_mem }, 
  { "CG1/32:", set_mem }, { "CG1/33:", set_mem }, { "CG1/34:", set_mem }, 
  { "CG1/35:", set_mem }, { "CG1/36:", set_mem }, { "CG1/37:", set_mem }, 
  { "CG1/38:", set_mem }, { "CG1/39:", set_mem }, { "CG1/40:", set_mem }, 
  { "CG1/41:", set_mem }, { "CG1/42:", set_mem }, { "CG1/43:", set_mem }, 
  { "CG1/44:", set_mem }, { "CG1/45:", set_mem }, { "CG1/46:", set_mem }, 
  { "CG1/47:", set_mem }, { "CG1/48:", set_mem }, { "CG1/49:", set_mem }, 
  { "CG1/50:", set_mem }, { "CG1/51:", set_mem }, { "CG1/52:", set_mem }, 
  { "CG1/L1:", set_mem }, { "CG1/U1:", set_mem }, { "CG1/L2:", set_mem }, 
  { "CG1/U2:", set_mem }, { "CG1/L3:", set_mem }, { "CG1/U3:", set_mem }, 
  { "CG1/L4:", set_mem }, { "CG1/U4:", set_mem }, { "CG1/L5:", set_mem }, 
  { "CG1/U5:", set_mem }, { "CG1/L6:", set_mem }, { "CG1/U6:", set_mem }, 
  { "CG1/L7:", set_mem }, { "CG1/U7:", set_mem }, { "CG1/L8:", set_mem }, 
  { "CG1/U8:", set_mem }, { "CG1/L9:", set_mem }, { "CG1/U9:", set_mem }, 
  { "CG1/L10:", set_mem }, { "CG1/U10:", set_mem },
  /* BC mem */
  { "BCBAND/H:", set_bc_mem }, { "BCBAND/1:", set_bc_mem },
  { "BCBAND/2:", set_bc_mem }, { "BCBAND/3:", set_bc_mem },
  { "BCBAND/4:", set_bc_mem }, { "BCBAND/5:", set_bc_mem }, 
  { "BCBAND/6:", set_bc_mem }, { "BCBAND/7:", set_bc_mem },
  { "BCBAND/8:", set_bc_mem }, { "BCBAND/9:", set_bc_mem }, 
  /* Autodialer */
  { "dtmf-1:", set_ad }, { "dtmf-2:", set_ad }, { "dtmf-3:", set_ad }, 
  { "dtmf-4:", set_ad }, { "dtmf-5:", set_ad }, { "dtmf-6:", set_ad }, 
  { "dtmf-7:", set_ad }, { "dtmf-8:", set_ad }, 
  /* Has to end with pair of nulls. Scanning loop depends on this. */
  { 0, 0 }
};

int main (int argc, char *argv[])
{
  int ret, fd_temp, fd_out, i, matches, lineno = 0;
  FILE *fp_in;
  unsigned char cksum, cksum2;
  char inbuf[512], saved[512];
  char *cp, *new_inbuf, *tag;
  char *d1, *d2, *d3, *d4, *d5, *d6, *d7, *d8, *d9, *d10, *d11, *d12, *d13;
  char *d14;
 
  if (argc != 4) {
    printf ("Usage: mkbin <infile> <template> <outfile>\n");
    exit (1);
  }

  fp_in = fopen (argv[1], "r");
  if (! fp_in) {
    perror (argv[1]);
    exit (1);
  }

  fd_temp = open (argv[2], O_RDONLY);
  if (fd_temp < 0) {
    perror (argv[2]);
    exit (1);
  }

  /* Read template.  Basically every line from user modifies this data. */

  ret = read_vx1_save_file (fd_temp, b1, sizeof b1);
  if (ret != 0) {
    perror ("read_vx1_save_file");
    exit (1);
  }

  cksum = calculate_vx1_cksum (b1, sizeof b1 - 1);
  cksum2 = calculate_vx1_cksum_2 (b1, 0xa0);

  if ((cksum != b1[sizeof b1 - 1]) || (cksum2 != b1[0xa0])) {
    printf ("Checksum error!\n");
    printf ("Calculated checksums = %02x, %02x; file cheksums = %02x, %02x\n",
	    cksum, cksum2, b1[sizeof b1 - 1], b1[0xa0]);
    printf ("Aborting.\n");
    exit (1);
  }

  /* Read data from user and act accordingly. */

  lineno = 0;
  while (fgets (inbuf, sizeof (inbuf), fp_in)) {

    /* Save buffer for later reference. */
    (void) strcpy (saved, inbuf);
    lineno ++;

    /* Trim leading .. */
    new_inbuf = inbuf;
    for (cp = inbuf; *cp; cp ++)
      if (isspace (*cp))
	new_inbuf = cp + 1;
      else
	break;
    
    /* .. and trailing white space. */
    for (cp = new_inbuf; *cp; cp ++) ;
    for (cp --; cp > new_inbuf; cp --)
      if (isspace (*cp))
	*cp = 0;
      else
	break;
    
    /* Strip comments. */
    for (cp = new_inbuf; *cp; cp ++)
      if (*cp == '#') {
	*cp = 0;
	break;
      }
    
    /* Skip empty lines. */
    if (*new_inbuf == 0)
      continue;
    
    /* At this point, we should have valid input.  Modify template data
       accordingly. */
    if (debug)
      printf ("%5d %s\n", lineno, new_inbuf);

    /* Separate tag and data. */
    tag = strtok (new_inbuf, " \t\n");
    d1 = strtok (NULL, " \t\n"); d2 = strtok (NULL, " \t\n");
    d3 = strtok (NULL, " \t\n"); d4 = strtok (NULL, " \t\n");
    d5 = strtok (NULL, " \t\n"); d6 = strtok (NULL, " \t\n");
    d7 = strtok (NULL, " \t\n"); d8 = strtok (NULL, " \t\n");
    d9 = strtok (NULL, " \t\n"); d10 = strtok (NULL, " \t\n");
    d11 = strtok (NULL, " \t\n"); d12 = strtok (NULL, " \t\n");
    d13 = strtok (NULL, " \t\n"); d14 = strtok (NULL, " \t\n");

    if (! tag) {
      printf ("Ignored: No valid tag on line %d >%s", lineno, saved);
      continue;
    }

    /* Go thru tag table and try to find a match. */
    for (i = 0, matches = 0; tags[i].tag; i ++) {
      if (strcasecmp (tags[i].tag, tag) == 0) {
	/* match. */
	matches ++;
	ret = (*tags[i].func)(tag, d1, d2, d3, d4, d5, d6, d7, d8, d9, d10,
			      d11, d12, d13, d14);
	if (ret != 0) {
	  printf ("Problem with line %d >%s",
		  lineno, saved);
	  switch (ret) {
	  case -1:
	    printf ("Unspecified error.\n");
	    break;
	  case -2:
	    printf ("Wrong number of arguments in line.\n");
	    break;
	  case -10:
	    printf ("Problem with tag (%s).\n", tag);
	    break;
	  case -11:
	  case -12:
	  case -13:
	  case -14:
	  case -15:
	  case -16:
	  case -17:
	  case -18:
	  case -19:
	  case -20:
	  case -21:
	  case -22:
	  case -23:
	  case -24:
	    printf ("Error in argument %d.\n", -10 - ret);
	    break;
	  default:
	    printf ("Unknown error.\n");
	    break;
	  }
	  printf ("Aborting..\n");
	  exit (1);
	}
      }
    }
    switch (matches) {
    case 0:
      printf ("Ignored: Tag invalid, line %d >%s", lineno, saved);
      break;
    case 1:
      if (debug)
	printf ("Line %d processed ok\n", lineno);
      break;
    default:
      printf ("Warning: Processed more than once (%d times), line %d >%s",
	      lineno, matches, saved);
      break;
    }
    
    if ((debug) && (b1[0x0002] != b1[0x0302]))
      printf ("XXX 0x0002 and 0x0302 differ: %02x, %02x\n",
	      b1[0x0002], b1[0x0302]);

  }

  fclose (fp_in);

  /* Save template as modified per user data. */
  
  fd_out = open (argv[3], O_WRONLY | O_TRUNC | O_CREAT, 0666);
  if (fd_out < 0) {
    perror (argv[1]);
    printf ("Failed to open savefile %s. Aborting.\n", argv[3]);
    exit (1);
  }

  b1[0xa0] = calculate_vx1_cksum_2 (b1, 0xa0);
  b1[sizeof b1 - 1] = calculate_vx1_cksum (b1, sizeof b1 - 1);

  ret = write_vx1_save_file (fd_out, b1, sizeof b1);
  if (ret != 0) {
    perror ("write_save_file");
    exit (1);
  }

  close (fd_out);

  exit (0);
  
}
