
/* mktext.c - generate text version from radio data file.

   Copyright (c) 1997 Riku Kalinen, OH2LWO, oh2lwo@sral.fi

   RCS $Id: vx1mktext.c,v 1.3 1998/02/03 11:25:05 riku Exp $ */

static char rcs_revision[] =
"$Revision: 1.3 $";

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "yaesu.h"

/* Print lots of hexdump.  Controlled by -d - option. */

int debug = 0;

static unsigned char bitinx (unsigned char uc)
{
  switch (uc) {
  case 0x01: return 0;
  case 0x02: return 1;
  case 0x04: return 2;
  case 0x08: return 3;
  case 0x10: return 4;
  case 0x20: return 5;
  case 0x40: return 6;
  case 0x80: return 7;
  default: return 0xff;
  }
}

/* Dump flags for one band. */

static void flagdump (unsigned char *b, int cg, int n_channels)
{
  int i;
  int off;
  int bit;
  int output;
  unsigned char mask;
  char head[64];

  sprintf (head, "CG%d/%s", cg,
	   vx1_band_string ((unsigned char) (bitinx (b[0]) + 1)));
  for (i = 1, output = 0; i < n_channels; i ++) {
    off = (i - 1) / 8;
    bit = (i - 1) % 8;
    mask = 0x01 << bit;
    if (b[off + 1] & mask) {
      if ((output % 16) == 0)
	printf ("%10s: ", head);
      if (cg == 1)
	printf ("%3s ", vx1_chan_cg1_string ((unsigned char) (i + 7)));
      else
	printf ("%3s ", vx1_chan_cg2_string ((unsigned char) (i + 7)));
      output ++;
      if ((output % 16) == 0)
	printf ("\n");
    }
  }
  if ((output % 16) != 0)
    printf ("\n");

}

/* Dump AD memory. */

static void addump (unsigned char *b, char *s)
{
  unsigned char *cp;
  char c;
  int j, limit;

  printf ("%10s: ", s);
  limit = 16;
  for (j = 0, cp = b; j < limit; j ++, cp ++) {
    if (*cp == 0x3d)
      break;
    switch (*cp) {
    case 0x00: c = '0'; break;
    case 0x01: c = '1'; break;
    case 0x02: c = '2'; break;
    case 0x03: c = '3'; break;
    case 0x04: c = '4'; break;
    case 0x05: c = '5'; break;
    case 0x06: c = '6'; break;
    case 0x07: c = '7'; break;
    case 0x08: c = '8'; break;
    case 0x09: c = '9'; break;
    case 0x0a: c = 'A'; break;
    case 0x0b: c = 'B'; break;
    case 0x0c: c = 'C'; break;
    case 0x0d: c = 'D'; break;
    case 0x0e: c = '*'; break;
    case 0x0f: c = '#'; break;
    case 0x28: c = '-'; break;
    default:   c = '?'; break;
    }
    putchar (c);
  }
  putchar ('\n');

}

/* Dump one channel slot. */

static void chanhead8 (void)
{
  printf ("#         :    Rx Name\n");
}

static void chandump8 (unsigned char *b, unsigned char c, char *s)
{
  printf ("%10s: ", s);

  if (b[0] == 0xff) {
    /* unused channel marker */
    printf ("XX\n");
    return;
  }

  /* Valid data at this point */
  printf ("%s ", vx1_bcfreq_string (b));

  printf ("%s ", vx1_text_string (b + 2));

  if (debug) {
    printf (" "); hexdump (b, 2);
  } else {
    printf ("\n");
  }

  return;
    
}

static char *band_string (unsigned char *c, int i)
{
  static char retval[32];
  int off, bit;
  unsigned char mask;
  int j;

  strcpy (retval, "Masked");
  
  off = (i - 8) / 8;
  bit = (i - 8) % 8;
  mask = 0x01 << bit;
  for (j = 0; j < 8; j ++) {
    if (c [j*32+off+1] & mask) {
      sprintf (retval, "%6.6s",
	       vx1_band_string ((unsigned char) (bitinx (c[j*32]) + 1)));
    }
  }

  return retval;
}

static void chanhead12 (void)
{
  printf ("#         :   Band      Rx S F/A Mode Rp Sel Po C Step Name\n");
}

static void chandump12 (unsigned char *b, unsigned char *c, char *s, int inx)
{
  unsigned char tuc[16];
  
  printf ("%10s: ", s);

  if (b[0] == 0xff) {
    /* unused channel marker */
    printf ("XX\n");
    return;
  }

  /* Dump band as well if possible. */
  if (inx > 7)
    printf ("%6.6s ", band_string (c, inx));
  else
    printf ("%6.6s ", "-");

  /* Valid data at this point */
  printf ("%s ", vx1_freq_string (b));

  printf ("%s ", vx1_skip_string ((b[4] >> 7) & 0x01));
  printf ("%s ", vx1_nameflag_string ((b[4] >> 6) & 0x01));
  printf ("%s ", vx1_mode_string ((b[4] & 0x30) >> 4));
  printf ("%s ", shift_string ((b[4] >> 2) & 0x03));
  printf ("%s ", selcal_string (b[4] & 0x03));

  printf ("%s ", vx1_txpwr_string ((b[5] >> 4) & 0x01));
  printf ("%s ", vx1_clksft_string ((b[5] >> 3) & 0x01));
  printf ("%s ", vx1_step_string (b[5] & 0x07));

  printf ("%s ", vx1_text_string (b + 6));

  if (debug) {
    tuc[0] = b[3] & 0xff;
    tuc[1] = b[4] & 0x00;
    tuc[2] = b[5] & 0xe0;
    printf (" "); hexdump (tuc, 3);
  } else {
    printf ("\n");
  }

  return;
    
}

static void chanhead16 (void)
{
  printf ("#         :   Band      Rx  Tx/off S F/A Mode Rp Sel Po C Step  Tone DCS Name\n");
}

static void chandump16 (unsigned char *b, unsigned char *c, char *s, int inx)
{
  unsigned char tuc[16];
  
  printf ("%10s: ", s);

  if (b[0] == 0xff) {
    /* unused channel marker */
    printf ("XX\n");
    return;
  }

  /* Dump band as well if possible. */
  if (inx > 7)
    printf ("%6.6s ", band_string (c, inx));
  else
    printf ("%6.6s ", "-");

  /* Valid data at this point */
  printf ("%s ", vx1_freq_string (b));
  printf ("%s ", vx1_freq_string (b + 3));

  printf ("%s ", vx1_skip_string ((b[6] >> 7) & 0x01));
  printf ("%s ", vx1_nameflag_string ((b[6] >> 6) & 0x01));
  printf ("%s ", vx1_mode_string ((b[6] & 0x30) >> 4));
  printf ("%s ", shift_string ((b[6] >> 2) & 0x03));
  printf ("%s ", selcal_string (b[6] & 0x03));

  printf ("%s ", vx1_txpwr_string ((b[7] >> 4) & 0x01));
  printf ("%s ", vx1_clksft_string ((b[7] >> 3) & 0x01));
  printf ("%s ", vx1_step_string (b[7] & 0x07));

  printf ("%s ", ctcss_string (b[8] & 0x3f));

  printf ("%s ", dcs_string (b[9]));

  printf ("%s ", vx1_text_string (b + 10));

  if (debug) {
    tuc[0] = b[6] & 0x00;
    tuc[1] = b[7] & 0xe0;
    tuc[2] = b[8] & 0xc0;
    tuc[3] = b[9] & 0x00;
    printf (" "); hexdump (tuc, 4);
  } else {
    printf ("\n");
  }

  return;
    
}

static void dump_b1 (unsigned char *b, int s,
	      unsigned char cksum, unsigned char cksum2)
{
  int i;
  char str[64];
  int cg;

  if (b[0x000b] & 0x04)
    cg = 1;
  else
    cg = 2;

  /* known data */

  /* Band, VOL, SQL */

  printf ("# Miscellaneous data\n");
  printf ("      BAND: %s\n", vx1_band_string (b[0x2]));
  if (b[0x2] != b[0x302]) {
    printf ("# WARNING: bytes 0x2 and 0x302 mismatch (%02x != %02x)\n",
	    b[0x2], b[0x302]);
  }
  printf ("    BSYLED: %s\n", onoff_string ((b[0x7] >> 6) & 0x01));
  printf ("      BCLO: %s\n", onoff_string ((b[0x7] >> 5) & 0x01));
  printf ("      BEEP: %s\n", onoff_string ((b[0x7] >> 4) & 0x01));
  printf ("      SCNL: %s\n", onoff_string ((b[0x7] >> 3) & 0x01));
  printf ("    RESUME: %s\n", vx1_resume_string ((b[0x7] >> 2) & 0x01));
  printf ("      ATMD: %s\n", vx1_atmd_string ((b[0x7] >> 1) & 0x01));
  printf ("       ARS: %s\n", onoff_string (b[0x7] & 0x01));
  if (b[0x07] != b[0x307]) {
    printf ("# WARNING: bytes 0x7 and 0x307 mismatch (%02x != %02x)\n",
	    b[0x07], b[0x307]);
  }
  printf ("      LAMP: %s\n", vx1_lamp_string ((b[0x8] >> 6) & 0x03));
  printf ("       APO: %s\n", apo_string ((b[0x8] >> 3) & 0x07));
  printf ("    RXSAVE: %s\n", rsav_string (b[0x8] & 0x07));
  if (b[0x08] != b[0x308]) {
    printf ("# WARNING: bytes 0x8 and 0x308 mismatch (%02x != %02x)\n",
	    b[0x08], b[0x308]);
  }
  printf ("        LK: %s\n", onoff_string ((b[0x9] >> 7) & 0x01));
  printf ("      LOCK: %s\n", lock_string ((b[0x9] >> 4) & 0x07));
  printf ("      BELL: %s\n", bell_string (b[0x9] & 0x07));
  if (b[0x09] != b[0x309]) {
    printf ("# WARNING: bytes 0x9 and 0x309 mismatch (%02x != %02x)\n",
	    b[0x09], b[0x309]);
  }
  printf ("    ARTSBP: %s\n", arts_beep_string ((b[0xa] >> 6) & 0x03));
  printf ("     SMTMD: %s\n", vx1_smtmd_string ((b[0xa] >> 5) & 0x01));
  printf ("       TOT: %s\n", vx1_tot_string ((b[0xa] >> 2) & 0x07));
  printf ("     DIALM: %s\n", vx1_dialm_string ((b[0xa] >> 1) & 0x01));
  printf ("     MONTC: %s\n", vx1_montc_string (b[0xa] & 0x01));
  if (b[0x0a] != b[0x30a]) {
    printf ("# WARNING: bytes 0xa and 0x30a mismatch (%02x != %02x)\n",
	    b[0x0a], b[0x30a]);
  }
  printf ("       GRP: %s\n", vx1_group_string ((b[0xb] >> 2) & 0x01));
  printf ("      CWID: %s\n", onoff_string ((b[0xb] >> 1) & 0x01));
  if (b[0x0b] != b[0x30b]) {
    printf ("# WARNING: bytes 0xb and 0x30b mismatch (%02x != %02x)\n",
	    b[0x0b], b[0x30b]);
  }
  printf ("     DTMFM: %s\n", vx1_dtmfm_string (b[0x0e] & 0x0f));
  if (b[0x0e] != b[0x30e]) {
    printf ("# WARNING: bytes 0xe and 0x30e mismatch (%02x != %02x)\n",
	    b[0x0e], b[0x30e]);
  }
  printf ("       VOL: %s\n", vx1_vol_string (b[0x0f]));
  if (b[0x0f] != b[0x30f]) {
    printf ("# WARNING: bytes 0xf and 0x30f mismatch (%02x != %02x)\n",
	    b[0x0f], b[0x30f]);
  }
  printf ("       SQL: %s\n", vx1_sql_string (b[0x10]));
  if (b[0x10] != b[0x310]) {
    printf ("# WARNING: bytes 0x10 and 0x310 mismatch (%02x != %02x)\n",
	    b[0x10], b[0x310]);
  }
  printf ("\n");
  
  printf ("# Current settings\n");
  chanhead16 ();
  chandump16 (b + 0x11, 0, "CURR", 0);
  printf ("\n");

  /* Second checksum */

  if (cksum2 != b[0x00a0])
    printf ("# WARNING: Second checksum mismatch (calculated %02x, file %02x)\n",
	    cksum2, b[0x00a0]);
  else
    printf ("# Second checksum is %02x (matches)\n", cksum2);
  printf ("\n");

  /* Smart Search memories */

  printf ("# Smart Search memories\n");
  chanhead16 ();
  for (i = 0; i < 31; i ++) {
    sprintf (str, "SS/%d", i - 15);
    chandump16 (b + 0x0f1 + i * 16, 0, str, 0);
  }
  printf ("\n");

  /* Call sign for ID */

  printf ("# Call sign for CW ID\n");
  printf ("        ID: %s\n", vx1_id_string (b + 0x02f1));
  printf ("\n");

  /* BC bamd VFO */

  printf ("# BC band VFO\n");
  chanhead8 ();
  chandump8 (b + 0x0311, 0, "VFO/BCBAND");
  printf ("\n");
  
  /* VFOs */

  printf ("# VFOs\n");
  chanhead16 ();
  for (i = 0; i < 8; i ++) {
    sprintf (str, "VFO/%s", vx1_band_string (i + 1));
    chandump16 (b + 0x0321 + i * 16, 0, str, 0);
  }
  printf ("\n");

  /* Priority channel */
  printf ("# Priority channel\n");
  if (cg == 1)
    printf ("       PRI: %s\n", vx1_chan_cg1_string (b[0x03a1]));
  else
    printf ("       PRI: %s\n", vx1_chan_cg2_string (b[0x03a1]));
  printf ("\n");
  
  /* channel frequencies */

  printf ("# Channels, Configuration Group 2\n");
  chanhead12 ();
  for (i = 0; i < 170; i ++) {
    sprintf (str, "CG2/%s", vx1_chan_cg2_string (i));
    chandump12 (b + 0x03d1 + i * 12, b + 0x10d1, str, i);
  }
  printf ("\n");

  /* channel frequencies */

  printf ("# Channels, Configuration Group 1\n");
  chanhead16 ();
  for (i = 0; i < 80; i ++) {
    sprintf (str, "CG1/%s", vx1_chan_cg1_string (i));
    chandump16 (b + 0x0bd1 + i * 16, b + 0x10e7, str, i);
  }
  printf ("\n");

  if (debug) {
    
    /* Channel/band flags */
    
    printf ("# Channel/band flags, CG2\n");
    for (i = 0; i < 8; i++)
      flagdump (b + 0x10d1 + i * 32, 2, 163);
    printf ("\n");
    
    printf ("# Channel/band flags, CG1\n");
    for (i = 0; i < 8; i++)
      flagdump (b + 0x10e7 + i * 32, 1, 73);
    printf ("\n");

  }

  /* BC band memories */

  printf ("# BC band memories\n");
  chanhead8 ();
  for (i = 0; i < 10; i ++) {
    sprintf (str, "BCBAND/%s", vx1_chan_bc_string (i));
    chandump8 (b + 0x11f1 + i * 8, 0, str);
  }
  printf ("\n");

  /* Autodialers */

  printf ("# Autodialer memories\n");
  for (i = 0; i < 8; i ++)
    addump (b + 0x1251 + i * 16, vx1_dtmfm_string (i));
  printf ("\n");

  /* Checksum */

  if (cksum != b[s - 1])
    printf ("# WARNING: Checksum mismatch (calculated %02x, file %02x)\n",
	    cksum, b[s - 1]);
  else
    printf ("# Checksum is %02x (matches)\n", cksum);
  printf ("\n");

  /* unknown data only if debugging */

  if (debug) {

    printf ("0000: "); hexdump_no_n (b + 0x0000, 8); printf ("   ");
    hexdump_no_n (b + 0x0009, 1); printf ("   "); hexdump (b + 0x000b, 4);
    printf ("0020:    "); hexdump (b + 0x0021, 15);
    for (i = 0x0030; i < 0x00a0; i += 0x10) { 
      printf ("%04x: ", i);
      hexdump (b + i, 16);
    }
    printf ("00a0:    "); hexdump (b + 0x00a1, 15);
    printf ("00b0: "); hexdump (b + 0x00b0, 16);
    printf ("00c0: "); hexdump (b + 0x00b0, 16);
    printf ("00d0: "); hexdump (b + 0x00b0, 16);
    printf ("00e0: "); hexdump (b + 0x00b0, 16);
    printf ("00f0: "); hexdump (b + 0x00f0, 1);
    printf ("\n");
    printf ("02e0:    "); hexdump (b + 0x02e1, 15);
    printf ("02f0: "); hexdump (b + 0x02f0, 16);
    printf ("0300: "); hexdump_no_n (b + 0x0300, 8); printf ("   ");
    hexdump (b + 0x0309, 6);
    printf ("\n");
    printf ("0319: "); hexdump (b + 0x0319, 8);
    printf ("\n");
    printf ("03a0:       "); hexdump (b + 0x03a2, 14);
    for (i = 0x03b0; i < 0x03d0; i += 0x10) { 
      printf ("%04x: ", i);
      hexdump (b + i, 16);
    }
    printf ("03d0: "); hexdump (b + 0x03d0, 1);
    printf ("\n");
    printf ("0bc9: "); hexdump (b + 0x0bc9, 8);
    printf ("\n");
    printf ("11d0:    "); hexdump (b + 0x11d1, 15);
    printf ("11e0: "); hexdump (b + 0x11e0, 16);
    printf ("11f0: "); hexdump (b + 0x11f0, 1);
    printf ("\n");
    printf ("1240:    "); hexdump (b + 0x1241, 15);
    printf ("1250: "); hexdump (b + 0x1250, 1);
  }

}

int main (int argc, char *argv[])
{
  unsigned char b1[4818];
  int ret, fd, c;
  unsigned char cksum, cksum2;

  while ((c = getopt (argc, argv, "d")) != -1)
    switch (c) {
    case 'd': debug ++; break;
    case '?': printf ("Usage: mktext [-d] <infile>\n"); exit (1); break;
    }
 
  if ((argc - optind) != 1) {
    printf ("Usage: mktext [-d] <infile>\n");
    exit (1);
  }

  fd = open (argv[optind], O_RDONLY);
  if (fd < 0) {
    perror (argv[optind]);
    exit (1);
  }

  ret = read_vx1_save_file (fd, b1, sizeof b1);
  if (ret != 0) {
    perror ("read_save_file");
    exit (1);
  }

  close (fd);

  cksum = calculate_vx1_cksum (b1, sizeof b1 - 1);
  cksum2 = calculate_vx1_cksum_2 (b1, 0xa0);

  printf ("# Dump of file %s (vx1mktext %s)\n\n", argv[optind], rcs_revision);

  dump_b1 (b1, sizeof (b1), cksum, cksum2);

  exit (0);
  
}
