/******************************************************************************/
/*                                                                            */
/*  Title       : makepassesx.c                                               */
/*  Author      : Manfred Bester                                              */
/*  Date        : 24Oct94                                                     */
/*  Last change : 13Mar95                                                     */
/*                                                                            */
/*  Synopsis    : This program reads a satlist file and creates a C shell     */
/*                script to run SatTrack in the batch mode for all satellites */
/*                in the satlist file.                                        */
/*                                                                            */
/*  Usage       : makepassesx [group]                                         */
/*                                                                            */
/*  Input files : satlist_nnn.dat, batch.dat                                  */
/*  Output file : passesx                                                     */
/*                                                                            */
/*                                                                            */
/*  SatTrack is Copyright (c) 1992, 1993, 1994, 1995 by Manfred Bester.       */
/*  All Rights Reserved.                                                      */
/*                                                                            */
/*  Permission to use, copy, and distribute SatTrack and its documentation    */
/*  in its entirety for educational, research and non-profit purposes,        */
/*  without fee, and without a written agreement is hereby granted, provided  */
/*  that the above copyright notice and the following three paragraphs appear */
/*  in all copies. SatTrack may be modified for personal purposes, but        */
/*  modified versions may NOT be distributed without prior consent of the     */
/*  author.                                                                   */
/*                                                                            */
/*  Permission to incorporate this software into commercial products may be   */
/*  obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,    */
/*  Berkeley, CA 94709, USA. Note that distributing SatTrack 'bundled' in     */
/*  with ANY product is considered to be a 'commercial purpose'.              */
/*                                                                            */
/*  IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, */
/*  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF   */
/*  THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED  */
/*  OF THE POSSIBILITY OF SUCH DAMAGE.                                        */
/*                                                                            */
/*  THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT      */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A   */
/*  PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"      */
/*  BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, */
/*  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                  */
/*                                                                            */
/******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef STDLIB
#include <stdlib.h>
#endif

#include "sattrack.h"

#ifdef HPTERM
#include "hpterm.h"
#else
#include "vt100.h"
#endif

/******************************************************************************/
/*                                                                            */
/* definitions of default command line arguments                              */
/* these will be used if a particular argument is unspecified in 'batch.dat'  */
/*                                                                            */
/******************************************************************************/

#define  SATTRACK     "sattrack"
#define  ARGUMENT     "-b"
#define  TIMEZONEX    "PDT"
#define  SITENAME     "Mt._Wilson"
#define  TLEFILE      "tlex.dat"
#define  TLETYPE      "NASA"
#define  PREDTYPE     "shortpr"
#define  STARTDATE    "now"
#define  STARTTIME    "now"
#define  TIMESTEP     "auto"
#define  DURATION     "7"
#define  MINELEV      "0"
#define  HARDCOPYX    "nohardcopy"

/******************************************************************************/
/*                                                                            */
/* external functions                                                         */
/*                                                                            */
/******************************************************************************/

extern void clipString(), fillBlanks(), mGets(), truncBlanks(), upperCase();

/******************************************************************************/
/*                                                                            */
/* main program                                                               */
/*                                                                            */
/******************************************************************************/

void main(argc,argv)

int  argc;
char *argv[];

{
    char satListName[100], sysComm[120], data[100], run[100], satGroup[10];
    char batchStr[120], inputSatFile[100], outputFile[100], resourceFile[100];
    char resourceLine[120];
    char arg0[40], arg1[40], arg2[40], arg3[40], arg4[40], arg5[40], arg6[40];
    char arg7[40], arg8[40], arg9[40], arg10[40], arg11[40], arg12[40];
    char arg13[40];
    char *strpHome, *getenv();

    FILE *InputSatFile, *OutputFile, *ResourceFile;

/******************************************************************************/
/*                                                                            */
/* check home directory                                                       */
/*                                                                            */
/******************************************************************************/

#ifdef HOMEDIR
    strpHome = getenv("HOME");
#else
    strpHome = SATDIR;
#endif

/******************************************************************************/
/*                                                                            */
/* check command line arguments and make file names                           */
/*                                                                            */
/******************************************************************************/

    if (argc == 2)
    {
        strcpy(satGroup,argv[1]);
    }

    else
    {
        printf("\nenter satellite group (am,com,gps,mir,sci,tv,wx,etc.) : ");
        mGets(satGroup);
    }

    sprintf(data,"%s/%s",strpHome,DATA);
    sprintf(run,"%s/%s",strpHome,RUN);

    sprintf(resourceFile,"%s/batch.dat",data);
    sprintf(inputSatFile,"%s/satlist_%s.dat",data,satGroup);
    sprintf(outputFile,"%s/passesx",run);

/******************************************************************************/
/*                                                                            */
/* check if files are accessible                                              */
/*                                                                            */
/******************************************************************************/

    if ((ResourceFile = fopen(resourceFile,"r")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("%s not found\n",resourceFile);
        normal(); nl();
        exit(-1);
    }

    if ((InputSatFile = fopen(inputSatFile,"r")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("%s not found\n",inputSatFile);
        normal(); nl();
        exit(-1);
    }

    if ((OutputFile = fopen(outputFile,"w")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("can't write %s\n",outputFile);
        normal(); nl();
        exit(-1);
    }

/******************************************************************************/
/*                                                                            */
/* load default parameters                                                    */
/*                                                                            */
/******************************************************************************/

    sprintf(arg0, "%s",SATTRACK);
    sprintf(arg1, "%s",ARGUMENT);
    sprintf(arg2, "%s",TIMEZONEX);
    sprintf(arg3, "%s",SITENAME);

    sprintf(arg5, "%s",TLEFILE);
    sprintf(arg6, "%s",TLETYPE);
    sprintf(arg7, "%s",PREDTYPE);
    sprintf(arg8, "%s",STARTDATE);
    sprintf(arg9, "%s",STARTTIME);
    sprintf(arg10,"%s",TIMESTEP);
    sprintf(arg11,"%s",DURATION);
    sprintf(arg12,"%s",MINELEV);
    sprintf(arg13,"%s",HARDCOPYX);

/******************************************************************************/
/*                                                                            */
/* read resource file 'batch.dat'                                             */
/*                                                                            */
/******************************************************************************/

    while (fgets(resourceLine,100,ResourceFile))
    {
        resourceLine[strlen(resourceLine) - 1] = '\0';
        truncBlanks(resourceLine);

        sscanf(resourceLine,"Time zone: %s",arg2);

        if (!strncmp(resourceLine,"Site:",5))
        {
            clipString(resourceLine,5);
            sprintf(arg3,"%s",resourceLine);
            upperCase(arg3);
            fillBlanks(arg3);
        }

        sscanf(resourceLine,"Element set TLE: %s",arg5);
        sscanf(resourceLine,"Element set type: %s",arg6);
        sscanf(resourceLine,"Prediction type: %s",arg7);
        sscanf(resourceLine,"Start date: %s",arg8);
        sscanf(resourceLine,"Start time: %s",arg9);
        sscanf(resourceLine,"Time step: %s",arg10);
        sscanf(resourceLine,"Duration: %s",arg11);
        sscanf(resourceLine,"Min elevation: %s",arg12);
        sscanf(resourceLine,"Hardcopy: %s",arg13);
    }

    fclose(ResourceFile);

/******************************************************************************/
/*                                                                            */
/* create the C shell script 'passesx'                                        */
/*                                                                            */
/******************************************************************************/

    fprintf(OutputFile,"#! /bin/csh\n");
    fprintf(OutputFile,"#\n");
    fprintf(OutputFile,"# passesx\n");
    fprintf(OutputFile,"#\n");
    fprintf(OutputFile,"# ATTENTION: this C shell script is subject to ");
    fprintf(OutputFile,"automatic regeneration by 'makepassesx'.\n");
    fprintf(OutputFile,"#\n");

    while (fgets(satListName,80,InputSatFile))
    {
        satListName[strlen(satListName)-1] = '\0';
        upperCase(satListName);
        fillBlanks(satListName);
        sprintf(arg4,"%s",satListName);

        sprintf(batchStr,"%s  %s",arg0,arg1);
        sprintf(batchStr,"%s  %s",batchStr,arg2);
        sprintf(batchStr,"%s  %s",batchStr,arg3);
        sprintf(batchStr,"%s  %-11s",batchStr,arg4);
        sprintf(batchStr,"%s  %s",batchStr,arg5);
        sprintf(batchStr,"%s  %s",batchStr,arg6);
        sprintf(batchStr,"%s  %s",batchStr,arg7);
        sprintf(batchStr,"%s  %s",batchStr,arg8);
        sprintf(batchStr,"%s  %s",batchStr,arg9);
        sprintf(batchStr,"%s  %s",batchStr,arg10);
        sprintf(batchStr,"%s  %s",batchStr,arg11);
        sprintf(batchStr,"%s  %s",batchStr,arg12);
        sprintf(batchStr,"%s  %s",batchStr,arg13);

        fprintf(OutputFile,"  %s\n",batchStr);
    }

    fclose(InputSatFile);
    fclose(OutputFile);

    sprintf(sysComm,"chmod 755 %s",outputFile);
    system(sysComm);
}

/******************************************************************************/
/*                                                                            */
/* clipString: clips N characters plus leading blanks off the beginning of a  */
/*             character string                                               */
/*                                                                            */
/******************************************************************************/

void clipString(charStr,N)

int  N;
char *charStr;

{
    int  i, j, firstChar;
    char tmpStr[100];

    firstChar = FALSE;
    j = 0;

    for (i = 0; i < strlen(charStr) - N; i++) 
    {

       if (charStr[i+N] != ' ' || (charStr[i+N] == ' ' && firstChar))
       {
           tmpStr[j] = charStr[i+N];
           j++;
           firstChar = TRUE;
       }
    }

    tmpStr[j] = '\0';                            /* add termination character */
    sprintf(charStr,"%s",tmpStr);
    return;
}

/******************************************************************************/
/*                                                                            */
/* fillBlanks: replaces blank characters by underscore characters             */
/*                                                                            */
/******************************************************************************/

void fillBlanks(charStr)

char *charStr;

{
    int  i;

    for (i = 0; i < strlen(charStr); i++)
    {
        if (charStr[i] == ' ')
            charStr[i] = '_';
    }

    return;
}

/******************************************************************************/
/*                                                                            */
/* truncBlanks: truncates trailing blanks from character string               */
/*                                                                            */
/******************************************************************************/

void truncBlanks(string)

char *string;

{
    int i;

    i = strlen(string) - 1;

    while (string[i] == ' ')
    {
        i--;
    }

    string[i+1] = '\0';                          /* add termination character */
    return;
}

/******************************************************************************/
/*                                                                            */
/* upperCase: changes lower to upper case letters                             */
/*                                                                            */
/******************************************************************************/

void upperCase(string)

char *string;

{
    int i;

    for (i = 0; i < strlen(string); i++)
    {
        if (string[i] >= 'a' && string[i] <= 'z')
            string[i] -= 'a' - 'A';
    }

    return;
}

/******************************************************************************/
/*                                                                            */
/* mGets: Manfred's version of fgets (wipes out newline character)            */
/*                                                                            */
/******************************************************************************/

void mGets(string)

char *string;

{
    int i;
    fgets(string,80,stdin);
    i = (int) strlen(string);

    if (i > 0)
        string[i-1] = '\0';
    else
        string[0]   = '\0';

    return;
}

/******************************************************************************/
/*                                                                            */
/* End of program makepassesx.c                                               */
/*                                                                            */
/******************************************************************************/
