/*
 * fpacwd.c : FPAC WP daemon
 *
 * F1OAT 970831
 */

#include "wpdefs.h"

static cfg_t cfg;		/* FPAC configuration file */
static int listening_socket;
static char *wp_file = FPACWP;	/* Default file */

int wp_trace_flag = 0;
static int daemon = 1;
static int wp_passive = 0;

/* WP contexts are indexed by socket handle */

struct wp_context *context[NB_MAX_HANDLES];
struct wp_adjacent *wp_adjacent_list = NULL;

/************************************************************************************
* Prototypes
************************************************************************************/

static void rose_handler(int s);
static void do_cmd(int s, char *cmd);
static struct wp_adjacent *find_adjacent(rose_address *srose_addr);
static void vector_request(struct wp_adjacent *wpa);

/************************************************************************************
* Clients handler
************************************************************************************/

static int init_client(int client, struct sockaddr_rose *address)
{
	assert(context[client] == 0);
	context[client] = calloc(1, sizeof(*context[client]));
	if (!context[client]) {
		close(client);
		return -1;
	}
	context[client]->address = *address;
	
	if (strcmp("WP-0", ax2asc(&address->srose_call)) == 0) {
		context[client]->type = WP_SERVER;
	}
	else {
		context[client]->type = WP_USER;
	}
	
	RegisterEventHandler(client, rose_handler);
	RegisterEventAwaited(client, READ_EVENT);	
	return 0;
}

static void close_client(int client, int active)
{
	if (!active) {
		syslog(LOG_INFO, "Disconnected by client %s @ %s", ax2asc(&context[client]->address.srose_call), rose2asc(&context[client]->address.srose_addr));
	}
	else {
		syslog(LOG_INFO, "Disconnecting client %s @ %s", ax2asc(&context[client]->address.srose_call), rose2asc(&context[client]->address.srose_addr));
	}
	
	if (context[client]->dirty_list) free(context[client]->dirty_list);
	close(client);
	UnRegisterEventAwaited(client, READ_EVENT);
	UnRegisterEventAwaited(client, WRITE_EVENT);
	if (context[client]->type == WP_SERVER && context[client]->adjacent) {
		struct wp_adjacent *wpa = context[client]->adjacent;
		wpa->state = WPA_DISCONNECTED;
		wpa->context = -1;
		wpa->retry_connect_date = time(NULL) + WPA_RETRY_CONNECT;
		context[client]->adjacent = NULL;
	}	
	
	free(context[client]);
	context[client] = NULL;
}


/************************************************************************************
* ROSE socket handler
************************************************************************************/

static void rose_write_handler(int s)
{
	int dirty;
	struct wp_adjacent *wpa;
	
	assert(context[s]);
	wpa = context[s]->adjacent;
	
	if (context[s]->adjacent && (context[s]->adjacent->state == WPA_CONNECTING)) {
		syslog(LOG_INFO, "Connected to adjacent %s", rose2asc(&context[s]->address.srose_addr));
		context[s]->adjacent->state = WPA_CONNECTED;
		context[s]->adjacent->vector_date = time(NULL);	
	}
	
	
	dirty = find_dirty_context(s);
	if (dirty < 0) {
		UnRegisterEventAwaited(s, WRITE_EVENT);
		if (wpa && wpa->vector_when_nodirty) {
			wpa->vector_when_nodirty = 0;
			vector_request(wpa);
		}
	}
	else {
		wp_pdu pdu;
		int rc;
		
		syslog(LOG_INFO, "Sending dirty record #%d %s to adjacent %s", 		 
			dirty,
			ax2asc(&db_records[dirty].address.srose_call),
			rose2asc(&context[s]->address.srose_addr));
		pdu.type = wp_type_set;
		pdu.data.wp = db_records[dirty];		
		rc = wp_send_pdu(s, &pdu);
		if (rc) {
			close_client(s, 1);
		}
		else {
			clear_dirty_context(dirty, s);
		}	
	}
}

static int wp_valid(wp_t *wp)
{
	char call[20];
		
	strcpy(call, ax2asc(&wp->address.srose_call));
	return !wp_check_call(call);
}

static void rose_read_handler(int s)
{
	int rc = 0;
	wp_pdu pdu;
	
	assert(context[s]);
	
	rc = wp_receive_pdu(s, &pdu);
	if (rc < 0) {	/* Connection lost or protocol error */
		close_client(s, 0);
		return;
	}
	
	switch (pdu.type) {
	case wp_type_set:
		/* For user client, check record validity */
		if (context[s]->type != WP_USER || wp_valid(&pdu.data.wp)) {		
			rc = db_set(&pdu.data.wp, context[s]->type != WP_USER);
		}
		else {
			rc = -1;
		}
		if (rc >= 0) {
			pdu.data.status = WP_OK;
			/* Broadcast this new record */
			broadcast_dirty(rc, s);
		}		
		else {
			pdu.data.status = WP_SET_ERROR;
		}
		
		if (context[s]->type == WP_USER) {	/* Answer only for standard clients */
			pdu.type = wp_type_response;
			rc = wp_send_pdu(s, &pdu);
		}
		else {
			syslog(LOG_INFO, "Receiving record %s from adjacent %s", 		 
				ax2asc(&pdu.data.wp.address.srose_call),
				rose2asc(&context[s]->address.srose_addr));			
			rc = 0;
		}
		break;
	case wp_type_get:
		rc = db_get(&pdu.data.call, &pdu.data.wp);
		if (wp_valid(&pdu.data.wp) && rc >= 0) {
			pdu.type = wp_type_get_response;
		}		
		else {
			pdu.data.status = WP_GET_ERROR;
			pdu.type = wp_type_response;
		}
		rc = wp_send_pdu(s, &pdu);
		break;
	case wp_type_ascii:
		do_cmd(s, pdu.data.string);
		break;
	case wp_type_vector_request:
		syslog(LOG_INFO, "Receiving vector request from %s", rose2asc(&context[s]->address.srose_addr));
		db_compute_vector(s, &pdu.data.vector);
		pdu.type = wp_type_vector_response;
		syslog(LOG_INFO, "Sending vector response to %s", rose2asc(&context[s]->address.srose_addr));
		rc = wp_send_pdu(s, &pdu);		
		break;
	case wp_type_vector_response:
		syslog(LOG_INFO, "Receiving vector response from %s", rose2asc(&context[s]->address.srose_addr));
		db_compute_vector(s, &pdu.data.vector);
		break;
	default:
		pdu.type = wp_type_response;
		pdu.data.status = WP_INVALID_COMMAND;
		rc = wp_send_pdu(s, &pdu);
		break;
	}
	
	if (rc) close_client(s, 1);
}

static void rose_handler(int s)
{
	if (GetEvent(s) & (1<<READ_EVENT)) rose_read_handler(s);
	else if (GetEvent(s) & (1<<WRITE_EVENT)) rose_write_handler(s);
}

static void listening_handler(int s)
{
	int new_client;
	struct sockaddr_rose address;
	int addrlen = sizeof(address);
	struct sockaddr_rose l_address;
	int l_addrlen = sizeof(l_address);
		
	new_client = accept(s, (struct sockaddr *)&address, &addrlen);
	if (new_client < 0) return;
	syslog(LOG_INFO, "New client %s @ %s", ax2asc(&address.srose_call), rose2asc(&address.srose_addr));
	
	if (init_client(new_client, &address)) return;
	
	if (context[new_client]->type == WP_SERVER) {
		struct wp_adjacent *wpa;
		wpa = find_adjacent(&address.srose_addr);
		if (wpa) {
			syslog(LOG_INFO, "New client is adjacent %s", wpa->node->name);
			switch (wpa->state) {
			case WPA_CONNECTED:
				syslog(LOG_INFO, "Already connected to adjacent %s", wpa->node->name);
				close_client(new_client, 1);
				break;
			case WPA_CONNECTING:
				assert(!getsockname(listening_socket, (struct sockaddr *)&l_address, &l_addrlen));
				/* Crossed connection requests : higher calling address win */
				if (memcmp(&l_address.srose_addr, &address.srose_addr, sizeof(address.srose_addr)) > 0) {
					syslog(LOG_INFO, "Already connecting to adjacent %s", wpa->node->name);
					close_client(new_client, 1);
					break;
				}
				else {
					close_client(wpa->context, 1);
				}
				/* Here, there is no break !!! */
			case WPA_DISCONNECTED:
				context[new_client]->adjacent = wpa;
				wpa->context = new_client;
				wpa->state = WPA_CONNECTED;
				wpa->vector_date = time(NULL) + WPA_VECTOR_PERIOD;
				RegisterEventAwaited(new_client, WRITE_EVENT);
				break;			
			}
		}
	}
}

static int init_rose(void)
{	
	listening_socket = wp_listen();
	if (listening_socket < 0) return -1;
	
	RegisterEventHandler(listening_socket, listening_handler);
	RegisterEventAwaited(listening_socket, READ_EVENT);
	return 0;	
}

/************************************************************************************
* Adjacents functions 
************************************************************************************/

static int init_adjacents(cfg_t *cfg)
{
	node_t *node;
	struct wp_adjacent *wpa;
		
	for (node=cfg->node; node; node=node->next) {
		if (node->nowp) continue;
		wpa = calloc(sizeof(*wpa), 1);
		if (!wpa) {
			syslog(LOG_ERR, "init_adjacents: Out of memory");
			return -1;
		}
		
		wpa->state = WPA_DISCONNECTED;
		wpa->node = node;
		wpa->context = -1;
		wpa->retry_connect_date = time(NULL);
		wpa->next = wp_adjacent_list;
		wp_adjacent_list = wpa;
	}
	
	return 0;
}

static struct wp_adjacent *find_adjacent(rose_address *srose_addr)
{
	char *fulladdr;
	char dnic[5], addr[7];
	struct wp_adjacent *wpa;
	
	fulladdr = rose2asc(srose_addr);
	strncpy(dnic, fulladdr, 4);
	dnic[4] = 0;
	strncpy(addr, fulladdr+4, 6);
	addr[6] = 0;

	for (wpa=wp_adjacent_list; wpa; wpa=wpa->next) {
		if (!strcmp(addr, wpa->node->addr) && !strcmp(dnic, wpa->node->dnic)) {
			return wpa;
		}
	}
	
	return NULL;	
}

static void connect_adjacent(struct wp_adjacent *wpa)
{
	int s;
	char addr[11];
	struct sockaddr_rose remote;

	strcpy(addr, wpa->node->dnic);
	strcat(addr, wpa->node->addr);
		
	syslog(LOG_INFO, "Trying to connect adjacent %s", addr);

	wpa->retry_connect_date = time(NULL) + 120;

	remote.srose_family = AF_ROSE;
	remote.srose_ndigis = 0;
	convert_call_entry("WP", remote.srose_call.ax25_call);
	convert_rose_address(addr, remote.srose_addr.rose_addr);
	s = wp_open_remote("WP", &remote, 1);	/* Non blocking mode */
	if (s < 0) {
		perror("wp_open_remote");
		return;
	}
	init_client(s, &remote);
	context[s]->adjacent = wpa;
	wpa->context = s;
	RegisterEventAwaited(s, WRITE_EVENT);
	wpa->state = WPA_CONNECTING;	
}

static void vector_request(struct wp_adjacent *wpa)
{
	wp_pdu pdu;
	vector_t vector;
	int s = wpa->context;
	int rc;
			
	wpa->vector_date = time(NULL) + WPA_VECTOR_PERIOD;
	pdu.type = wp_type_vector_request;

	vector.date_base = 0;
	vector.seed = random();
	db_compute_vector(-1, &vector);
	pdu.data.vector = vector;
	syslog(LOG_INFO, "Sending vector request to %s", rose2asc(&context[s]->address.srose_addr));
	rc = wp_send_pdu(s, &pdu);
	if (rc) {
		close_client(s, 1);
	}
}

static void poll_adjacents(void)
{
	struct wp_adjacent *wpa;
	time_t mytime = time(NULL);
	
	for (wpa=wp_adjacent_list; wpa; wpa=wpa->next) {
		switch (wpa->state) {
		case WPA_CONNECTED:
			if (mytime >= wpa->vector_date) {
				vector_request(wpa);
			}			
			break;
		case WPA_CONNECTING:
			break;
		case WPA_DISCONNECTED:
			if (mytime >= wpa->retry_connect_date) {
				if (!wp_passive) { /* Debug node is in passive mode */
					connect_adjacent(wpa);
				}
			}
			break; 
		}
	}

}

/************************************************************************************
* Debug dump handler 
************************************************************************************/

static void printf_pdu(int s, char *fmt, ...)
{
	wp_pdu pdu;
	va_list ap;
	
	va_start(ap, fmt);
	vsnprintf(pdu.data.string, sizeof(pdu.data.string), fmt, ap);
	pdu.type = wp_type_ascii;
	wp_send_pdu(s, &pdu);
	
	va_end(ap);	
}

static void record_dump(int index)
{
	printf(ax2asc(&db_records[index].address.srose_call));
}

static void debug_dump(int s)
{
	int i;
	struct wp_adjacent *wpa;
	
	printf_pdu(s, "WP Server version %s\r", WP_VERSION);
	
	printf_pdu(s, "=== Cient table dump ===\r");
	
	for (i=0; i<NB_MAX_HANDLES; i++) {
		if (!context[i]) continue;
		printf_pdu(s, "Client #%d %s @ %s type=%d #dirty=%d\r", 
			i, 
			ax2asc(&context[i]->address.srose_call),
			rose2asc(&context[i]->address.srose_addr),
			context[i]->type,
			count_dirty_context(i));
	}

	printf_pdu(s, "=== Adjacents table dump ===\r");

	for (wpa=wp_adjacent_list; wpa; wpa=wpa->next) {
		printf_pdu(s, "Node %s %s @ %s%s", 
			wpa->node->name, wpa->node->call, wpa->node->dnic, wpa->node->addr);
		switch (wpa->state) {
		case WPA_CONNECTED:
			printf_pdu(s, " connected #%d #dirty=%d\r",
				wpa->context,  count_dirty_context(wpa->context));
			break;
		case WPA_CONNECTING:
			printf_pdu(s, " connection in progress\r");
			break;
		case WPA_DISCONNECTED:
			printf_pdu(s, " disconnected, retry in %d s\r", 
				wpa->retry_connect_date - time(NULL));
			break; 
		}
	}

	printf_pdu(s, "=== End of dump     ===\r");
}

static void do_cmd(int s, char *cmd)
{
	debug_dump(s);
}
 
/************************************************************************************
* Main entry point 
************************************************************************************/

static void Usage(void)
{
	fprintf(stderr, "Usage : fpacwpd [-h] [-d] [-x] [-f wpfile]\n");
	fprintf(stderr, "-h : display this message\n");
	fprintf(stderr, "-d : start in foreground mode\n");
	fprintf(stderr, "-x : turn on debug mode\n");
	fprintf(stderr, "-p : turn on passive mode\n");
	fprintf(stderr, "-f wpfile : specify another wp file (default /var/ax25/fpac/fpacwp.dat)\n");
	exit(1);
}

static void process_options(int argc, char *argv[])
{
	int c;
	
	do {
		c = getopt(argc, argv, "hdxpf:");
		switch (c) {
		case 'h':
		case '?':
			Usage();
			break;
		case 'd':
			fprintf(stderr, "Foreground mode\n");
			daemon = 0;
			break;
		case 'x':
			fprintf(stderr, "Debug mode\n");
			wp_debug = 1;
			break;
		case 'p':
			fprintf(stderr, "Passive mode\n");
			wp_passive = 1;
			break;		
		case 'f':
			wp_file = optarg;
			fprintf(stderr, "Using file %s\n", wp_file);
			break;
		}
	} while (c != EOF);
}

int main(int argc, char *argv[]) 
{
	int rc;
	
	process_options(argc, argv);
	
	openlog("fpacwpd", LOG_CONS | LOG_PERROR | LOG_PID, LOG_USER);	
	syslog(LOG_WARNING, "Starting");
	srand(time(NULL));
	
	if (cfg_open(&cfg) != 0) {
		perror("Error in configuration reading");
		exit(1);
	}
	
	if (init_adjacents(&cfg)) {
		exit(1);
	}
		
	rc = init_rose();
	if (rc < 0) {
		perror("Cannot init ROSE access point");
		exit(1);
	}
	
	rc = db_open(wp_file);
	if (rc) {
		fprintf(stderr, "Cannot open database\n");
		exit(1);	
	}
		
	if (daemon && !daemon_start(TRUE)) {
		fprintf(stderr, "fpacwpd cannot become daemon\n");
		exit(1);
	}
		
	while (1) {
		int fd;
		fd = WaitEvent(1000);
		if (fd >= 0) ProcessEvent(fd);
		else poll_adjacents();
	}
}
