/****************************************************************************
 * tools/mkconfig.c
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "cfgdefine.h"

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

#define DEFCONFIG ".config"

/****************************************************************************
 * Private Functions
 ****************************************************************************/

static inline char *getfilepath(const char *name)
{
  snprintf(line, PATH_MAX, "%s/" DEFCONFIG, name);
  line[PATH_MAX] = '\0';
  return strdup(line);
}

static void show_usage(const char *progname)
{
  fprintf(stderr, "USAGE: %s <abs path to .config>\n", progname);
  exit(1);
}

/****************************************************************************
 * Public Functions
 ****************************************************************************/

int main(int argc, char **argv, char **envp)
{
  char *filepath;
  FILE *stream;

  if (argc != 2)
    {
      fprintf(stderr, "Unexpected number of arguments\n");
      show_usage(argv[0]);
    }

  filepath = getfilepath(argv[1]);
  if (!filepath)
    {
      fprintf(stderr, "getfilepath failed\n");
      exit(2);
    }

  stream = fopen(filepath, "r");
  if (!stream)
    {
      fprintf(stderr, "open %s failed: %s\n", filepath, strerror(errno));
      exit(3);
    }

  printf(
    "/* config.h -- Autogenerated! Do not edit. */\n\n"
    "#ifndef __INCLUDE_NUTTX_CONFIG_H\n"
    "#define __INCLUDE_NUTTX_CONFIG_H\n\n"
    "/* Used to represent the values of tristate options */\n\n"
    "#define CONFIG_y 1\n"
    "#define CONFIG_m 2\n\n"
    "/* General Definitions ***********************************/\n");

  generate_definitions(stream);

  printf(
    "\n/* Sanity Checks *****************************************/\n\n"
    "/* If the end of RAM is not specified then it is assumed to be\n"
    " * the beginning of RAM plus the RAM size.\n"
    " */\n\n"
    "#ifndef CONFIG_RAM_END\n"
    "#  define CONFIG_RAM_END (CONFIG_RAM_START+CONFIG_RAM_SIZE)\n"
    "#endif\n\n"
    "#ifndef CONFIG_RAM_VEND\n"
    "#  define CONFIG_RAM_VEND (CONFIG_RAM_VSTART+CONFIG_RAM_SIZE)\n"
    "#endif\n\n"
    "/* If the end of FLASH is not specified then it is assumed to be\n"
    " * the beginning of FLASH plus the FLASH size.\n"
    " */\n\n"
    "#ifndef CONFIG_FLASH_END\n"
    "#  define CONFIG_FLASH_END (CONFIG_FLASH_START+CONFIG_FLASH_SIZE)\n"
    "#endif\n\n"
    "#endif /* __INCLUDE_NUTTX_CONFIG_H */\n");

  fclose(stream);

  /* Exit (without bothering to clean up allocations) */

  free(filepath);
  return 0;
}
