/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Apache.Qpid.Proton.Types.Transport;
using NUnit.Framework;

namespace Apache.Qpid.Proton.Types.Transactions
{
   [TestFixture]
   public class OpenTest
   {
      [Test]
      public void TestGetPerformativeType()
      {
         Assert.AreEqual(PerformativeType.Open, new Open().Type);
      }

      [Test]
      public void TestToStringOnFreshInstance()
      {
         Assert.IsNotNull(new Open().ToString());
      }

      [Test]
      public void TestSetGetOfOutgoingLocales()
      {
         Open open = new Open();
         Symbol[] expected = { Symbol.Lookup("1"), Symbol.Lookup("2") };

         open.OutgoingLocales = new Symbol[] { Symbol.Lookup("1"), Symbol.Lookup("2") };

         Assert.AreEqual(expected, open.OutgoingLocales);
         Assert.IsNotNull(open.ToString());
      }

      [Test]
      public void TestSetGetOfIncomingLocales()
      {
         Open open = new Open();
         Symbol[] expected = { Symbol.Lookup("1"), Symbol.Lookup("2") };

         open.IncomingLocales = new Symbol[] { Symbol.Lookup("1"), Symbol.Lookup("2") };

         Assert.AreEqual(expected, open.IncomingLocales);
         Assert.IsNotNull(open.ToString());
      }

      [Test]
      public void TestSetChannelMaxFromShort()
      {
         Open open = new Open();

         open.ChannelMax = 65535;
         Assert.AreEqual(65535, open.ChannelMax);
         open.ChannelMax = 0;
         Assert.AreEqual(0, open.ChannelMax);
      }

      [Test]
      public void TestInitialState()
      {
         Open open = new Open();

         Assert.AreEqual(1, open.GetElementCount());
         Assert.IsFalse(open.IsEmpty());
         Assert.IsFalse(open.HasChannelMax());
         Assert.IsTrue(open.HasContainerId());
         Assert.IsFalse(open.HasDesiredCapabilities());
         Assert.IsFalse(open.HasHostname());
         Assert.IsFalse(open.HasIdleTimeout());
         Assert.IsFalse(open.HasIncomingLocales());
         Assert.IsFalse(open.HasMaxFrameSize());
         Assert.IsFalse(open.HasOfferedCapabilities());
         Assert.IsFalse(open.HasOutgoingLocales());
         Assert.IsFalse(open.HasProperties());
      }

      [Test]
      public void TestChannelMaxAlwaysPresentOnceSet()
      {
         Open open = new Open();

         Assert.IsFalse(open.HasChannelMax());
         open.ChannelMax = 0;
         Assert.IsTrue(open.HasChannelMax());
         open.ChannelMax = 32767;
         Assert.IsTrue(open.HasChannelMax());
         open.ChannelMax = 65535;
         Assert.IsTrue(open.HasChannelMax());
      }

      [Test]
      public void TestCopyFromNew()
      {
         Open original = new Open();
         Open copy = original.Copy();

         Assert.IsFalse(original.IsEmpty());
         Assert.IsFalse(copy.IsEmpty());

         Assert.AreEqual(1, original.GetElementCount());
         Assert.AreEqual(1, copy.GetElementCount());
      }
   }
}