/*
* File: scanner.c
* Company: TWAIN Working Group, JFL Peripheral Solutions Inc.
* Author: Jon Harju
* Description:
*		Simulates basic scanner operations
*
*	Copyright  1998 TWAIN Working Group: Adobe Systems Incorporated, 
*	Canon Information Systems, Eastman Kodak Company, 
*	Fujitsu Computer Products of America, Genoa Technology, 
*	Hewlett-Packard Company, Intel Corporation, Kofax Image Products, 
*	JFL Peripheral Solutions Inc., Ricoh Corporation, and Xerox Corporation.  
*	All rights reserved.
*/

#include <windows.h>
#include <windowsx.h>

#include "..\\twcommon\\twain.h"
#include "..\\twcommon\\twndebug.h"

#define  WIDTHBYTES(i)    ((i+31)/32*4)
#define  PACKBYTES(i)    ((i+7)/8)

#include "twd_type.h"
#include "scanner.h"
#include "dscaps.h"
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <lug.h>
#include <lugfnts.h>

extern HANDLE hDSInst;

/*
*	Static globals
*/
static int _giMaxPagesToScan = 1;
static int _giPagesRemaining = 1;
static PROCESS_INFORMATION _giSubProcess;


static int _giRowsCopied = 0;
static char _gszBuffer[8192];
DWORD _gdwLineSize = 0l;
int _gnRowsTotal = 0;
static int _gfd = -1;
static long _gnOffset = 0L;
static int _gnColorMax = 0;
static int _gnBitsLeft;
extern int nxMM, nyMM, ntMM, nlMM;
extern int stop_on_error;
double current_gamma=1.0F;
int current_bright=100;
int sharpen_bm=0, blur_bm=0;
HGLOBAL _ghDIB=NULL;

/*
*	Globals that we use to simulate scanning transfer from a DIB
*/
BOOL IsDocumentInFeeder()
{
	return _giPagesRemaining?TRUE:FALSE;
}
BOOL StartDIB(DWORD *pdwLineSize)
{
	char szModulePath[_MAX_PATH];
	char szCompletePath[_MAX_PATH];
	char szFileLocation[_MAX_PATH];
	char szDriveName[_MAX_DRIVE];
	int in_comment=0;
	int bits;

	if(_gfd != -1)
		close(_gfd);
	_gfd=-1;

	VERIFY(GetModuleFileName(hDSInst, szModulePath, _MAX_PATH));
	_splitpath(szModulePath, szDriveName, szFileLocation, NULL, NULL);
	wsprintf(szCompletePath, "%s%s%s", szDriveName, szFileLocation, "temp.ppm");

	_gfd=open(szCompletePath, O_RDONLY | O_BINARY);
	if(_gfd == -1)
	{
		return(FALSE);
	}

	if(!read(_gfd, _gszBuffer, sizeof(_gszBuffer)))
	{
		close(_gfd);
		return(FALSE);
	}

	if(_gszBuffer[0] != 'P' || (_gszBuffer[1] != '6' && _gszBuffer[1] != '5' && _gszBuffer[1] != '4'))
	{
		MessageBox(NULL, "Unerwartetes Datenformat fr Scannerdaten", "SANE Scan", MB_OK | MB_ICONSTOP);
		close(_gfd);
		return(FALSE);
	}

	_gnOffset=2; /* Start after magic */
	while(isspace(_gszBuffer[_gnOffset]))
		_gnOffset++;

	while(!isdigit(_gszBuffer[_gnOffset]) || in_comment)
	{
		if(_gszBuffer[_gnOffset] == '#')
			in_comment=1;
		if(_gszBuffer[_gnOffset] == '\n' || _gszBuffer[_gnOffset] == '\r')
			in_comment=0;
		_gnOffset++;
	};

	if(GetCurrentPixelType()==TWPT_RGB)
		bits=24;
	else if(GetCurrentPixelType()==TWPT_GRAY)
		bits=8;
	else if(GetCurrentPixelType()==TWPT_BW)
		bits=1;

	/* At first non-comment digit now */
	_gdwLineSize = atoi(&_gszBuffer[_gnOffset]);
	if(GetCurrentPixelType()==TWPT_GRAY)
		*pdwLineSize=WIDTHBYTES(_gdwLineSize*24);
	else
		*pdwLineSize=WIDTHBYTES(_gdwLineSize*bits);

	while(isdigit(_gszBuffer[_gnOffset]))
		_gnOffset++;

	while(!isdigit(_gszBuffer[_gnOffset]) || in_comment)
	{
		if(_gszBuffer[_gnOffset] == '#')
			in_comment=1;
		if(_gszBuffer[_gnOffset] == '\n' || _gszBuffer[_gnOffset] == '\r')
			in_comment=0;
		_gnOffset++;
	};

	_gnRowsTotal = atoi(&_gszBuffer[_gnOffset]);

	while(isdigit(_gszBuffer[_gnOffset]))
		_gnOffset++;

	if(GetCurrentPixelType()!=TWPT_BW)
	{
		while(!isdigit(_gszBuffer[_gnOffset]) || in_comment)
		{
			if(_gszBuffer[_gnOffset] == '#')
				in_comment=1;
			if(_gszBuffer[_gnOffset] == '\n' || _gszBuffer[_gnOffset] == '\r')
				in_comment=0;
			_gnOffset++;
		};

		_gnColorMax = atoi(&_gszBuffer[_gnOffset]);

		while(isdigit(_gszBuffer[_gnOffset]))
			_gnOffset++;
	}
	else
	{
		_gnColorMax=1;
	}
	_gnOffset++;
	_giRowsCopied=0;
	_gnBitsLeft=0;

	return(TRUE);
}

BOOL StartScan(DWORD *pdwLineSize)
{
	char szModulePath[_MAX_PATH];
	char szCompletePath[_MAX_PATH];
	char szFileLocation[_MAX_PATH];
	char szDriveName[_MAX_DRIVE];
	STARTUPINFO si;
	int cc;
	char tmp[256];
	DWORD exit_code=0;
	char *pszMode;
	int tries;

	if(_gfd != -1)
		close(_gfd);
	_gfd=-1;

	if(GetCurrentPixelType()==TWPT_RGB)
	{
		pszMode="Colour";
	}
	else if(GetCurrentPixelType()==TWPT_GRAY)
	{
		pszMode="GreyScale";
	}
	else if(GetCurrentPixelType()==TWPT_BW)
	{
		pszMode="LineArt";
	}

	VERIFY(GetModuleFileName(hDSInst, szModulePath, _MAX_PATH));
	_splitpath(szModulePath, szDriveName, szFileLocation, NULL, NULL);
	wsprintf(szCompletePath, "%s%s%s", szDriveName, szFileLocation, "temp.ppm");

	memset((char *)&si, 0, sizeof(si));
	si.cb=sizeof(si);

	for(tries=0;tries < 6 ; tries++)
	{
		unlink(szCompletePath);
#error Edit here to your configuration
		sprintf(tmp, "c:\\windows\\twain_32\\scanimg.exe -d <your server IP>:snapscan:/dev/usb/scanner0 -f %s -l 0 -t 0 -x %d -y %d --resolution %.0f --mode %s -l %d -t %d", szCompletePath, nxMM, nyMM, GetCurrentXResolution(), pszMode, nlMM, ntMM);
/*		MessageBox(NULL, tmp, "test", MB_OK);
*/		cc=CreateProcess(NULL, tmp,
			NULL, NULL, FALSE, DETACHED_PROCESS, NULL, NULL, &si, &_giSubProcess);

		if(!cc)
		{
			MessageBox(NULL, "Der Scanvorgang konnte nicht gestartet werden", "SANE Scan", MB_OK | MB_ICONSTOP);
			return(FALSE);
		}

		while(GetExitCodeProcess(_giSubProcess.hProcess, &exit_code) && exit_code == STILL_ACTIVE)
			Sleep(1);

		if(exit_code == STILL_ACTIVE)
		{
			CloseHandle(_giSubProcess.hProcess);
			CloseHandle(_giSubProcess.hThread);

			MessageBox(NULL, "Es ist ein interner Fehler aufgetreten", "SANE Scan", MB_OK | MB_ICONSTOP);
			return(FALSE);
		}

		CloseHandle(_giSubProcess.hProcess);
		CloseHandle(_giSubProcess.hThread);
		if(StartDIB(pdwLineSize))
			break;
		Sleep(5);
	}
	if(tries == 6)
	{
		MessageBox(NULL, "Der Scanner ist nicht bereit oder antwortet nicht", "SANE Scan", MB_OK | MB_ICONSTOP);
		return(FALSE);
	}
	return(TRUE);
}

BOOL ReadDataBlock(BYTE *pBuffer, DWORD dwMaxSize, DWORD *pdwBytesWritten, BOOL *pbLastBuffer)
{
	int bits=0;
	DWORD dwPixelIndex=0;
	static LPSTR lpDIB;
	static int nLineSize;

	if(!_ghDIB)
	{
		LPBITMAPINFOHEADER lpBi;

		_ghDIB=ScanToDIB();
		lpDIB=GlobalLock(_ghDIB);
		lpBi=(LPBITMAPINFOHEADER)lpDIB;

		lpDIB+=sizeof(BITMAPINFOHEADER);
		lpDIB+=sizeof(RGBQUAD)*lpBi->biClrUsed;
		nLineSize=WIDTHBYTES(lpBi->biBitCount*lpBi->biWidth);
		lpDIB+=(lpBi->biHeight-1)*nLineSize;
	}

	*pbLastBuffer=FALSE;

	if(GetCurrentPixelType()==TWPT_RGB)
		bits=24;
	else if(GetCurrentPixelType()==TWPT_GRAY)
		bits=8;
	else if(GetCurrentPixelType()==TWPT_BW)
		bits=1;

	if(bits == 24)
	{
		for(dwPixelIndex = 0; (dwPixelIndex+2) < _gdwLineSize*3; dwPixelIndex+= 3)
		{
			pBuffer[dwPixelIndex+2]=lpDIB[dwPixelIndex];
			pBuffer[dwPixelIndex+1]=lpDIB[dwPixelIndex+1];
			pBuffer[dwPixelIndex]=lpDIB[dwPixelIndex+2];
		}
		
		lpDIB-=nLineSize;
		*pdwBytesWritten=nLineSize;

		_giRowsCopied++;
		if(_giRowsCopied >= _gnRowsTotal)
			*pbLastBuffer=TRUE;
	}
	else if(bits == 8)
	{
		memcpy(pBuffer, lpDIB, nLineSize);
		lpDIB-=nLineSize;
		*pdwBytesWritten=nLineSize;
		
		_giRowsCopied++;
		if(_giRowsCopied >= _gnRowsTotal)
			*pbLastBuffer=TRUE;
	}
	else if(bits == 1)
	{
		memset(pBuffer, 0, WIDTHBYTES(_gdwLineSize));

		for(dwPixelIndex = 0; dwPixelIndex < _gdwLineSize; dwPixelIndex++)
		{
			if((lpDIB[(dwPixelIndex)/8] & (1 << (7-(dwPixelIndex % 8)))))
				pBuffer[dwPixelIndex/8] |= 1<<(7-(dwPixelIndex % 8));
		}
		lpDIB-=nLineSize;
		*pdwBytesWritten=WIDTHBYTES(_gdwLineSize);
		
		_giRowsCopied++;
		if(_giRowsCopied >= _gnRowsTotal)
			*pbLastBuffer=TRUE;
	}

	return TRUE;
}

void StopScan()
{
	/*
	* Decrement the page count
	*/
	_giPagesRemaining--;
	if(_giPagesRemaining < 0)
	{
		_giPagesRemaining = 0;
	}

	/*
	*	Clean up the DIB
	*/
	_giRowsCopied = 0;
	_gdwLineSize = 0l;

	if(_ghDIB)
	{
		GlobalFree(_ghDIB);
		_ghDIB=NULL;
	}
	close(_gfd);
}

/*
*	Additional functions for our simulation
*/
int GetNumPages()
{
	return _giMaxPagesToScan;
}

void SetNumPages(int iNumPages)
{
	_giMaxPagesToScan = iNumPages;
	return;
}

void LoadFeeder()
{
	_giPagesRemaining = _giMaxPagesToScan;
	return;
}

void ClearFeeder()
{
	_giPagesRemaining = 0;
}

HGLOBAL ScanToDIB()
{
	HGLOBAL hDIBHandle = NULL;
	LPBITMAPINFOHEADER bh;
	LPSTR lpBits;
	int nRowsCopied;
	int bits;
	int palettesize;
	LPRGBQUAD lpPalette;
	int i;
	int bitsleft=0;
	int bytes;
	bitmap_hdr bh_in, bh_out;
	byte *r, *g, *b;

	stop_on_error=0;	

	if(GetCurrentPixelType()==TWPT_RGB)
	{
		bits=24;
		palettesize=0;
	}
	else if(GetCurrentPixelType()==TWPT_GRAY)
	{
		bits=24;
		palettesize=0;
	}
	else if(GetCurrentPixelType()==TWPT_BW)
	{
		bits=1;
		palettesize=2;
	}

	memset((char *)&bh_in, 0, sizeof(bh_in));
	memset((char *)&bh_out, 0, sizeof(bh_out));
	bh_in.magic = LUGUSED;
	bh_in.xsize = _gdwLineSize;
	bh_in.ysize = _gnRowsTotal;
	bh_in.depth = bits < 8 || GetCurrentPixelType()==TWPT_GRAY ? 8 : bits;
	bh_in.colors = ( 1 <<  bh_in.depth);
	bh_in.cmap = NULL;
	bh_in.r= (byte *) malloc( _gdwLineSize*_gnRowsTotal );
	bh_in.g= (byte *) malloc( _gdwLineSize*_gnRowsTotal );
	bh_in.b= (byte *) malloc( _gdwLineSize*_gnRowsTotal );

	r=bh_in.r;
	g=bh_in.g;
	b=bh_in.b;

	/* Palette */
	if(bits < 24 || GetCurrentPixelType()==TWPT_GRAY)
	{
		bh_in.cmap=malloc(3*bh_in.colors);
		if(bits == 1)
		{
			bh_in.cmap[0]=0;
			bh_in.cmap[1]=0;
			bh_in.cmap[2]=0;
			bh_in.cmap[3]=255;
			bh_in.cmap[4]=255;
			bh_in.cmap[5]=255;
		}
		else
		{
			for(i=0;i<256;i++)
			{
				bh_in.cmap[i*3]=i;
				bh_in.cmap[i*3+1]=i;
				bh_in.cmap[i*3+2]=i;
			}
		}
	}

	lseek(_gfd, _gnOffset, SEEK_SET);
	for(nRowsCopied=0;nRowsCopied < _gnRowsTotal;nRowsCopied++)
	{
		DWORD dwPixelIndex = 0;
		BYTE bTempByte = 0;

		if(bitsleft)
			_gnOffset--;
		lseek(_gfd, _gnOffset, SEEK_SET);
		if(GetCurrentPixelType()==TWPT_GRAY)
		{
			bytes=read(_gfd, _gszBuffer, PACKBYTES(_gdwLineSize*8)+1)-1;
			_gnOffset+=bytes;
		}
		else
		{
			bytes=read(_gfd, _gszBuffer, PACKBYTES(_gdwLineSize*bits)+1)-1;
			_gnOffset+=bytes;
		}

		if(GetCurrentPixelType()==TWPT_RGB)
		{
			for(dwPixelIndex = 0; (dwPixelIndex+2) < _gdwLineSize*3; dwPixelIndex+= 3)
			{
				*(r++) = _gszBuffer[dwPixelIndex+2];
				*(g++) = _gszBuffer[dwPixelIndex+1];
				*(b++) = _gszBuffer[dwPixelIndex];
			}
		}
		else if(GetCurrentPixelType()==TWPT_GRAY)
		{
			for(dwPixelIndex = 0; (dwPixelIndex+2) < _gdwLineSize*3; dwPixelIndex+= 3)
			{
				*(r++) = _gszBuffer[dwPixelIndex/3];
			}
		}
		else if(GetCurrentPixelType()==TWPT_BW)
		{
			for(dwPixelIndex = 0; dwPixelIndex < _gdwLineSize; dwPixelIndex++)
			{
				if(!(_gszBuffer[(dwPixelIndex+bitsleft)/8] & (1 << (7-((dwPixelIndex+bitsleft) % 8)))))
					(*r++)=1;
				else
					(*r++)=0;
			}
			bitsleft=(bitsleft+_gdwLineSize) % 8;
		}
	}

	if(GetCurrentPixelType()==TWPT_GRAY)
	{
		if(current_bright < 100)
			for(i=0;i<3*bh_in.colors;i++)
				bh_in.cmap[i]=LIMITPIXEL(MulDiv(bh_in.cmap[i], current_bright, 100));
		else if(current_bright > 100)
			for(i=0;i<3*bh_in.colors;i++)
				bh_in.cmap[i]=255-LIMITPIXEL(MulDiv(255-bh_in.cmap[i], 200-current_bright, 100));
	}
	if(GetCurrentPixelType()==TWPT_RGB)
	{
		if(current_bright < 100)
			for(i=0;i<bh_in.xsize*bh_in.ysize;i++)
			{
				bh_in.r[i]=LIMITPIXEL(MulDiv(bh_in.r[i], current_bright, 100));
				bh_in.g[i]=LIMITPIXEL(MulDiv(bh_in.g[i], current_bright, 100));
				bh_in.b[i]=LIMITPIXEL(MulDiv(bh_in.b[i], current_bright, 100));
			}
		else if(current_bright > 100)
			for(i=0;i<bh_in.xsize*bh_in.ysize;i++)
			{
				bh_in.r[i]=255-LIMITPIXEL(MulDiv(255-bh_in.r[i], 200-current_bright, 100));
				bh_in.g[i]=255-LIMITPIXEL(MulDiv(255-bh_in.g[i], 200-current_bright, 100));
				bh_in.b[i]=255-LIMITPIXEL(MulDiv(255-bh_in.b[i], 200-current_bright, 100));
			}
	}

	if(current_gamma != 1.0)
	{
		gamma_correction(&bh_in, &bh_out, current_gamma);
		freebitmap(&bh_in);
		memcpy((char *)&bh_in, (char *)&bh_out, sizeof(bh_in));
		memset((char *)&bh_out, 0, sizeof(bh_out));
	}

	if(sharpen_bm)
	{
		sharpen_bitmap(&bh_in, &bh_out);
		freebitmap(&bh_in);
		memcpy((char *)&bh_in, (char *)&bh_out, sizeof(bh_in));
		memset((char *)&bh_out, 0, sizeof(bh_out));
	}

	if(blur_bm)
	{
		blur_bitmap(&bh_in, &bh_out);
		freebitmap(&bh_in);
		memcpy((char *)&bh_in, (char *)&bh_out, sizeof(bh_in));
		memset((char *)&bh_out, 0, sizeof(bh_out));
	}

	r=bh_in.r;
	g=bh_in.g;
	b=bh_in.b;

	hDIBHandle=GlobalAlloc(GMEM_MOVEABLE, sizeof(BITMAPINFOHEADER)+WIDTHBYTES(_gdwLineSize*bits)*_gnRowsTotal+sizeof(RGBQUAD)*palettesize);
	if(!hDIBHandle)
	{
		MessageBox(NULL, "No memory for bitmap", "test", MB_OK);
		return NULL;
	}

	bh=(LPBITMAPINFOHEADER)GlobalLock(hDIBHandle);

	bh->biSize               = sizeof(BITMAPINFOHEADER);
	bh->biWidth              = _gdwLineSize;
	bh->biHeight             = _gnRowsTotal;
	bh->biPlanes             = 1;
	bh->biBitCount           = bits;
	bh->biCompression        = BI_RGB;
	bh->biClrUsed            = palettesize;
	bh->biClrImportant       = palettesize;

	bh->biSizeImage          = 0;
	bh->biXPelsPerMeter      = (LONG)(GetCurrentXResolution()*39.37007874F);
	bh->biYPelsPerMeter      = (LONG)(GetCurrentXResolution()*39.37007874F);

	lpBits=(LPSTR)(bh+1);
	lpPalette=(LPRGBQUAD)lpBits;
	if(palettesize > 2)
	{
		for(i=0;i<palettesize;i++)
		{
			lpPalette[i].rgbBlue=i;
			lpPalette[i].rgbRed=i;
			lpPalette[i].rgbGreen=i;
			lpPalette[i].rgbReserved=0;
		}
	}
	else if(palettesize == 2)
	{
		lpPalette[0].rgbBlue=0;
		lpPalette[0].rgbRed=0;
		lpPalette[0].rgbGreen=0;
		lpPalette[0].rgbReserved=0;
		lpPalette[1].rgbBlue=255;
		lpPalette[1].rgbRed=255;
		lpPalette[1].rgbGreen=255;
		lpPalette[1].rgbReserved=0;
	}

	lpBits+=sizeof(RGBQUAD)*palettesize;

	lpBits+=WIDTHBYTES(_gdwLineSize*bits)*(_gnRowsTotal-1); /* Last line */

	for(nRowsCopied=0;nRowsCopied < _gnRowsTotal;nRowsCopied++)
	{
		DWORD dwPixelIndex = 0;
		BYTE bTempByte = 0;

		if(GetCurrentPixelType()==TWPT_RGB)
		{
			for(dwPixelIndex = 0; (dwPixelIndex+2) < _gdwLineSize*3; dwPixelIndex+= 3)
			{
				lpBits[dwPixelIndex] = *(r++);
				lpBits[dwPixelIndex+1] = *(g++);
				lpBits[dwPixelIndex+2] = *(b++);
			}
		}
		else if(GetCurrentPixelType()==TWPT_GRAY)
		{
			for(dwPixelIndex = 0; (dwPixelIndex+2) < _gdwLineSize*3; dwPixelIndex+= 3)
			{
				lpBits[dwPixelIndex] = bh_in.cmap[*(r)*3];
				lpBits[dwPixelIndex+1] = bh_in.cmap[*(r)*3];
				lpBits[dwPixelIndex+2] = bh_in.cmap[*(r++)*3];
			}
		}
		else if(GetCurrentPixelType()==TWPT_BW)
		{
			memset(lpBits, 0, WIDTHBYTES(_gdwLineSize*bits));

			for(dwPixelIndex = 0; dwPixelIndex < _gdwLineSize; dwPixelIndex++)
			{
				if(*(r++))
					lpBits[dwPixelIndex/8] |= 1<<(7-(dwPixelIndex % 8));
			}
		}

		lpBits-=WIDTHBYTES(_gdwLineSize*bits);
	}
	GlobalUnlock(hDIBHandle);

	freebitmap(&bh_in);
	freebitmap(&bh_out);

	return hDIBHandle;
}
