(***************************************************************************
 *
 *  GDriver     Graphics utility driver generic interface                1.1
 *
 *  Generic interface for all drivers of the graphics utility used
 *  by the FreeType test programs.
 *
 *  Copyright 1996-1999 David Turner, Robert Wilhelm and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ***************************************************************************)

Unit GDriver;

interface

uses GEvents, GMain;

  (* It is the responsability of the driver to map its events   *)
  (* to the TEvent structure when called through Get_Event.     *)

type
  Event = record
            what  : GEvent; (* event class     *)
            info  : Int;    (* event parameter *)
          end;

  (* The event classes are defined in the unit 'GEvents' included *)
  (* by the test programs, not by the graphics utility            *)

  procedure Get_Event( var ev : Event );
  (* get last event. In full-screen modes, a key-stroke must be *)
  (* translated to an event class with a parameter.             *)

  function  Driver_Set_Graphics( mode : Int ) : boolean;
  (* A call to this function will set the graphics mode, the Vio    *)
  (* variable, as well as the values vio_ScanLineWidth, vio_Width   *)
  (* and vio_Height                                                 *)

  function Driver_Restore_Mode : boolean;
  (* Restore previous mode or release display buffer/window *)

  procedure Driver_Display_Bitmap( var buff; line, col : Int );
  (* Display bitmap on screen *)

implementation

{$IFDEF OS2}

{ $IFDEF VIRTUALPASCAL}
  uses Os2Base, CRT;
  {$I GDRV_OS2.INC}
{ $ENDIF}

{$ELSE} (* not OS/2 *)

{$IFDEF WINDOWS}
  uses WinCRT, WinProcs, WinTypes {$IFDEF VER80} , Messages {$ENDIF};
  {$I GDRV_WIN.INC}

{$ELSE} (* neither OS/2 nor (16-bit) Windows *)

{$IFDEF WIN32}
  uses Windows {$IFNDEF FPC} , Messages {$ENDIF};
  {$I GDRV_WIN.INC} (* used to be different, but now same as 16-bit Windows *)

{$ELSE} (* neither OS/2 nor Windows nor Win32 *)

(* we have no usable API, so should go down to the bare metal... *)

{$IFDEF FPC}

  (* deal separately with each (remaining) versions of FPC *)

  {$IFDEF GO32V2}

    uses CRT;
    {$I GDrvGO32.INC}

  {$ELSE}

  ERROR : FPC without a known target, unable to compile the graphic driver

  {$ENDIF}

{$ELSE} (* no API, no 32-bit compiler, should be some version of TP/BP *)

{$IFDEF DPMI}

  uses CRT;
  {$I GDrv_DOS.INC} (* used to be different, but now same as real-mode DOS *)

{$ELSE} (* no usable API neither prot.mode BP7 (DPMI) *)

{$IFDEF MSDOS}

  uses CRT;
  {$I GDRV_DOS.INC}

{$ELSE} (* not a known target... *)

ERROR : dont know your current target, unable to compile the graphic driver

{$ENDIF}{$ENDIF}{$ENDIF}{$ENDIF}{$ENDIF}{$ENDIF}{ouf! no more $endif...:-}

type
  Translator = record
                 key      : char;
                 ev_class : GEvent;
                 ev_info  : Int;
               end;

const
  Num_Translators = 15;

  Translators : array[1..Num_Translators] of Translator
              = (
                  (key:#27; ev_class:event_Quit ;         ev_info:0),

                  (key:'x'; ev_class: event_Rotate_Glyph; ev_info:  -1),
                  (key:'c'; ev_class: event_Rotate_Glyph; ev_info:   1),
                  (key:'v'; ev_class: event_Rotate_Glyph; ev_info: -16),
                  (key:'b'; ev_class: event_Rotate_Glyph; ev_info:  16),

                  (key:'9'; ev_class: event_Change_Glyph; ev_info:-100),
                  (key:'0'; ev_class: event_Change_Glyph; ev_info: 100),
                  (key:'i'; ev_class: event_Change_Glyph; ev_info: -10),
                  (key:'o'; ev_class: event_Change_Glyph; ev_info:  10),
                  (key:'k'; ev_class: event_Change_Glyph; ev_info:  -1),
                  (key:'l'; ev_class: event_Change_Glyph; ev_info:   1),

                  (key:'+'; ev_class: event_Scale_Glyph; ev_info:  10),
                  (key:'-'; ev_class: event_Scale_Glyph; ev_info: -10),
                  (key:'u'; ev_class: event_Scale_Glyph; ev_info:   1),
                  (key:'j'; ev_class: event_Scale_Glyph; ev_info:  -1)
                );

  procedure Process_Key( var ev : Event; c : Char );
  var
    i : Int;
  begin
    for i := 1 to Num_Translators do
    begin
      if c = translators[i].key then
      begin
        ev.what := translators[i].ev_class;
        ev.info := translators[i].ev_info;
        Exit;
      end;
    end;

    (* unrecognized keystroke *)
    ev.what := event_Keyboard;
    ev.info := Ord(c);
  end;


  procedure Get_Event( var ev : Event );
  begin
{$IFDEF HAVE_OUREVENT}
    (* Windows drivers should be backcalled in the event handlers. *)
    MessageLoop;
    ev := ourevent;
{$ELSE}
    Process_Key( ev, ReadKey );
{$ENDIF}

  end;

end.
