// SPDX-FileCopyrightText: 2006 Istituto Nazionale di Fisica Nucleare
//
// SPDX-License-Identifier: Apache-2.0

package org.italiangrid.voms.store;

import java.security.cert.X509Certificate;
import java.util.List;

/**
 * The VOMS LSC information.
 * 
 * @author Andrea Ceccanti
 *
 */
public interface LSCInfo {

  /**
   * Sets the name of the file from where this LSC info was parsed from.
   * 
   * @param filename
   *          the name of the file from where this LSC info was parsed from.
   */
  public void setFilename(String filename);

  /**
   * Returns the name of file from where this LSC info was parsed from.
   * 
   * @return the name of the file from where this LSC info was parsed from.
   */
  public String getFilename();

  /**
   * Returns the VO name this LSC info applies to.
   * 
   * @return the VO name this LSC info applies to
   */
  public String getVOName();

  /**
   * Returns the host name of the VOMS AA this LSC info applies to.
   * 
   * @return the host name of the VOMS AA this LSC info applies to
   */
  public String getHostname();

  /**
   * Returns the certificate chain description of the VOMS AA for the given VO
   * and hostname.
   * 
   * The certificate chain description is a list of X.500 distinguished names
   * encoded as strings according to the OpenSSL slash-separated format, as in:
   * <code>/C=IT/O=INFN/CN=INFN CA</code>
   * 
   * The first element in the description is the leaf certificate, while the
   * last is the CA certificate.
   * 
   * @return the certificate chain description of the VOMS AA for the given VO
   *         and hostname.
   */
  public List<String> getCertificateChainDescription();

  /**
   * Checks if the certificate chain description maintained in the LSC
   * information matches the certificate chain passed as argument.
   * 
   * @param certChain
   *          the certificate chain to be checked
   * @return <code>true</code> if the description matches, <code>false</code>
   *         otherwise
   */
  public boolean matches(X509Certificate[] certChain);

}
