/* -*-pgsql-c-*- */
/*
 *
 * $Header: /home/t-ishii/repository/pgpool/pool_config.l,v 1.5 2003/07/20 14:44:37 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * pool_config.l: read configuration file
 *
 */

%{

#include "pool.h"

#include <stdio.h>
#include <string.h>

/* to shut off compiler warnings */
int yylex(void);

POOL_CONFIG pool_config;	/* configuration values */
static unsigned Lineno;

typedef enum {
  POOL_KEY = 1,
  POOL_INTEGER,
  POOL_REAL,
  POOL_STRING,
  POOL_UNQUOTED_STRING,
  POOL_EQUALS,
  POOL_EOL,
  POOL_PARSE_ERROR
} POOL_TOKEN;

static char *extract_string(char *value, POOL_TOKEN token);

%}

%option 8bit
%option never-interactive
%option nounput
%option noyywrap

SIGN            ("-"|"+")
DIGIT           [0-9]
HEXDIGIT        [0-9a-fA-F]

INTEGER         {SIGN}?({DIGIT}+|0x{HEXDIGIT}+)

EXPONENT        [Ee]{SIGN}?{DIGIT}+
REAL            {SIGN}?{DIGIT}*"."{DIGIT}*{EXPONENT}?

LETTER          [A-Za-z_\200-\377]
LETTER_OR_DIGIT [A-Za-z_0-9\200-\377]

KEY              {LETTER}{LETTER_OR_DIGIT}*

UNQUOTED_STRING {LETTER}({LETTER_OR_DIGIT}|[-._:/])*
STRING          \'([^'\n]|\\.)*\'

%%

\n              Lineno++; return POOL_EOL;
[ \t\r]+        /* eat whitespace */
#.*$            /* eat comment */

{KEY}           return POOL_KEY;
{STRING}        return POOL_STRING;
{UNQUOTED_STRING} return POOL_UNQUOTED_STRING;
{INTEGER}       return POOL_INTEGER;
{REAL}          return POOL_REAL;
=               return POOL_EQUALS;

.               return POOL_PARSE_ERROR;

%%

int pool_get_config(char *confpath)
{
	FILE *fd;
	int token;
	char key[1024];

	/* set hardcoded default values */
	pool_config.inetdomain = 1;
	pool_config.port = 9999;
	pool_config.backend_host_name = "";
	pool_config.backend_port = 5432;
	pool_config.secondary_backend_host_name = "";
	pool_config.secondary_backend_port = 0;		/* no secondary backend */
	pool_config.num_init_children = 16;
	pool_config.child_life_time = 0;
	pool_config.connection_life_time = 0;
	pool_config.max_pool = 1;
	pool_config.logdir = DEFAULT_LOGDIR;

	pool_config.current_backend_host_name = "";
	pool_config.current_backend_port = 5432;

#define PARSE_ERROR()		pool_error("pool_config: parse error at line %d '%s'", Lineno, yytext)

	/* open config file */
	fd = fopen(confpath, "r");
	if (!fd)
	{
		fprintf(stderr, "pool_config: could not open configuration file (%s)\n",
				POOL_CONF_FILE_NAME);
		fprintf(stderr, "pool_config: using default values...\n");
		return 0;
	}

	yyin = fd;
	Lineno = 1;

	while ((token = yylex()))
	{
		if (token == POOL_PARSE_ERROR)
		{
			PARSE_ERROR();
			return(-1);
		}
		if (token == POOL_EOL)
			continue;

		if (token != POOL_KEY)
		{
			PARSE_ERROR();
			return(-1);
		}

		strncpy(key, yytext, sizeof(key));

		pool_debug("key: %s", key);

		token = yylex();

		if (token == POOL_EQUALS)
			token = yylex();

		pool_debug("value: %s kind: %d", yytext, token);

		if (!strcmp(key, "allow_inet_domain_socket"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || (v != 0 && v != 1))
			{
				pool_error("pool_config: %s must be 0 or 1", key);
				return(-1);
			}
			pool_config.inetdomain = v;
		}
		else if (!strcmp(key, "port"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || v < 1024)
			{
				pool_error("pool_config: %s must be 1024 or higher numeric value", key);
				return(-1);
			}
			pool_config.port = v;
		}
		else if (!strcmp(key, "backend_host_name"))
		{
			char *str;

			if (token != POOL_STRING && token != POOL_UNQUOTED_STRING && token != POOL_KEY)
			{
				PARSE_ERROR();
				return(-1);
			}
			str = extract_string(yytext, token);
			if (str == NULL)
			{
				return(-1);
			}
			pool_config.backend_host_name = str;
			pool_debug(":%s:", pool_config.backend_host_name);
		}
		else if (!strcmp(key, "backend_port"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || v < 1024)
			{
				pool_error("pool_config: %s must be 1024 or higher numeric value", key);
				return(-1);
			}
			pool_config.backend_port = v;
		}
		else if (!strcmp(key, "secondary_backend_host_name"))
		{
			char *str;

			if (token != POOL_STRING && token != POOL_UNQUOTED_STRING && token != POOL_KEY)
			{
				PARSE_ERROR();
				return(-1);
			}
			str = extract_string(yytext, token);
			if (str == NULL)
			{
				return(-1);
			}
			pool_config.secondary_backend_host_name = str;
			pool_debug(":%s:", pool_config.secondary_backend_host_name);
		}
		else if (!strcmp(key, "secondary_backend_port"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || (v != 0 && v < 1024))
			{
				pool_error("pool_config: %s must be 1024 or higher numeric value", key);
				return(-1);
			}
			pool_config.secondary_backend_port = v;
		}
		else if (!strcmp(key, "num_init_children"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || v < 1)
			{
				pool_error("pool_config: %s must be higher than 1 numeric value", key);
				return(-1);
			}
			pool_config.num_init_children = v;
		}
		else if (!strcmp(key, "child_life_time"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || v < 0)
			{
				pool_error("pool_config: %s must be higher than 0 numeric value", key);
				return(-1);
			}
			pool_config.child_life_time = v;
		}
		else if (!strcmp(key, "connection_life_time"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || v < 0)
			{
				pool_error("pool_config: %s must be higher than 0 numeric value", key);
				return(-1);
			}
			pool_config.connection_life_time = v;
		}
		else if (!strcmp(key, "max_pool"))
		{
			int v = atoi(yytext);

			if (token != POOL_INTEGER || v < 0)
			{
				pool_error("pool_config: %s must be higher than 0 numeric value", key);
				return(-1);
			}
			pool_config.max_pool = v;
		}
		else if (!strcmp(key, "logdir"))
		{
			char *str;

			if (token != POOL_STRING && token != POOL_UNQUOTED_STRING && token != POOL_KEY)
			{
				PARSE_ERROR();
				return(-1);
			}
			str = extract_string(yytext, token);
			if (str == NULL)
			{
				return(-1);
			}
			pool_config.logdir = str;
		}
	}

	return 0;
}

static char *extract_string(char *value, POOL_TOKEN token)
{
	char *ret;

	ret = strdup(value);
	if (!ret)
	{
		pool_error("extract_string: out of memory");
		return NULL;
	}

	if (token == POOL_STRING)
	{
		ret[strlen(ret)-1] = '\0';
		return (ret+1);
	}
	return ret;
}

