/* -*-pgsql-c-*- */
/*
 * $Header: /home/t-ishii/repository/pgpool/pool_process_query.c,v 1.16.2.1 2004/05/16 03:27:32 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003, 2004	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * pool_process_query.c: query processing stuff
 *
*/
#include "config.h"
#include <errno.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include <stdlib.h>
#include <unistd.h>

#include "pool.h"

#define REPLICATION (pool_config.replication_enabled)
#define MASTER(p) MASTER_CONNECTION(p)->con
#define SECONDARY(p) SECONDARY_CONNECTION(p)->con

static POOL_STATUS NotificationResponse(POOL_CONNECTION *frontend, 
										POOL_CONNECTION_POOL *backend);

static POOL_STATUS Query(POOL_CONNECTION *frontend, 
						 POOL_CONNECTION_POOL *backend);

static POOL_STATUS ReadyForQuery(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION_POOL *backend, int send_ready);

static POOL_STATUS CompleteCommandResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend);

static POOL_STATUS CopyInResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend);

static POOL_STATUS CopyOutResponse(POOL_CONNECTION *frontend, 
								   POOL_CONNECTION_POOL *backend);

static POOL_STATUS CopyDataRows(POOL_CONNECTION *frontend,
								POOL_CONNECTION_POOL *backend, int copyin);

static POOL_STATUS CursorResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend);

#ifdef NOTUSED
static POOL_STATUS CancelRequest(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION_POOL *backend);
#endif

static POOL_STATUS NoticeResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend);

static POOL_STATUS EmptyQueryResponse(POOL_CONNECTION *frontend,
									  POOL_CONNECTION_POOL *backend);

static int RowDescription(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION_POOL *backend);

static POOL_STATUS AsciiRow(POOL_CONNECTION *frontend, 
							POOL_CONNECTION_POOL *backend,
							short num_fields);

static POOL_STATUS BinaryRow(POOL_CONNECTION *frontend, 
							 POOL_CONNECTION_POOL *backend,
							 short num_fields);

static POOL_STATUS FunctionCall(POOL_CONNECTION *frontend, 
								POOL_CONNECTION_POOL *backend);

static POOL_STATUS FunctionResultResponse(POOL_CONNECTION *frontend, 
										  POOL_CONNECTION_POOL *backend);

static POOL_STATUS ProcessFrontendResponse(POOL_CONNECTION *frontend, 
											POOL_CONNECTION_POOL *backend);

static int synchronize(POOL_CONNECTION *cp);
static void process_reporting(POOL_CONNECTION *frontend);

POOL_STATUS pool_process_query(POOL_CONNECTION *frontend, 
							   POOL_CONNECTION_POOL *backend,
							   int connection_reuse)
{
	char kind, kind1;	/* packet kind (backend) */
	char fkind;	/* packet kind (frontend) */
	short num_fields = 0;
	fd_set	readmask;
	fd_set	writemask;
	fd_set	exceptmask;
	int fds;
	POOL_STATUS status;

	pool_debug("pool_process_query");

	if (connection_reuse)
	{
		status = ReadyForQuery(frontend, backend, 0);
		if (status != POOL_CONTINUE)
			return status;
	}

	for (;;)
	{
		kind = kind1 = 0;
		fkind = 0;

		if (!REPLICATION && MASTER(backend)->len == 0 && frontend->len == 0 ||
			REPLICATION && MASTER(backend)->len == 0 &&
			SECONDARY(backend)->len == 0
			&& frontend->len == 0)
		{

			struct timeval timeout;

			timeout.tv_sec = 1;
			timeout.tv_usec = 0;

			FD_ZERO(&readmask);
			FD_ZERO(&writemask);
			FD_ZERO(&exceptmask);
			FD_SET(frontend->fd, &readmask);
			FD_SET(MASTER(backend)->fd, &readmask);
			if (REPLICATION)
				FD_SET(SECONDARY(backend)->fd, &readmask);
			FD_SET(frontend->fd, &exceptmask);
			FD_SET(MASTER(backend)->fd, &exceptmask);

#ifdef DEBUG
			pool_debug("pid: %d select in child", getpid());
#endif
			if (REPLICATION)
				fds = select(Max(SECONDARY(backend)->fd,
								 Max(frontend->fd, MASTER(backend)->fd)+1),
							 &readmask, &writemask, &exceptmask, NULL);
			else
				fds = select(Max(frontend->fd, MASTER(backend)->fd)+1,
							 &readmask, &writemask, &exceptmask, NULL);

			if (fds == -1)
			{
				if (errno == EINTR)
					continue;

				pool_error("select() failed. reason: %s", strerror(errno));
				return POOL_FATAL;
			}

			if (fds == 0)
			{
				return POOL_CONTINUE;
			}

			if (FD_ISSET(MASTER(backend)->fd, &readmask))
			{
				pool_read(MASTER(backend), &kind, 1);
				pool_debug("read kind from backend %c", kind);
			}

			if (REPLICATION && FD_ISSET(SECONDARY(backend)->fd, &readmask))
			{
				pool_read(SECONDARY(backend), &kind1, 1);
				pool_debug("read kind from secondary backend %c", kind1);
			}

			if (FD_ISSET(frontend->fd, &exceptmask))
			{
				return POOL_END;
			}
			if (FD_ISSET(MASTER(backend)->fd, &exceptmask))
			{
				return POOL_FATAL;
			}

			if (FD_ISSET(frontend->fd, &readmask))
			{
				status = ProcessFrontendResponse(frontend, backend);
				if (status != POOL_CONTINUE)
					return status;

				continue;
			}
		}
		else
		{
			if (MASTER(backend)->len > 0)
			{
				pool_read(MASTER(backend), &kind, 1);
				if (REPLICATION)
				{
					pool_read(SECONDARY(backend), &kind1, 1);
					if (kind != kind1)
					{
						pool_error("pool_process_query: kind does not match between backends master(%c) secondary(%c)",
								   kind, kind1);
						return POOL_FATAL;
					}
				}
				pool_debug("read kind from backend pending data %c len: %d po: %d", kind, MASTER(backend)->len, MASTER(backend)->po);
			}
			if (frontend->len > 0)
			{
				status = ProcessFrontendResponse(frontend, backend);
				if (status != POOL_CONTINUE)
					return status;

				continue;
			}
		}

		/* this is the synchronous point */
		if (REPLICATION)
		{
			if (kind == 0)
			{
				pool_read(MASTER(backend), &kind, 1);
			}
			if (kind1 == 0)
			{
				pool_read(SECONDARY(backend), &kind1, 1);
			}
			if (kind != kind1)
			{
				pool_error("pool_process_query: kind does not match between backends master(%c) secondary(%c)",
						   kind, kind1);
				return POOL_FATAL;
			}
		}

		/* Prrocess backend Response */
		switch (kind)
		{
			case 'A':
				/* Notification  response */
				status = NotificationResponse(frontend, backend);
				break;

			case 'B':
				/* BinaryRow */
				status = BinaryRow(frontend, backend, num_fields);
				break;

			case 'C':
				/* Complete command response */
				status = CompleteCommandResponse(frontend, backend);
				break;

			case 'D':
				/* AsciiRow */
				status = AsciiRow(frontend, backend, num_fields);
				break;

			case 'E':
				/* Error Response */
				status = ErrorResponse(frontend, backend);
				break;

			case 'G':
				/* CopyIn Response */
				status = CopyInResponse(frontend, backend);
				break;

			case 'H':
				/* CopyOut Response */
				status = CopyOutResponse(frontend, backend);
				break;

			case 'I':
				/* Empty Query Response */
				status = EmptyQueryResponse(frontend, backend);
				break;

			case 'N':
				/* Notice Response */
				status = NoticeResponse(frontend, backend);
				break;

			case 'P':
				/* CursorResponse */
				status = CursorResponse(frontend, backend);
				break;

			case 'T':
				/* RowDescription */
				status = RowDescription(frontend, backend);
				if (status < 0)
					return POOL_ERROR;

				num_fields = status;
				status = POOL_CONTINUE;
				break;

			case 'V':
				/* FunctionResultResponse and FunctionVoidResponse */
				status = FunctionResultResponse(frontend, backend);
				break;
				
			case 'Z':
				/* Ready for query */
				status = ReadyForQuery(frontend, backend, 1);
				break;
				
			default:
				pool_error("Unknown message type %c(%02x)", kind, kind);
				exit(1);
		}
		if (status != POOL_CONTINUE)
			return status;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS Query(POOL_CONNECTION *frontend, 
						 POOL_CONNECTION_POOL *backend)
{
	char kind = 'Q';
	char *string;
	int len;
	static char *sq = "show pool_status";

	/* read actual query */
	string = pool_read_string(frontend, &len, 0);
	if (string == NULL)
		return POOL_END;

	/* process status reporting? */
	if (strncasecmp(sq, string, strlen(sq)) == 0)
	{
		pool_debug("process reporting");
		process_reporting(frontend);
		return POOL_CONTINUE;
	}

	/* forward the query to the backend */
	pool_write(MASTER(backend), &kind, 1);
	if (pool_write_and_flush(MASTER(backend), string, len) < 0)
	{
		return POOL_END;
	}

	if (REPLICATION)
	{
		/* in "strict mode" we need to wait for master completing the query */
		if (pool_config.replication_strict || STRICT_MODE(string))
			if (synchronize(MASTER(backend)))
				return POOL_END;

		pool_write(SECONDARY(backend), &kind, 1);
		if (pool_write_and_flush(SECONDARY(backend), string, len) < 0)
		{
			return POOL_END;
		}
	}
	return POOL_CONTINUE;
}

static POOL_STATUS ReadyForQuery(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION_POOL *backend, int send_ready)
{
	char kind;

	if (send_ready)
	{
		kind = 'Z';
		if (pool_write_and_flush(frontend, &kind, 1) < 0)
		{
			return POOL_END;
		}
	}

	return ProcessFrontendResponse(frontend, backend);
}

static POOL_STATUS CompleteCommandResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend)
{
	char *string, *string1;
	int len, len1;

	/* read command tag */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;

	if (REPLICATION)
	{
		string1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (string1 == NULL)
			return POOL_END;

		if (len != len1)
		{
			pool_log("Complete Command Response: message length does not match between master(%d \"%s\",) and secondary(%d \"%s\",)",
					 len, string, len1, string1);
		}
	}

	/* forward to the frontend */
	pool_write(frontend, "C", 1);
	pool_debug("Complete Command Response: string: \"%s\"", string);
	if (pool_write(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static int RowDescription(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION_POOL *backend)
{
	short num_fields, num_fields1;
	int oid, mod;
	int oid1, mod1;
	short size, size1;
	char *string, *string1;
	int len, len1;
	int i;

	/* # of fields (could be 0) */
	pool_read(MASTER(backend), &num_fields, sizeof(short));
	if (REPLICATION)
	{
		pool_read(SECONDARY(backend), &num_fields1, sizeof(short));
		if (num_fields != num_fields1)
		{
			pool_error("RowDescription: num_fields deos not match between backends master(%d) and secondary(%d)",
					   num_fields, num_fields1);
			return POOL_FATAL;
		}
	}

	/* forward it to the frontend */
	pool_write(frontend, "T", 1);
	pool_write(frontend, &num_fields, sizeof(short));

	num_fields = ntohs(num_fields);
	for (i = 0;i<num_fields;i++)
	{
		/* field name */
		string = pool_read_string(MASTER(backend), &len, 0);
		if (string == NULL)
			return POOL_END;

		if (REPLICATION)
		{
			string1 = pool_read_string(SECONDARY(backend), &len1, 0);
			if (string == NULL)
				return POOL_END;
			if (len != len1)
			{
				pool_error("RowDescription: field length deos not match between backends master(%d) and secondary(%d)",
						   ntohl(len), ntohl(len1));
				return POOL_FATAL;
			}
		}

		pool_write(frontend, string, len);

		/* oid */
		pool_read(MASTER(backend), &oid, sizeof(int));
		if (REPLICATION)
		{
			pool_read(SECONDARY(backend), &oid1, sizeof(int));

			/* we do not regard oid mismatch as fatal */
			if (oid != oid1)
			{
				pool_log("RowDescription: field oid deos not match between backends master(%d) and secondary(%d)",
						 ntohl(oid), ntohl(oid1));
			}
		}
		pool_write(frontend, &oid, sizeof(int));

		/* size */
		pool_read(MASTER(backend), &size, sizeof(short));
		if (REPLICATION)
		{
			pool_read(SECONDARY(backend), &size1, sizeof(short));
			if (size1 != size1)
			{
				pool_log("RowDescription: field size deos not match between backends master(%d) and secondary(%d)",
						 ntohs(size), ntohs(size1));
				return POOL_FATAL;
			}
		}
		pool_debug("RowDescription: field size:%d", ntohs(size));
		pool_write(frontend, &size, sizeof(short));

		/* modifier */
		pool_read(MASTER(backend), &mod, sizeof(int));
		if (REPLICATION)
		{
			pool_read(SECONDARY(backend), &mod1, sizeof(int));
			if (mod != mod1)
			{
				pool_log("RowDescription: modifier deos not match between backends master(%d) and secondary(%d)",
						 ntohl(mod), ntohl(mod1));
			}
		}
		pool_write(frontend, &mod, sizeof(int));
	}

#ifdef NOT_USED
	if (pool_flush(frontend) < 0)
	{
		return POOL_END;
	}
#endif

	return num_fields;
}

static POOL_STATUS AsciiRow(POOL_CONNECTION *frontend, 
							POOL_CONNECTION_POOL *backend,
							short num_fields)
{
	static char nullmap[8192];
	int nbytes;
	int i;
	unsigned char mask;
	int size, size1;
	char *buf, *buf1;
	char msgbuf[1024];

	pool_write(frontend, "D", 1);

	nbytes = (num_fields + 7)/8;

	if (nbytes <= 0)
		return POOL_CONTINUE;

	/* NULL map */
	pool_read(MASTER(backend), nullmap, nbytes);
	if (pool_write(frontend, nullmap, nbytes) < 0)
	{
		return POOL_END;
	}
	if (REPLICATION)
	{
		pool_read(SECONDARY(backend), nullmap, nbytes);
	}

	mask = 0;

	for (i = 0;i<num_fields;i++)
	{
		if (mask == 0)
			mask = 0x80;

		/* NOT NULL? */
		if (mask & nullmap[i/8])
		{
			/* field size */
			if (pool_read(MASTER(backend), &size, sizeof(int)) < 0)
				return POOL_END;
			if (REPLICATION)
			{
				/* XXX: field size maybe different between
				   backends. If we were a paranoid, we have to treat
				   this as a fatal error. However in the real world
				   we'd better to adapt this situation. Just throw a
				   log... */

				if (pool_read(SECONDARY(backend), &size1, sizeof(int)) < 0)
					return POOL_END;

				if (size != size1)
					pool_log("AsciiRow: %d th field size does not match between master(%d) and secondary(%d)",
							 i, ntohl(size), ntohl(size1));
				size1 = ntohl(size1) - 4;

			}

			pool_write(frontend, &size, sizeof(int));
			size = ntohl(size) - 4;

			/* read and send actual data only when size > 0 */
			if (size > 0)
			{
				buf = malloc(size);
				if (buf == NULL)
				{
					pool_error("AsciiRow: out of memory");
					return POOL_END;
				}
				/* actual data */
				if (pool_read(MASTER(backend), buf, size) < 0)
				{
					free(buf);
					return POOL_END;
				}
				if (REPLICATION && size1 > 0)
				{
					buf1 = malloc(size1);
					if (buf1 == NULL)
					{
						pool_error("AsciiRow: out of memory");
						return POOL_END;
					}
					if (pool_read(SECONDARY(backend), buf1, size1) < 0)
					{
						free(buf1);
						return POOL_END;
					}
					free(buf1);
				}
				pool_write(frontend, buf, size);
				snprintf(msgbuf, Min(sizeof(msgbuf), size), "%s", buf);
				free(buf);
			}
			else
			{
				*msgbuf = '\0';
			}
			pool_debug("AsciiRow: len:%d data: %s", size, msgbuf);
		}
		mask >>= 1;
	}

#ifdef NOT_USED
	if (pool_flush(frontend) < 0)
	{
		return POOL_END;
	}
#endif

	return POOL_CONTINUE;
}

static POOL_STATUS BinaryRow(POOL_CONNECTION *frontend, 
							 POOL_CONNECTION_POOL *backend,
							 short num_fields)
{
	static char nullmap[8192];
	int nbytes;
	int i;
	unsigned char mask;
	int size;
	char *buf;

	pool_write(frontend, "B", 1);

	nbytes = (num_fields + 7)/8;

	if (nbytes <= 0)
		return POOL_CONTINUE;

	/* NULL map */
	pool_read(MASTER(backend), nullmap, nbytes);
	if (REPLICATION)
		pool_read(SECONDARY(backend), nullmap, nbytes);

	if (pool_write(frontend, nullmap, nbytes) < 0)
	{
		return POOL_END;
	}

	mask = 0;

	for (i = 0;i<num_fields;i++)
	{
		if (mask == 0)
			mask = 0x80;

		/* NOT NULL? */
		if (mask & nullmap[i/8])
		{
			/* field size */
			pool_read(MASTER(backend), &size, sizeof(int));
			if (REPLICATION)
				pool_read(SECONDARY(backend), &size, sizeof(int));
			pool_write(frontend, &size, sizeof(int));
			size = ntohl(size);
			buf = malloc(size);
			if (buf == NULL)
			{
				pool_error("BinaryRow: out of memory");
				return POOL_END;
			}
			/* actual data */
			pool_read(MASTER(backend), buf, size);
			if (REPLICATION)
				pool_read(SECONDARY(backend), buf, size);
			pool_write(frontend, buf, size);
			free(buf);
		}
		mask >>= 1;
	}

	if (pool_flush(frontend) < 0)
	{
		return POOL_END;
	}

	return POOL_CONTINUE;
}

static POOL_STATUS CursorResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend)
{
	char *string, *string1;
	int len, len1;

	/* read cursor name */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		string1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (string1 == NULL)
			return POOL_END;
		if (len != len1)
		{
			pool_error("CursorResponse: length does not match between master(%d) and secondary(%d)",
					 len, len1);
			pool_error("CursorResponse: master(%s) secondary(%s)", string, string1);
			return POOL_END;
		}
	}

	/* forward to the frontend */
	pool_write(frontend, "P", 1);
	if (pool_write(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

POOL_STATUS ErrorResponse(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION_POOL *backend)
{
	char *string;
	int len;

	/* read error message */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		string = pool_read_string(SECONDARY(backend), &len, 0);
		if (string == NULL)
			return POOL_END;
	}

	/* forward to the frontend */
	pool_write(frontend, "E", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
		return POOL_END;
			
	return POOL_CONTINUE;
}

static POOL_STATUS NoticeResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend)
{
	char *string, *string1;
	int len, len1;

	/* read notice message */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		string1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (string1 == NULL)
			return POOL_END;
	}

	/* forward to the frontend */
	pool_write(frontend, "N", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS CopyInResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend)
{
	POOL_STATUS status;

	/* forward to the frontend */
	if (pool_write_and_flush(frontend, "G", 1) < 0)
	{
		return POOL_END;
	}
	status = CopyDataRows(frontend, backend, 1);
	return status;
}

static POOL_STATUS CopyOutResponse(POOL_CONNECTION *frontend, 
								   POOL_CONNECTION_POOL *backend)
{
	POOL_STATUS status;

	/* forward to the frontend */
	if (pool_write_and_flush(frontend, "H", 1) < 0)
	{
		return POOL_END;
	}
	status = CopyDataRows(frontend, backend, 0);
	return status;
}

static POOL_STATUS CopyDataRows(POOL_CONNECTION *frontend,
								POOL_CONNECTION_POOL *backend, int copyin)
{
	char *string;
	int len;

#ifdef DEBUG
	int i = 0;
	char buf[1024];
#endif

	for (;;)
	{
		if (copyin)
			string = pool_read_string(frontend, &len, 1);
		else
		{
			string = pool_read_string(MASTER(backend), &len, 1);
			if (REPLICATION)
				string = pool_read_string(SECONDARY(backend), &len, 1);
		}

		if (string == NULL)
			return POOL_END;

#ifdef DEBUG
		strncpy(buf, string, len);
		pool_debug("copy line %d %d bytes :%s:", i++, len, buf);
#endif

		if (copyin)
		{
			pool_write(MASTER(backend), string, len);
			if (REPLICATION)
				pool_write(SECONDARY(backend), string, len);
		}
		else
			pool_write(frontend, string, len);			

		if (len == 3)
		{
			/* end of copy? */
			if (string[0] == '\\' &&
				string[1] == '.' &&
				string[2] == '\n')
			{
				break;
			}
		}
	}

	if (copyin)
	{
		if (pool_flush(MASTER(backend)) <0)
			return POOL_END;
		if (REPLICATION)
		{
			if (pool_flush(SECONDARY(backend)) <0)
				return POOL_END;
		}
	}
	else
		if (pool_flush(frontend) <0)
			return POOL_END;

	return POOL_CONTINUE;
}

static POOL_STATUS EmptyQueryResponse(POOL_CONNECTION *frontend,
									  POOL_CONNECTION_POOL *backend)
{
	char c;

	if (pool_read(MASTER(backend), &c, sizeof(c)) < 0)
		return POOL_END;

	if (REPLICATION)
	{
		if (pool_read(SECONDARY(backend), &c, sizeof(c)) < 0)
			return POOL_END;
	}

	pool_write(frontend, "I", 1);
	return pool_write_and_flush(frontend, "", 1);
}

static POOL_STATUS NotificationResponse(POOL_CONNECTION *frontend, 
										POOL_CONNECTION_POOL *backend)
{
	int pid, pid1;
	char *condition, *condition1;
	int len, len1;

	pool_write(frontend, "A", 1);

	if (pool_read(MASTER(backend), &pid, sizeof(pid)) < 0)
		return POOL_ERROR;

	if (REPLICATION)
	{
		if (pool_read(SECONDARY(backend), &pid1, sizeof(pid1)) < 0)
			return POOL_ERROR;
	}

	condition = pool_read_string(MASTER(backend), &len, 0);
	if (condition == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		condition1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (condition1 == NULL)
			return POOL_END;
	}

	pool_write(frontend, &pid, sizeof(pid));

	return pool_write_and_flush(frontend, condition, len);
}

static POOL_STATUS FunctionCall(POOL_CONNECTION *frontend, 
								POOL_CONNECTION_POOL *backend)
{
	char dummy[2];
	int oid;
	int argn;
	int i;

	pool_write(MASTER(backend), "F", 1);
	if (REPLICATION)
		pool_write(SECONDARY(backend), "F", 1);

	/* dummy */
	if (pool_read(frontend, dummy, sizeof(dummy)) < 0)
		return POOL_ERROR;
	pool_write(MASTER(backend), dummy, sizeof(dummy));
	if (REPLICATION)
		pool_write(SECONDARY(backend), dummy, sizeof(dummy));

	/* function object id */
	if (pool_read(frontend, &oid, sizeof(oid)) < 0)
		return POOL_ERROR;

	pool_write(MASTER(backend), &oid, sizeof(oid));
	if (REPLICATION)
		pool_write(SECONDARY(backend), &oid, sizeof(oid));

	/* number of arguments */
	if (pool_read(frontend, &argn, sizeof(argn)) < 0)
		return POOL_ERROR;
	pool_write(MASTER(backend), &argn, sizeof(argn));
	if (REPLICATION)
		pool_write(SECONDARY(backend), &argn, sizeof(argn));

	argn = ntohl(argn);

	for (i=0;i<argn;i++)
	{
		int len;
		char *arg;

		/* length of each argument in bytes */
		if (pool_read(frontend, &len, sizeof(len)) < 0)
			return POOL_ERROR;

		pool_write(MASTER(backend), &len, sizeof(len));
		if (REPLICATION)
			pool_write(SECONDARY(backend), &len, sizeof(len));

		len = ntohl(len);

		arg = malloc(len);
		if (arg == NULL)
		{
			pool_error("FuncationCall: out of memory");
			return POOL_ERROR;
		}

		/* argument value itself */
		if (pool_read(frontend, arg, len) < 0)
			return POOL_ERROR;
		pool_write(MASTER(backend), arg, len);
		if (REPLICATION)
			pool_write(SECONDARY(backend), arg, len);

		free(arg);
	}

	if (pool_flush(MASTER(backend)))
		return POOL_ERROR;
	if (REPLICATION)
		if (pool_flush(SECONDARY(backend)))
			return POOL_ERROR;
	return POOL_CONTINUE;
}

static POOL_STATUS FunctionResultResponse(POOL_CONNECTION *frontend, 
										  POOL_CONNECTION_POOL *backend)
{
	char dummy;
	int len;
	char *result;

	pool_write(frontend, "V", 1);

	if (pool_read(MASTER(backend), &dummy, 1) < 0)
		return POOL_ERROR;
	if (REPLICATION)
		if (pool_read(SECONDARY(backend), &dummy, 1) < 0)
			return POOL_ERROR;

	pool_write(frontend, &dummy, 1);

	/* non empty result? */
	if (dummy == 'G')
	{
		/* length of result in bytes */
		if (pool_read(MASTER(backend), &len, sizeof(len)) < 0)
			return POOL_ERROR;
		if (REPLICATION)
			if (pool_read(SECONDARY(backend), &len, sizeof(len)) < 0)
				return POOL_ERROR;

		pool_write(frontend, &len, sizeof(len));

		len = ntohl(len);

		result = malloc(len);
		if (result == NULL)
		{
			pool_error("FuncationResultResponse: out of memory");
			return POOL_ERROR;
		}

		/* result value itself */
		if (pool_read(MASTER(backend), result, len) < 0)
			return POOL_ERROR;
		if (REPLICATION)
			if (pool_read(SECONDARY(backend), result, len) < 0)
				return POOL_ERROR;

		pool_write(frontend, result, len);
		free(result);
	}

	/* unused ('0') */
	if (pool_read(MASTER(backend), &dummy, 1) < 0)
		return POOL_ERROR;
	if (REPLICATION)
		if (pool_read(SECONDARY(backend), &dummy, 1) < 0)
			return POOL_ERROR;

	pool_write(frontend, "0", 1);

	return pool_flush(frontend);
}

static POOL_STATUS ProcessFrontendResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend)
{
	char fkind;
	POOL_STATUS status;

	if (pool_read(frontend, &fkind, 1) < 0)
	{
		pool_error("ProcessFrontendResponse: failed to read kind");
		return POOL_END;
	}

	pool_debug("read kind from frontend %c(%02x)", fkind, fkind);

	switch (fkind)
	{
		case 'X':
			status = POOL_END;
			break;

		case 'Q':
			status = Query(frontend, backend);
			break;

		case 'F':
			status = FunctionCall(frontend, backend);
			break;

		default:
			pool_error("ProcessFrontendResponse: unknown message type %c(%02x)", fkind, fkind);
			status = POOL_ERROR;
			break;
	}

	return status;
}

static int timeoutmsec;

/*
 * enable read timeout
 */
void pool_enable_timeout()
{
	timeoutmsec = pool_config.replication_timeout;
}

/*
 * disable read timeout
 */
void pool_disable_timeout()
{
	timeoutmsec = 0;
}

/*
 * wait until read data is ready
 */
static int synchronize(POOL_CONNECTION *cp)
{
	return pool_check_fd(cp, 1);
}

/*
 * wait until read data is ready
 * if notimeout is non 0, wait forever.
 */
int pool_check_fd(POOL_CONNECTION *cp, int notimeout)
{
	fd_set readmask;
	fd_set exceptmask;
	int fd;
	int fds;
	struct timeval timeout;
	struct timeval *tp;

	fd = cp->fd;

	for (;;)
	{
		FD_ZERO(&readmask);
		FD_ZERO(&exceptmask);
		FD_SET(fd, &readmask);
		FD_SET(fd, &exceptmask);

		if (notimeout || timeoutmsec == 0)
			tp = NULL;
		else
		{
			timeout.tv_sec = 0;
			timeout.tv_usec = pool_config.replication_timeout*1000;
			tp = &timeout;
		}

		fds = select(fd+1, &readmask, NULL, &exceptmask, tp);

		if (fds == -1)
		{
			if (errno == EAGAIN || errno == EINTR)
				continue;

			pool_error("pool_check_fd: select() failed. reason %s", strerror(errno));
			break;
		}

		if (FD_ISSET(fd, &exceptmask))
		{
			pool_error("pool_check_fd: exception occurred");
			break;
		}

		if (fds == 0)
		{
			pool_error("pool_check_fd: data is not ready tp->tv_sec %d tp->tp_usec %d", tp->tv_sec, tp->tv_usec);
			break;
		}
		return 0;
	}
	return -1;
}

static void process_reporting(POOL_CONNECTION *frontend)
{
	static char *cursorname = "blank";
	static short num_fields = 3;
	static char *field_names[] = {"item", "value", "description"};
	static int oid = 0;
	static short fsize = -1;
	static int mod = 0;
	short n;
	int i;
	short s;

	static char nullmap[2] = {0xff, 0xff};
	int nbytes = (num_fields + 7)/8;

#define MAXVALLEN 512

	typedef struct {
		char *name;
		char value[MAXVALLEN+1];
		char *desc;
	} POOL_REPORT_STATUS;

#define MAXITEMS 128

	POOL_REPORT_STATUS status[MAXITEMS];

	int nrows;
	int size;
	int hsize;

	i = 0;

	status[i].name = "inetdomain";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.inetdomain);
	status[i].desc = "1 if accepting TCP/IP connection";
	i++;

	status[i].name = "port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.port);
	status[i].desc = "pgpool accepting port number";
	i++;

	status[i].name = "socket_dir";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.socket_dir);
	status[i].desc = "pgpool socket directory";
	i++;

	status[i].name = "backend_host_name";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.backend_host_name);
	status[i].desc = "master backend host name";
	i++;

	status[i].name = "backend_port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.backend_port);
	status[i].desc = "master backend port number";
	i++;

	status[i].name = "secondary_backend_host_name";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.secondary_backend_host_name);
	status[i].desc = "secondary backend host name";
	i++;

	status[i].name = "secondary_backend_port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.secondary_backend_port);
	status[i].desc = "secondary backend port number";
	i++;

	status[i].name = "num_init_children";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.num_init_children);
	status[i].desc = "# of children initially pre-forked";
	i++;

	status[i].name = "child_life_time";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.child_life_time);
	status[i].desc = "if idle for this seconds, child exits (not implemented yet)";
	i++;

	status[i].name = "connection_life_time";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.connection_life_time);
	status[i].desc = "if idle for this seconds, connection closes";
	i++;

	status[i].name = "max_pool";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.max_pool);
	status[i].desc = "max # of connection pool per child";
	i++;

	status[i].name = "logdir";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.logdir);
	status[i].desc = "logging directory";
	i++;

	status[i].name = "backend_socket_dir";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.backend_socket_dir);
	status[i].desc = "Unix domain socket directory for the PostgreSQL server";
	i++;

	status[i].name = "replication_mode";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_mode);
	status[i].desc = "non 0 if operating in replication mode";
	i++;

	status[i].name = "replication_strict";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_strict);
	status[i].desc = "non 0 if operating in strict mode";
	i++;

	status[i].name = "replication_timeout";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_timeout);
	status[i].desc = "if secondary does not respond in this milli seconds, abort the session";
	i++;

	status[i].name = "current_backend_host_name";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.current_backend_host_name);
	status[i].desc = "current master host name";
	i++;

	status[i].name = "current_backend_port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.current_backend_port);
	status[i].desc = "current master port #";
	i++;

	status[i].name = "replication_enabled";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_enabled);
	status[i].desc = "non 0 if operating in replication mode";
	i++;

	nrows = i;

	/* cursor response */
	pool_write(frontend, "P", 1);
	pool_write(frontend, cursorname, strlen(cursorname)+1);

	/* row description */
	pool_write(frontend, "T", 1);
	n = htons(num_fields);
	pool_write(frontend, &n, sizeof(short));
	for (i=0;i<num_fields;i++)
	{
		char *f = field_names[i];
		pool_write(frontend, f, strlen(f)+1);

		pool_write(frontend, &oid, sizeof(oid));
		s = htons(fsize);
		pool_write(frontend, &s, sizeof(fsize));
		pool_write(frontend, &mod, sizeof(mod));
	}
	pool_flush(frontend);

	/* ascii row */
	for (i=0;i<nrows;i++)
	{
		pool_write(frontend, "D", 1);
		pool_write_and_flush(frontend, nullmap, nbytes);

		size = strlen(status[i].name);
		hsize = htonl(size+4);
		pool_write(frontend, &hsize, sizeof(hsize));
		pool_write(frontend, status[i].name, size);

		size = strlen(status[i].value);
		hsize = htonl(size+4);
		pool_write(frontend, &hsize, sizeof(hsize));
		pool_write(frontend, status[i].value, size);

		size = strlen(status[i].desc);
		hsize = htonl(size+4);
		pool_write(frontend, &hsize, sizeof(hsize));
		pool_write(frontend, status[i].desc, size);
	}

	/* complete command response */
	pool_write(frontend, "C", 1);
	pool_write(frontend, "SELECT", strlen("SELECT")+1);

	/* ready for query */
	pool_write(frontend, "Z", 1);

	pool_flush(frontend);
}

void pool_send_frontend_exits(POOL_CONNECTION_POOL *backend)
{
	pool_write(MASTER(backend), "X", 1);

	/*
	 * XXX we cannot call pool_flush() here since backend may already
	 * close the socket and pool_flush() automatically invokes fail
	 * over handler. This could happen in copy command (remember the
	 * famouse "lostsynchronization with server, resettin g
	 * connection" message)
	 */
	fflush(MASTER(backend)->write_fd);

	if (REPLICATION)
	{
		pool_write(SECONDARY(backend), "X", 1);
		fflush(SECONDARY(backend)->write_fd);
	}
}
