#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <linux/ax25.h>

#include "config.h"
#include "node.h"
#include "io.h"
#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"
#include "procutils.h"
#include "version.h"
#include "mheard.h"

struct cmd Nodecmds[] = {
	{ "?",		do_help		},
	{ "Bye",	do_bye		},
	{ "Connect",	do_connect	},
	{ "CAllbook",	do_finger	},
	{ "Finger",	do_finger	},
	{ "Help",	do_help		},
	{ "Info",	do_help		},
	{ "Links",	do_links	},
	{ "Mheard",	do_mheard	},
	{ "Nodes",	do_nodes	},
	{ "Ports",	do_ports	},
	{ "PIng",	do_ping		},
	{ "Routes",	do_routes	},
	{ "Telnet",	do_connect	},
	{ "Users",	user_list	},
	{ NULL,		NULL		}
};

int do_bye(int argc, char **argv)
{
	end_io(User.fd);
	close(User.fd);
	logout_user();
	exit(0);
}

struct mheard_list {
	struct mheard_struct	data;
	struct mheard_list	*next;
};

int do_mheard(int argc, char **argv)
{
	FILE *fp;
        struct mheard_struct mh;
        struct mheard_list *list, *new, *tmp, *p;
	char *s, *t, *u;

	if (argc < 2) {
		tprintf("%s} Usage: mheard <port>\n", NrId);
		return 0;
	}
	if (ax25_config_get_dev(argv[1]) == NULL ||
	    (check_perms(PERM_HIDDEN, 0) == -1 && is_hidden(argv[1]))) {
		tprintf("%s} Invalid port\n", NrId);
		return 0;
	}
        if ((fp = fopen(DATA_MHEARD_FILE, "r")) == NULL) {
		node_perror(DATA_MHEARD_FILE, errno);
		return 0;
	}
	list = NULL;
	while (fread(&mh, sizeof(struct mheard_struct), 1, fp) == 1) {
		if (strcmp(argv[1], mh.portname))
			continue;
                if ((new = calloc(1, sizeof(struct mheard_list))) == NULL) {
			node_perror("calloc", errno);
			break;
		}
		new->data = mh;
		if (list == NULL || mh.last_heard > list->data.last_heard) {
                        tmp = list;
                        list = new;
                        new->next = tmp;
		} else {
                        for (p = list; p->next != NULL; p = p->next)
                                if (mh.last_heard > p->next->data.last_heard)
                                        break;
                        tmp = p->next;
                        p->next = new;
                        new->next = tmp;
		}
	}
	fclose(fp);
	tprintf("%s} Heard list for port %s:\n", NrId, argv[1]);
	tprintf("Callsign  Pkts rcvd Last heard\n");
	while (list != NULL) {
		s = ctime(&list->data.last_heard);
		s[19] = '\0';
		t = ax2asc(&list->data.from_call);
		if ((u = strstr(t, "-0")) != NULL)
			*u = '\0';
		tprintf("%-9s %-9ld %s\n", t, list->data.count, s);
		tmp = list;
		list = list->next;
		free(tmp);
	}
	return 0;
}

int do_help(int argc, char **argv)
{
	FILE *fp;
	char fname[80], line[256];
	struct cmd *cmdp;
	struct alias *ap;
	struct ecmd *ep;
	int i = 0;

	if (*argv[0] == '?') {				/* "?"		*/
		tprintf("%s} Commands:\n", NrId);
		for (ap = Aliases; ap != NULL; ap = ap->next) {
			tprintf("%s%s", i ? ", " : "", strupr(ap->name));
			if (++i == 10) {
				tprintf("\n");
				i = 0;
			}
		}
		for (ep = Extcmds; ep != NULL; ep = ep->next) {
			tprintf("%s%s", i ? ", " : "", strupr(ep->name));
			if (++i == 10) {
				tprintf("\n");
				i = 0;
			}
		}
		for (cmdp = Nodecmds; cmdp->name != NULL; cmdp++) {
			tprintf("%s%s", i ? ", " : "", cmdp->name);
			if (++i == 10) {
				tprintf("\n");
				i = 0;
			}
		}
		if (i) tprintf("\n");
		return 0;
	}
	strcpy(fname, DATA_NODE_HELP_DIR);
	if (*argv[0] == 'i') {				/* "info"	*/
		strcat(fname, "info.hlp");
		tprintf("%s} %s (%s)\n", NrId, VERSION, version);
	} else if (argc == 1) {				/* "help"	*/
		strcat(fname, "help.hlp");
	} else {					/* "help <cmd>"	*/
		if (strchr(argv[1], '/') == NULL) {
			strlwr(argv[1]);
			strcat(fname, argv[1]);
			strcat(fname, ".hlp");
		}
	}
	if ((fp = fopen(fname, "r")) == NULL) {
		if (*argv[0] != 'i')
			tprintf("%s} No help for command %s\n", NrId, argv[1] ? argv[1] : "help");
		return 0;
	}
	if (*argv[0] != 'i')
		tprintf("%s} Help for command %s\n", NrId, argv[1] ? argv[1] : "help");
	while (fgets(line, 256, fp) != NULL)
		usputs(line, User.fd);
	usputs("\n-- \n", User.fd);
	fclose(fp);
	return 0;
}

int do_ports(int argc, char **argv)
{
	char *cp = NULL;

	tprintf("%s} Ports:\nPort   Description\n", NrId);
	while ((cp = ax25_config_get_next(cp)) != NULL) {
		if (is_hidden(cp) && check_perms(PERM_HIDDEN, 0L) == -1)
			continue;
		tprintf("%-6s %s\n", cp, ax25_config_get_desc(cp));
	}
	return 0;
}

int do_links(int argc, char **argv)
{
	struct proc_ax25 *p, *list;
	char *cp;

	tprintf("%s} AX.25 Link Status:\n", NrId);
	if ((list = read_proc_ax25()) == NULL) {
		if (errno)
			node_perror("read_proc_ax25():", errno);
		return 0;
	}
	tprintf("Port   Dest addr Src addr  State        Unack T1      Retr  Rtt Snd-Q Rcv-Q\n");
	for (p = list; p != NULL; p = p->next) {
		cp = ax25_config_get_name(p->dev);
		if (is_hidden(cp) && check_perms(PERM_HIDDEN, 0L) == -1)
			continue;
		tprintf("%-6s %-9s %-9s ", cp, p->dest_addr, p->src_addr);
		if (!strcmp(p->dest_addr, "*")) {
			tprintf("Listening\n");
			continue;
		}
		switch (p->st) {
		case 0:
			cp = "Disconnected";
			break;
		case 1:
			cp = "Conn pending";
			break;
		case 2:
			cp = "Disc pending";
			break;
		case 3:
			cp = "Connected   ";
			break;
		case 4:
			cp = "Recovery    ";
			break;
		default:
			cp = "Unknown     ";
			break;
		}
		tprintf("%s %02d/%02d %03d/%03d %02d/%02d %-3d %-5d %-5d\n",
			cp,
			p->vs < p->va ? p->vs - p->va + 8 : p->vs - p->va,
			p->window,
			p->t1timer, p->t1,
			p->n2count, p->n2,
			p->rtt,
			p->sndq, p->rcvq);
	}
	free_proc_ax25(list);
	return 0;
}

int do_routes(int argc, char **argv)
{
	struct proc_nr_neigh *p, *list;
	struct proc_ax25 *ap;
	char *cp;
	int link;

	tprintf("%s} Routes:\nLink Port   Callsign  Quality Destinations Lock\n", NrId);
	if ((list = read_proc_nr_neigh()) == NULL) {
		if (errno)
			node_perror("read_proc_nr_neigh()", errno);
		return 0;
	}
	for (p = list; p != NULL; p = p->next) {
		cp = ax25_config_get_name(p->dev);
		if (is_hidden(cp) && check_perms(PERM_HIDDEN, 0L) == -1)
			continue;
		if ((ap = find_link(nr_config_get_addr(NULL), p->call, p->dev)) != NULL && ap->st >= 3)
			link = 1;
		else
			link = 0;
		tprintf("%c    %-6s %-9s %-7d %-12d %c\n",
			link == 1 ? '>' : ' ',
			cp,
			p->call,
			p->qual,
			p->cnt,
			p->lock == 1 ? '!' : ' ');
	}
	free_proc_nr_neigh(list);
	return 0;
}

int do_nodes(int argc, char **argv)
{
	struct proc_nr_nodes *p, *list;
	struct proc_nr_neigh *np, *nlist;
	int i = 0;

	if ((list = read_proc_nr_nodes()) == NULL) {
		if (errno)
			node_perror("read_proc_nr_nodes()", errno);
		else
			tprintf("%s} No known nodes\n", NrId);
		return 0;
	}
	/* "nodes" */
	if (argc == 1) {
		tprintf("%s} Nodes:\n", NrId);
		for (p = list; p != NULL; p = p->next) {
			tprintf("%-16.16s %c",
				print_node(p->alias, p->call),
				(++i % 4) ? ' ' : '\n');
		}
		if ((i % 4) != 0) tprintf("\n");
		free_proc_nr_nodes(list);
		return 0;
	}
	if ((nlist = read_proc_nr_neigh()) == NULL) {
		node_perror("read_proc_nr_neigh()", errno);
		return 0;
	}
	/* "nodes *" */
	if (*argv[1] == '*') {
		tprintf("%s} Nodes:\n", NrId);
		tprintf("Node              Quality Obsolescence Port   Neighbour\n");
		for (p = list; p != NULL; p = p->next) {
			tprintf("%-16.16s  ", print_node(p->alias, p->call));
			if ((np = find_neigh(p->addr1, nlist)) != NULL) {
				tprintf("%-7d %-12d %-6s %s\n",
					p->qual1, p->obs1,
					ax25_config_get_name(np->dev),
					np->call);
			}
			if (p->n > 1 && (np = find_neigh(p->addr2, nlist)) != NULL) {
				tprintf("                  ");
				tprintf("%-7d %-12d %-6s %s\n",
					p->qual2, p->obs2,
					ax25_config_get_name(np->dev),
					np->call);
			}
			if (p->n > 2 && (np = find_neigh(p->addr3, nlist)) != NULL) {
				tprintf("                  ");
				tprintf("%-7d %-12d %-6s %s\n",
					p->qual3, p->obs3,
					ax25_config_get_name(np->dev),
					np->call);
			}
		}
		free_proc_nr_nodes(list);
		free_proc_nr_neigh(nlist);
		return 0;
	}
	/* "nodes <node>" */
	p = find_node(argv[1], list);
	if (p != NULL) {
		tprintf("%s} Routes to: %s\n", NrId, print_node(p->alias, p->call));
		tprintf("Which Quality Obsolescence Port   Neighbour\n");
		if ((np = find_neigh(p->addr1, nlist)) != NULL) {
			tprintf("%c     %-7d %-12d %-6s %s\n",
				p->w == 1 ? '>' : ' ',
				p->qual1, p->obs1,
				ax25_config_get_name(np->dev),
				np->call);
		}
		if (p->n > 1 && (np = find_neigh(p->addr2, nlist)) != NULL) {
			tprintf("%c     %-7d %-12d %-6s %s\n",
				p->w == 2 ? '>' : ' ',
				p->qual2, p->obs2,
				ax25_config_get_name(np->dev),
				np->call);
		}
		if (p->n > 1 && (np = find_neigh(p->addr3, nlist)) != NULL) {
			tprintf("%c     %-7d %-12d %-6s %s\n",
				p->w == 3 ? '>' : ' ',
				p->qual3, p->obs3,
				ax25_config_get_name(np->dev),
				np->call);
		}
	} else {
		tprintf("%s} No such node\n", NrId);
	}
	free_proc_nr_nodes(list);
	free_proc_nr_neigh(nlist);
	return 0;
}
