
/******************************************************
 *                                                    *
 * FPAC project.            FPAC PAD                  *
 *                                                    *
 * Parts of code from different sources of ax25-utils *
 *                                                    *
 * F6FBB 05-1997                                      *
 *                                                    *
 ******************************************************/

/******************************************************
 * 12/05/97 1.00 F6FBB First draft !
 *
 ******************************************************/
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <ctype.h>
#include <stdarg.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "axutils.h"
#include "axconfig.h"
#include "daemon.h"

/* Variables are not extern in main */
#define extern

#include "fpad.h"
#include "wp.h"

#define	AX25_HBIT 0x80

#define CPROGRESS 1
#define CONNECTED 2
#define	WAITCALL  3

int log_valid = TRUE;

user_t *head = NULL;

/*** Prototypes *******************/
static char *reason(unsigned char cause, unsigned char diagnostic);
static int new_alias_connection(alias_t *a, int verbose);
static int new_l2_connection(int fd, int verbose);
static int new_l3_connection(int fd, char *port, int verbose);
static int new_appli_connection(int fd, int accepted, char *appli);
static int new_socket(int type, char *callsign, char *addr, char *port, int digi, int pid);
static int rose_add(char *str);
static void end(user_t *p);
#ifdef __TCP__
static int new_tcp_connection(int fd, int verbose);
static int get_tcp_callsign(user_t *pl, char *str, int len);
#endif

#define BUFLEN 4096
#define MAXCOVER 11

static void sighup_handler(int sig)
{
	if (conf_changed())
		exit(99);
	signal(SIGHUP, sighup_handler);
}

#ifdef __TCP__
static int msg(user_t *u, char *fmt, ...)
{
	char buf[512];
	char *ptr = buf;
	int len;
	va_list args;
	
	if ((u == NULL) || (u->fd == -1))
		return 0;

	va_start(args, fmt);
	vsprintf(buf+1, fmt, args);
	va_end(args);

	len = strlen(buf+1);

	switch(u->type)
	{
	case AF_INET:
		ptr = buf+1;
		break;

	case AF_AX25:
		buf[0] = 0xf0;
		ptr = buf;
		len += 1;
		break;

	case AF_ROSE:
		buf[0] = '\0';
		ptr = buf;
		len += 1;
		break;

	}

	write(u->fd, ptr, len);

	return 1;
}
#endif

int main(int argc, char **argv)
{
	int i;
	int n;
	int daemon = 1;
	int verbose = 1;
	int maxfd;
	int fd_node[MAXCOVER];
/*	int fd_appli; */
#ifdef __TCP__
	int fd_tcp = -1;
#endif
	long lg;
	long size;
	user_t *u;
	alias_t *a;
	port_t *p;
	addrp_t *d;
	cover_t *o;
	fd_set fdread, fdwrite;
#define BUFSIZE 4096
	char buffer[BUFSIZE];
	struct timeval timeval;

	openlog("fpad", LOG_PID, LOG_DAEMON);

	syslog(LOG_INFO, "starting");

	if ((argc == 2) && (strcmp(argv[1], "-t") == 0))
		daemon = 0;

	/* No signal ! */
	for (i = 1 ; i < NSIG ; i++)
	{
		if ((i == SIGABRT) || (i == SIGINT))
			continue;
		else if (i == SIGHUP)
			signal(SIGHUP, sighup_handler);
		else
			signal(i, SIG_IGN);
	}

	/* Load AX25 L2 configuration */
	if (ax25_config_load_ports() == 0)
	{
		fprintf(stderr, "problem with axports file\n");
		syslog(LOG_ERR, "problem with axports file\n");
		closelog();
		return(1);
	}

	/* Read FPAC configuration file */
	if (read_conf(0) == 0)
	{
		fprintf(stderr, "problem with fpac.conf file\n");
		syslog(LOG_ERR, "problem with fpac.conf file\n");
		closelog();
		return(1);
	}

#ifdef __TCP__
	/* Open TCPIP socket */
	if (cfg.inetport)
		fd_tcp = new_socket(AF_INET, NULL, NULL, NULL, FALSE, FALSE);
#endif

	/* Open AX25 L2 sockets */
	for (p = cfg.port ; (p != NULL) ; p = p->next)
	{
		/* Open an L2 digi socket with the FPAC main callsign */
		p->fd_call = new_socket(AF_AX25, cfg.callsign, NULL, p->name, TRUE, TRUE);

		/* Open an L2 digi socket with the FPAC digi callsign */
		p->fd_digi = new_socket(AF_AX25, cfg.alt_callsign, NULL, p->name, TRUE, TRUE);

		/* Open an L2 application socket with the FPAC digi callsign */
		p->fd_appli = new_socket(AF_AX25, cfg.alt_callsign, NULL, p->name, FALSE, FALSE);

		/* Open L2 aliases sockets */
		for (a = p->alias ; a != NULL ; a = a->next)
			a->fd = new_socket(AF_AX25, a->alias, NULL, p->name, FALSE, TRUE);
	}

	/* Open AX25 L3 sockets */

	/* fd_appli = -1; */

	for (i = 0 ; i < MAXCOVER ; i++)
		fd_node[i] = -1;

	if (l3_attach(cfg.fulladdr, verbose) == 0)
		return(1);

	fd_node[0] = new_socket(AF_ROSE, NULL, cfg.fulladdr, "rose0", TRUE, TRUE);

	/*
	fpacnode application, callsign = NODE
	fd_appli[0] = new_socket(AF_ROSE, "NODE", cfg.fulladdr, "rose0", TRUE, FALSE);

	fpacnode application, callsign = digi
	fd_appli[1] = new_socket(AF_ROSE, fpac_digi, node_addr, "rose0", TRUE, FALSE);
	*/

	/* Open AX25 L3 sockets for AddPorts */
	for (d = cfg.addrp ; (d != NULL) ; d = d->next)
	{
		char *device;
		char address[11];

		strcpy(address, cfg.fulladdr);
		memcpy(address+4, d->addr, 6);

		device = l3_attach(address, verbose);
		if (device == NULL)
		{
			return(1);
		}
		d->fd = new_socket(AF_ROSE, NULL, address, device, TRUE, TRUE);
	}

	/* Open AX25 L3 sockets for coverage */
	for (i = 1, o = cfg.cover ; o != NULL ; i++, o = o->next)
	{
		char *device;
		char address[11];

		if (i == MAXCOVER)
			break;
			
		strcpy(address, cfg.fulladdr);
		memcpy(address+4, o->addr, 6);

		device = l3_attach(address, verbose);
		if (device == NULL)
		{
			return(1);
		}
		fd_node[i] = new_socket(AF_ROSE, NULL, address, device, TRUE, TRUE);
	}

	/* Get the size of one outq buffer */
	ioctl(fd_node[0], TIOCOUTQ, &size);

	size = BUFLEN;


	if ((daemon) && (!daemon_start(TRUE)) )
	{
		fprintf(stderr, "fpad: cannot become a daemon\n");
		closelog();
		return 1;
	}

	/* Read WP information */
	if (wp_open("FPAD"))
	{
		fprintf(stderr, "Cannot open WP service\n");
		syslog(LOG_ERR, "Cannot open WP service\n");
		closelog();
		return(1);
	}
	
	for (;;) 
	{
		user_t *prev = NULL;

		maxfd = -1;
		FD_ZERO(&fdread);
		FD_ZERO(&fdwrite);

#ifdef __TCP__
		if (fd_tcp != -1)
		{
			FD_SET(fd_tcp, &fdread);
			if (fd_tcp > maxfd)
				maxfd = fd_tcp;
		}
#endif

		for (p = cfg.port ; p != NULL ; p = p->next)
		{
			FD_SET(p->fd_call, &fdread);
			if (p->fd_call > maxfd)
				maxfd = p->fd_call;

			FD_SET(p->fd_digi, &fdread);
			if (p->fd_digi > maxfd)
				maxfd = p->fd_digi;

			FD_SET(p->fd_appli, &fdread);
			if (p->fd_appli > maxfd)
				maxfd = p->fd_appli;

			for (a = p->alias ; a != NULL ; a = a->next)
			{
				FD_SET(a->fd, &fdread);
				if (a->fd > maxfd)
					maxfd = a->fd;
			}
		}

		for (d = cfg.addrp ; d != NULL ; d = d->next)
		{
			FD_SET(d->fd, &fdread);
			if (d->fd > maxfd)
				maxfd = d->fd;
		}

		for (i = 0 ; i < MAXCOVER ; i++)
		{
			if (fd_node[i] == -1)
				break;
			FD_SET(fd_node[i], &fdread);
			if (fd_node[i] > maxfd)
				maxfd = fd_node[i];
		}

		/* if (fd_appli != -1)
		{
			FD_SET(fd_appli, &fdread);
			if (fd_appli > maxfd)
				maxfd = fd_appli;
		} */

		for (u = head ; u != NULL ; )
		{
			if (u->fd == -1)
			{
				/* Remove from the list */
				if (prev == NULL)
				{
					head = u->next;
					free(u);
					u = head;
				}
				else
				{
					prev->next = u->next;
					free(u);
					u = prev->next;
				}
			}
			else
			{
				if (u->state == WAITCALL)
				{
					/* Wait for callsign */
					FD_SET(u->fd, &fdread);
				}

				/* read if output buffer available in u->peer */
				if ((u->peer) && (u->peer->fd != -1) && (u->peer->state == CONNECTED))
				{
					if (u->peer->queue < (size - 2048))
					{
						FD_SET(u->fd, &fdread);
					}
				}

				/* Wait for connection */
				if (u->state == CPROGRESS)
					FD_SET(u->fd, &fdwrite);

				if (u->queue > 2048)
					FD_SET(u->fd, &fdwrite);

				if (u->fd > maxfd)
					maxfd = u->fd;
				prev = u;
				u = u->next;
			}
		}

		/*
		 *
		 * HERE IS THE SELECT !!!
		 *
		 */

		timeval.tv_usec = 0;
		timeval.tv_sec  = 60;
		n = select(maxfd + 1, &fdread, &fdwrite, NULL, &timeval);
		
		if (n < 0)
			continue;

		if (n == 0)		/* Timeout */
		{
			/* check_transits_wp(); */
			continue;
		}
#ifdef __TCP__
		/* Check for new connections on tcpip port */
		if (FD_ISSET(fd_tcp, &fdread))
		{
fprintf(stderr, "Connection on AF_INET (%d)\n", fd_tcp);
			new_tcp_connection(fd_tcp, TRUE);
		}
#endif

		/* Check for new connections on all L2 ports */
		for (p = cfg.port ; (p != NULL) ; p = p->next)
		{
			if (FD_ISSET(p->fd_call, &fdread)) 
			{
				new_l2_connection(p->fd_call, TRUE);
			}

			if (FD_ISSET(p->fd_digi, &fdread)) 
			{
				new_l2_connection(p->fd_digi, FALSE);
			}

			if (FD_ISSET(p->fd_appli, &fdread)) 
			{
				new_appli_connection(p->fd_appli, FALSE, "fpacnode");
			}

			/* Aliases */
			for (a = p->alias ; a != NULL ; a = a->next)
			{
				if (FD_ISSET(a->fd, &fdread)) 
					new_alias_connection(a, TRUE);
			}
		}

		/* Check for new connections on all L3 ports Coverage*/
		for (i = 0 ; i < MAXCOVER ; i++)
		{
			if (fd_node[i] == -1)
				break;
			if (FD_ISSET(fd_node[i], &fdread)) 
			{
				new_l3_connection(fd_node[i], NULL, TRUE);
			}
		}

		for (d = cfg.addrp ; (d != NULL) ; d = d->next)
		{
			if (FD_ISSET(d->fd, &fdread)) 
			{
				new_l3_connection(d->fd, d->port, TRUE);
			}
		}

		/*
		if (FD_ISSET(fd_appli, &fdread)) 
		{
			new_node_connection(fd_appli);
		}
		*/

		/* data to be read */
		for (u = head ; u != NULL ; u = u->next)
		{
			int connection_done = 0;
			int len = 0;
			
			if ((u->fd != -1) && (FD_ISSET(u->fd, &fdwrite)))
			{
				if (u->state == CPROGRESS)
				{
					connection_done = 1;
					u->state = CONNECTED;
				}
			}

			if ((u->fd != -1) && (FD_ISSET(u->fd, &fdread)))
			{
				len = read(u->fd, buffer+2, sizeof(buffer)-2);
				if (len <= 0)
				{
					connection_done = 0;
					len = -1;
				}
			}

			if (connection_done)
			{
				if (u->type == AF_ROSE)
				{
					int addrlen;
					struct sockaddr_rose wp;

					addrlen = sizeof(struct sockaddr_rose);
					if (getpeername(u->fd, (struct sockaddr *)&wp, &addrlen) != -1) 
					{
						wp_update_addr(&wp);
					}

					if ((u->verbose) && (u->peer) && (u->peer->fd != -1))
					{
						buffer[0] = 0xf0; /* Pid */
						sprintf(buffer+1, "*** Connection done\r");
						write(u->peer->fd, buffer, strlen(buffer));
					}
				}
				else
				{
					/* Accept the L3 connection */
					if ((u->peer) && (u->peer->fd != -1))
					{
						int yes = 1;
						ioctl(u->peer->fd, SIOCRSACCEPT, &yes);
					}
				}
			}

			if (len == -1)
			{
				/* Disconnection */
				if ((u->type == AF_ROSE) && (u->verbose) && (u->peer) && (u->peer->fd != -1))
				{
					char origin[80];
					struct rose_facilities_struct facilities;
					struct rose_cause_struct rose_cause;

					rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
					rose_cause.diagnostic = 0x12;

					ioctl(u->fd, SIOCRSGCAUSE, &rose_cause); 

					buffer[0] = 0xf0;
					if (u->state == CONNECTED)
					{
						origin[0] = '\0';
						if ((ioctl(u->fd, SIOCRSGFACILITIES, &facilities) != -1) &&
							(facilities.fail_call.ax25_call[0]))
						{
							sprintf(origin, " at %s @ %s",
								ax2asc(&facilities.fail_call),
								fpac2asc(&facilities.fail_addr));
						}
						sprintf(buffer+1, "*** Disconnected from %s%s\r*** %02X%02X - %s\r",
							u->user,
							origin,
							rose_cause.cause, 
							rose_cause.diagnostic, 
							reason(rose_cause.cause, rose_cause.diagnostic));
								
					}
					else
					{
						sprintf(buffer+1, "*** No answer from %s\r*** %02X%02X - %s\r",
							u->user,
							rose_cause.cause, 
							rose_cause.diagnostic, 
							reason(rose_cause.cause, rose_cause.diagnostic));
					}
					write(u->peer->fd, buffer, strlen(buffer));
				}
				
				/* Local disconnection */
				if ((u->type == AF_AX25) && (u->verbose) && (u->peer) && (u->peer->fd != -1) && (u->peer->type == AF_AX25))
				{
					buffer[0] = 0xf0;
					if (u->state == CONNECTED)
					{
						sprintf(buffer+1, "*** Disconnected from %s\r",
							u->user);
					}
					else
					{
						sprintf(buffer+1, "*** No answer from %s\r",
							u->user);
					}
					write(u->peer->fd, buffer, strlen(buffer));
				}

				end(u);
				
			}
			else if (len > 0)
			{
				if ((u->peer) && (u->peer->fd != -1))
				{
					int nb;
					char *ptr;

					if (u->type == AF_AX25)
					{
						if (u->peer->type == AF_AX25)
						{
							/* Local connection */
							ptr = buffer+2;
						}
						else
						{
							/* Received is AX25 L3 */
							if (buffer[2] == (char)0xf0)
							{
								buffer[2] = 0;  /* Qbit = 0 */
								ptr = buffer + 2;
							}
							else
							{
								buffer[0] = 1;    /* Qbit = 1 */
								buffer[1] = 0x7f; /* Escape */
								ptr = buffer;
								len += 2;
							}
						}
					}
					else
					{
						if (buffer[2] == 0) 
						{		/* Q Bit not set */
							buffer[2] = 0xF0;
							ptr = buffer + 2;
						}
						else 
						{
							/* Lose the leading 0x7F */
							ptr = buffer + 3;
							len -= 1;
						}
					}
					nb = write(u->peer->fd, ptr, len);
				}
#ifdef __TCP__
				else if (u->state == WAITCALL)
				{
					if (get_tcp_callsign(u, buffer+2,len))
						u->state = CONNECTED;
				}
#endif
			}
		}

		/* check queues */
		for (u = head ; u != NULL ; u = u->next)
		{
			if ((u->fd != -1) && (ioctl(u->fd, TIOCOUTQ, &lg) != -1))
			{
				u->queue = size - lg;
			}

		}

	}

	/* NOT REACHED */
	return 0;
}

static int new_socket(int domain, char *call, char *addr, char *port, int digi, int pidincl)
{
	int fd;
	int yes;
	int addrlen;
	int buflen;
	int type;
	char *p_name;
#ifdef __TCP__
	struct sockaddr_in inet;
#endif
	struct sockaddr_rose rose;
	struct full_sockaddr_ax25 ax25;

	type = (domain == AF_INET) ? SOCK_STREAM : SOCK_SEQPACKET;

	if ((fd = socket(domain, type, 0)) < 0) 
	{
		syslog(LOG_ERR, "socket: %s\n", strerror(errno));
		return(-1);
	}

	buflen = BUFLEN;
	setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &buflen, sizeof(buflen));
	setsockopt(fd, SOL_SOCKET, SO_SNDBUF, &buflen, sizeof(buflen));

	switch (domain)
	{

#ifdef __TCP__
	case AF_INET:

		addrlen = sizeof(struct sockaddr_in);

		inet.sin_family      = AF_INET;
		inet.sin_addr.s_addr = INADDR_ANY;
		inet.sin_port        = htons(cfg.inetport);

		if (bind(fd, (struct sockaddr *)&inet, addrlen) < 0) 
		{
			close(fd);
			syslog(LOG_ERR, "bind INET: %s", strerror(errno));
			return(-1);
		}

		break;
#endif

	case AF_AX25:

		if (digi)
		{
			yes = 1;
			setsockopt(fd, SOL_AX25, AX25_IAMDIGI, &yes, sizeof(yes));
		}

		if (pidincl)
		{
			yes = 1;
			if (setsockopt(fd, SOL_AX25, AX25_PIDINCL, &yes, sizeof(yes)) == -1)
			{
				syslog(LOG_ERR, "cannot setsockopt(AX25_PIDINCL), %s\n", strerror(errno));
				close(fd);
				return(-1);
			}
		}
		ax25.fsa_ax25.sax25_family = AF_AX25;
		ax25.fsa_ax25.sax25_ndigis = 1;
		convert_call_entry(call, ax25.fsa_ax25.sax25_call.ax25_call);
		if (*port == '*')
		{
			ax25.fsa_digipeater[0] = null_ax25_address;
		}
		else
		{
			p_name = ax25_config_get_addr (port);
			if (p_name)
				convert_call_entry(p_name, ax25.fsa_digipeater[0].ax25_call);
			else
			{
				fprintf(stderr, "Unknown port %s\n", port);
				return(-1);
			}
		}

		addrlen = sizeof(struct full_sockaddr_ax25);

		if (bind(fd, (struct sockaddr *)&ax25, addrlen) < 0) 
		{
			close(fd);
			syslog(LOG_ERR, "bind: %s", strerror(errno));
			return(-1);
		}

		break;

	case AF_ROSE:

		if (pidincl)
		{
			yes = 1;
			if (setsockopt(fd, SOL_ROSE, ROSE_QBITINCL, &yes, sizeof(yes)) == -1) 
			{
				close(fd);
				syslog(LOG_ERR, "cannot setsockopt(ROSE_QBITINCL) - %s\n", strerror(errno));
				return(-1);
			}
		}

		yes = 1;
		setsockopt(fd, SOL_ROSE, ROSE_DEFER, &yes, sizeof(yes));

		rose.srose_family = AF_ROSE;
		rose.srose_ndigis = 0;
		if (call)
			convert_call_entry(call, rose.srose_call.ax25_call);
		else
			rose.srose_call = null_ax25_address;
		convert_rose_address(addr, rose.srose_addr.rose_addr);

		addrlen = sizeof(struct sockaddr_rose);

		if (bind(fd, (struct sockaddr *)&rose, addrlen) < 0) 
		{
			close(fd);
			syslog(LOG_ERR, "bind: %s", strerror(errno));
			return(-1);
		}

		break;
	}

	if (listen(fd, SOMAXCONN) < 0) 
	{
		close(fd);
		syslog(LOG_ERR, "listen: %s", strerror(errno));
		return(-1);
	}

	return(fd);
}

static void end(user_t *p)
{
	/* Should be better to have the sockets closed when the queues are empty */
	sleep(1);
	if (p->fd != -1)
	{
		close(p->fd);
		p->fd = -1;
	}
	if (p->peer)
	{
		p = p->peer;
		if (p->fd != -1)
		{
			close(p->fd);
			p->fd = -1;
		}
	}
}

/* Returns the rose address defined for the AX25 port */
char *get_rose_addr(struct full_sockaddr_ax25 *ax25)
{
	addrp_t	*d;
	char *portname;
	static char address[40];

	strcpy(address, cfg.fulladdr);

	portname = ax25_config_get_port(&(ax25->fsa_digipeater[0]));
	if (portname)
	{
		for (d = cfg.addrp ; d != NULL ; d = d->next)
		{
			if (strcmp(d->port, portname) == 0)
			{
				strcpy(address+4, d->addr);
				break;
			}
		}
	}
	
	return(address);
}

static int new_l2_connection(int fd, int verbose)
{
	int ok;
	int wp;
	int new;
	int yes;
	int rsfd;
	int ndigi;
	int buflen;
	int addrlen;
	char *ptr;
	char rsaddr[11];
	char text[256];
	user_t *pl2, *pl3;
	struct full_sockaddr_ax25 rec;
	struct full_sockaddr_ax25 exp;
	struct sockaddr_rose rosebind, roseconnect;

	yes = TRUE;
	ioctl(fd, FIONBIO, &yes);

	addrlen = sizeof(struct full_sockaddr_ax25);
	new = accept(fd, (struct sockaddr *)&exp, &addrlen);

	yes = FALSE;
	ioctl(fd, FIONBIO, &yes);

	if (new < 0) 
	{
		if (errno == EWOULDBLOCK)
			return(-1);	/* It's gone ??? */

		syslog(LOG_ERR, "accept error %m\n");
		return(-1); 
	}

	yes = 1;
	if (setsockopt(new, SOL_AX25, AX25_PIDINCL, &yes, sizeof(yes)) == -1)
	{
		syslog(LOG_ERR, "cannot setsockopt(AX25_PIDINCL) - %s\n", strerror(errno));
		close(new);
		return(-1);
	}

	addrlen = sizeof(rec);
	getsockname(new, (struct sockaddr *)&rec, &addrlen);
            
	{
		/* Add the local callsign to the WP */
		struct sockaddr_rose wp;

		wp.srose_family = AF_ROSE;
		wp.srose_ndigis = rec.fsa_ax25.sax25_ndigis - 1;

		convert_rose_address (get_rose_addr(&rec), wp.srose_addr.rose_addr);
		wp.srose_call = exp.fsa_ax25.sax25_call;
		if (wp.srose_ndigis)
			wp.srose_digi = rec.fsa_digipeater[1];

		wp_update_addr(&wp);
	}
	
	pl2 = malloc(sizeof(user_t));
	pl2->fd = new;
	pl2->queue = 0;
	pl2->type = AF_AX25;
	pl2->state = CONNECTED;
	pl2->peer = NULL;
	pl2->verbose = FALSE;
	strcpy(pl2->user, ax2asc(&exp.fsa_ax25.sax25_call));

	/* Insert the new connection into the list */
	pl2->next = head;
	head = pl2;

	/*
	dump_ax25("exp", &exp);
	dump_ax25("rec", &rec);
	*/

	/* Build ROSE address */
	ndigi = exp.fsa_ax25.sax25_ndigis - 2;

	strcpy(rsaddr, cfg.fulladdr);

	ok = FALSE;

	if (ndigi >= 0)
	{
		int len;
		char dnic[5];

		memset(&dnic, 0, sizeof(dnic));

		ptr = ax2asc(&exp.fsa_digipeater[ndigi]);
		len = rose_add(ptr);

		/* Check if dnic */
		if ((len == 4) && (ndigi > 0))
		{
			memcpy(dnic, ptr, 4);
			--ndigi;

			ptr = ax2asc(&exp.fsa_digipeater[ndigi]);
			len = rose_add(ptr);
		}

		if (len == 6)
		{
			if (*dnic)
				memcpy(rsaddr, dnic, 4);
			memcpy(rsaddr+4, ptr, 6);
			--ndigi;
			ok = TRUE;
		}
	}

	roseconnect.srose_family = rosebind.srose_family = AF_ROSE;
	roseconnect.srose_ndigis = ndigi+1;
	rosebind.srose_ndigis    = 0;

	/* Get the ROSE address for this port */
	convert_rose_address (get_rose_addr(&rec), rosebind.srose_addr.rose_addr);
        rosebind.srose_call = exp.fsa_ax25.sax25_call;

	convert_rose_address (rsaddr, roseconnect.srose_addr.rose_addr);
        roseconnect.srose_call = rec.fsa_ax25.sax25_call;

	if (roseconnect.srose_ndigis)
		roseconnect.srose_digi = exp.fsa_digipeater[0];

	wp = 0;

	if (!ok)
	{
		/* No X25L3 routing. Trying via WP */

		struct sockaddr_rose wpaddr;

		if (wp_search(&rec.fsa_ax25.sax25_call, &wpaddr) == 0)
		{
			roseconnect = wpaddr;
			wp = 1;
		}
		else 
		{
			wp = 2;
		}
	}

	if (verbose)
	{
		int i;
		static char *constr[] = { "Connecting", "WP routing", "Trying local" } ;

		text[0] = 0xf0;
		sprintf(text+1, "*** %s %s @ %s", constr[wp], ax2asc(&rec.fsa_ax25.sax25_call), fpac2asc(&roseconnect.srose_addr));
		for (i = 0 ; i < roseconnect.srose_ndigis ; i++)
		{
			char str[20];
			sprintf(str, ",%s", ax2asc(&roseconnect.srose_digi));
			strcat(text, str);
		}
		strcat(text, "\r");
		write(new, text, strlen(text));
	}

	/* Start the L3 connection */
	if ((rsfd = socket (AF_ROSE, SOCK_SEQPACKET, 0)) < 0)
	{
		syslog(LOG_ERR, "socket_rs\n");
		end(pl2);
		return(-1);
	}

	buflen = BUFLEN;
	setsockopt(rsfd, SOL_SOCKET, SO_RCVBUF, &buflen, sizeof(buflen));
	setsockopt(rsfd, SOL_SOCKET, SO_SNDBUF, &buflen, sizeof(buflen));

	yes = 1;
	if (setsockopt(rsfd, SOL_ROSE, ROSE_QBITINCL, &yes, sizeof(yes)) == -1) 
	{
		syslog(LOG_ERR, "cannot setsockopt(ROSE_QBITINCL) - %s\n", strerror(errno));
		end(pl2);
		return(-1);
	}

	addrlen = sizeof(struct sockaddr_rose);
	
	if (bind (rsfd, (struct sockaddr *) &rosebind, addrlen) == -1)
	{
		syslog(LOG_ERR, "bind_rs\n");
		close (rsfd);
		end(pl2);
      		return(-1);
	}

	/* Non blocking connection */
	yes = 1;
	ioctl(rsfd, FIONBIO, &yes);

	/*
	dump_rose("bind", &rosebind);
	dump_rose("connect", &roseconnect);
	*/

	if (connect (rsfd, (struct sockaddr *) &roseconnect, addrlen) == -1)
	{
		if (errno != EINPROGRESS)
		{
			if (verbose)
			{
				struct rose_cause_struct rose_cause;

				switch (errno) 
				{
				case ENETUNREACH:
					ioctl(rsfd, SIOCRSGCAUSE, &rose_cause); 
					break;
				case EISCONN:
					rose_cause.cause      = ROSE_NUMBER_BUSY;
					rose_cause.diagnostic = 0x48;
					break;
				case EINVAL:
				default:
					rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
					rose_cause.diagnostic = 0;
					break;
				}
				text[0] = 0xf0;
				sprintf(text+1, "*** Disconnected from %s\r*** %02X%02X - %s\r", 
					ax2asc(&roseconnect.srose_call), 
					rose_cause.cause, 
					rose_cause.diagnostic, 
					reason(rose_cause.cause, rose_cause.diagnostic));
				write(new, text, strlen(text));
			}
			close (rsfd);
			end(pl2);
			return(-1);
		}
	}
  
	yes = 0;
	ioctl(rsfd, FIONBIO, &yes);

	pl3 = malloc(sizeof(user_t));
	pl3->fd = rsfd;
	pl3->queue = 0;
	pl3->type = AF_ROSE;
	pl3->state = CPROGRESS;
	pl3->peer = pl2;
	pl3->verbose = verbose;
	strcpy(pl3->user, ax2asc(&roseconnect.srose_call));

	/* Insert the new connection into the list */
	pl3->next = head;
	head = pl3;

	/* give L2 the L3 link */
	pl2->peer = pl3;

	return(new);
}

static int callcmp(char *call1, char *call2)
{
	int len1 = 0;
	int len2 = 0;
	char *ptr;

	ptr = strchr(call1, '-');
	if ((ptr) && (*(ptr+1) != '*'))
	{
		return strcasecmp(call1, call2);
	}

	ptr = strchr(call1, '-');
	if (ptr)
		len1 = ptr - call1;

	ptr = strchr(call2, '-');
	if (ptr)
		len2 = ptr - call2;

	return strncasecmp(call1, call2, (len2 > len1) ? len2 : len1);
}

/* Checks if the connection is for a local application */
static int appli_connection(char *callsign, char *appli)
{
	appli_t *a;

	for (a = cfg.appli ; a != NULL ; a = a->next)
	{
		if (callcmp(a->call, callsign) == 0)
		{
			strcpy(appli, a->appli);
			return(1);
		}
	}
	return(0);
}

static int new_l3_connection(int fd, char *port, int verbose)
{
	int new;
	int yes;
	int axfd;
	int ndigi;
	int buflen;
	int addrlen;
	char *addr;
	char *p_name;
	char application[256];
	user_t *pl2, *pl3;
	struct sockaddr_rose rsrec, rsexp;
	struct full_sockaddr_ax25 axbind, axconnect;

	yes = 1;
	ioctl(fd, FIONBIO, &yes);

	addrlen = sizeof(struct sockaddr_rose);
	new = accept(fd, (struct sockaddr *)&rsexp, &addrlen);
	if (new < 0) 
	{
		if (errno == EWOULDBLOCK)
			return(-1);	/* It's gone ??? */

		syslog(LOG_ERR, "accept error %m\n");
			return(-1); 
	}
	yes = 0;
	ioctl(fd, FIONBIO, &yes);

	addrlen = sizeof(rsrec);
	getsockname(new, (struct sockaddr *)&rsrec, &addrlen);

	/* Check if it is a local application */
	if (appli_connection(ax2asc(&rsrec.srose_call), application))
	{
		yes = FALSE;
		if (setsockopt(new, SOL_ROSE, ROSE_QBITINCL, &yes, sizeof(yes)) == -1) 
		{
			close(fd);
			syslog(LOG_ERR, "cannot setsockopt(ROSE_QBITINCL) - %s\n", strerror(errno));
			return(-1);
		}
		return new_appli_connection(new, TRUE, application);
	}

	/* Add requester in the WP */
	wp_update_addr(&rsexp);

	yes = 1;
	if (setsockopt(new, SOL_ROSE, ROSE_QBITINCL, &yes, sizeof(yes)) == -1) 
	{
		syslog(LOG_ERR, "cannot setsockopt(ROSE_QBITINCL) - %s\n", strerror(errno));
		close(fd);
		return(-1);
	}

	/*
	dump_rose("exp", &rsexp);
	dump_rose("rec", &rsrec);
	*/

	pl3 = malloc(sizeof(user_t));
	pl3->fd = new;
	pl3->queue = 0;
	pl3->type = AF_ROSE;
	pl3->state = CONNECTED;
	pl3->peer = NULL;
	pl3->verbose = FALSE;
	strcpy(pl3->user, ax2asc(&rsexp.srose_call));

	/* Insert the new connection into the list */
	pl3->next = head;
	head = pl3;

	/* Build AX25 structures */
	axbind.fsa_ax25.sax25_family = AF_AX25;
	axbind.fsa_ax25.sax25_ndigis = 1;
	axbind.fsa_ax25.sax25_call   = rsexp.srose_call;

	if (port)
		p_name = port;
	else
		p_name = user_port(&rsrec.srose_call);

	if ((addr = ax25_config_get_addr(p_name)) == NULL) 
	{
		syslog(LOG_ERR, "invalid AX.25 port name - %s\n", p_name);
		return(-1);
	}

	if (convert_call_entry(addr, axbind.fsa_digipeater[0].ax25_call) == -1) 
	{
		syslog(LOG_ERR, "invalid AX.25 port callsign - %s\n", addr);
		return(-1);
	}

	axconnect.fsa_ax25.sax25_family = AF_AX25;
	axconnect.fsa_ax25.sax25_call   = rsrec.srose_call;

	ndigi = 0;

	/* The path at the far end has a digi in it. */
	if (rsexp.srose_ndigis == 1) 
	{
		axconnect.fsa_digipeater[ndigi] = rsexp.srose_digi;
		axconnect.fsa_digipeater[ndigi].ax25_call[6] |= AX25_HBIT;
		ndigi++;
	}

	addr = rose2asc(&rsexp.srose_addr);

	/* Check if local address used */
	if (strcmp(addr, cfg.fulladdr) != 0)
	{
		/*	Incoming call has a different DNIC */
		if (memcmp(rsexp.srose_addr.rose_addr, rsrec.srose_addr.rose_addr, 2) != 0)
		{
			char dnic[5];
			
			memcpy(dnic, addr, 4);
			dnic[4] = '\0';
			if (convert_call_entry(dnic, axconnect.fsa_digipeater[ndigi].ax25_call) == -1) {
				syslog(LOG_ERR, "invalid dnic - %s\n", dnic);
				return(-1);
			}
			axconnect.fsa_digipeater[ndigi].ax25_call[6] |= AX25_HBIT;
			ndigi++;		
		}
	
		/* Put the remote address sans DNIC into the digi chain. */
		addr = rose2asc(&rsexp.srose_addr);
		if (convert_call_entry(addr + 4, axconnect.fsa_digipeater[ndigi].ax25_call) == -1)
		{
			syslog(LOG_ERR, "invalid callsign - %s\n", addr + 4);
			return(-1);
		}
		axconnect.fsa_digipeater[ndigi].ax25_call[6] |= AX25_HBIT;
		ndigi++;		
	}
	
	/* And my local ROSE callsign. */
	if (convert_call_entry(cfg.callsign, axconnect.fsa_digipeater[ndigi].ax25_call) == -1)
	{
		syslog(LOG_ERR, "invalid callsign - %s\n", cfg.callsign);
		return(-1);
	}
	axconnect.fsa_digipeater[ndigi].ax25_call[6] |= AX25_HBIT;
	ndigi++;

	/* A digi has been specified for this end. */
	if (rsrec.srose_ndigis == 1) 
	{
		axconnect.fsa_digipeater[ndigi] = rsrec.srose_digi;
		ndigi++;
	}

	axconnect.fsa_ax25.sax25_ndigis = ndigi;

	/* Start the L2 connection */
	if ((axfd = socket (AF_AX25, SOCK_SEQPACKET, 0)) < 0)
	{
		syslog(LOG_ERR, "socket_ax\n");
		end(pl3);
		return(-1);
	}
	
	buflen = BUFLEN;
	setsockopt(axfd, SOL_SOCKET, SO_RCVBUF, &buflen, sizeof(buflen));
	setsockopt(axfd, SOL_SOCKET, SO_SNDBUF, &buflen, sizeof(buflen));

	yes = 1;
	if (setsockopt(axfd, SOL_AX25, AX25_IAMDIGI, &yes, sizeof(yes)) == -1) 
	{
		syslog(LOG_ERR, "cannot setsockopt(AX25_IAMDIGI), %s\n", strerror(errno));
		close(axfd);
		return(-1);
	}

	yes = 1;
	if (setsockopt(axfd, SOL_AX25, AX25_PIDINCL, &yes, sizeof(yes)) == -1)
	{
		syslog(LOG_ERR, "cannot setsockopt(AX25_PIDINCL), %s\n", strerror(errno));
		close(axfd);
		end(pl3);
		return(-1);
	}

	addrlen = sizeof(struct full_sockaddr_ax25);
	
	if (bind (axfd, (struct sockaddr *) &axbind, addrlen) == -1)
	{
		syslog(LOG_ERR, "bind_ax\n");
		close (axfd);
		end(pl3);
      		return(-1);
	}

	/* Non blocking connection */
	yes = 1;
	ioctl(axfd, FIONBIO, &yes);

	if (connect (axfd, (struct sockaddr *) &axconnect, addrlen) == -1)
	{
		if (errno != EINPROGRESS)
		{
			struct rose_cause_struct rose_cause;

			switch (errno) 
			{
			case EADDRINUSE:
				rose_cause.cause      = ROSE_NUMBER_BUSY;
				rose_cause.diagnostic = 0x48;
				break;
			case ECONNREFUSED:
				rose_cause.cause      = ROSE_NUMBER_BUSY;
				rose_cause.diagnostic = 0;
				break;
			case ETIMEDOUT:
				rose_cause.cause      = ROSE_SHIP_ABSENT;
				rose_cause.diagnostic = 0;
				break;
			default:
				rose_cause.cause      = ROSE_REMOTE_PROCEDURE;
				rose_cause.diagnostic = errno;
				break;
			}

			ioctl(new, SIOCRSSCAUSE, &rose_cause);

			close (axfd);
			end(pl3);
			return(-1);
		}
	}
  
	yes = 0;
	ioctl(axfd, FIONBIO, &yes);

	pl2 = malloc(sizeof(user_t));
	pl2->fd = axfd;
	pl2->queue = 0;
	pl2->type = AF_AX25;
	pl2->state = CPROGRESS;
	pl2->peer = pl3;
	pl2->verbose = verbose;
	strcpy(pl2->user, ax2asc(&axconnect.fsa_ax25.sax25_call));

	/* Insert the new connection into the list */
	pl2->next = head;
	head = pl2;

	/* give L3 the L2 link */
	pl3->peer = pl2;

	return(new);
}

static int new_alias_connection(alias_t *a, int verbose)
{
	int new;
	int yes;
	int rsfd;
	int ndigi;
	int buflen;
	int addrlen;
	char *ptr;
	char *call;
	char rsaddr[11];
	char text[256];
	char *stmp;
	user_t *pl2, *pl3;
	struct full_sockaddr_ax25 rec;
	struct full_sockaddr_ax25 exp;
	struct sockaddr_rose rosebind, roseconnect;

	yes = TRUE;
	ioctl(a->fd, FIONBIO, &yes);

	addrlen = sizeof(struct full_sockaddr_ax25);
	new = accept(a->fd, (struct sockaddr *)&exp, &addrlen);

	yes = FALSE;
	ioctl(a->fd, FIONBIO, &yes);

	if (new < 0) 
	{
		if (errno == EWOULDBLOCK)
			return(-1);	/* It's gone ??? */

		syslog(LOG_ERR, "accept error %m\n");
		return(-1); 
	}

	yes = 1;
	if (setsockopt(new, SOL_AX25, AX25_PIDINCL, &yes, sizeof(yes)) == -1)
	{
		syslog(LOG_ERR, "cannot setsockopt(AX25_PIDINCL) - %s\n", strerror(errno));
		close(new);
		return(-1);
	}

	addrlen = sizeof(rec);
	getsockname(new, (struct sockaddr *)&rec, &addrlen);
            
	pl2 = malloc(sizeof(user_t));
	pl2->fd = new;
	pl2->queue = 0;
	pl2->type = AF_AX25;
	pl2->state = CONNECTED;
	pl2->peer = NULL;
	pl2->verbose = FALSE;
	strcpy(pl2->user, ax2asc(&exp.fsa_ax25.sax25_call));

	/* Insert the new connection into the list */
	pl2->next = head;
	head = pl2;

	ndigi = -1;

	/* Build ROSE address */
	strcpy(rsaddr, cfg.fulladdr);

	stmp = strdup(a->path);

	/* callsign */
	call = strtok(stmp, " \t,");

	/* address */
	ptr = strtok(NULL, " \t,");
	if (ptr)
	{
		int len;
		int ok = FALSE;
		char dnic[5];

		memset(&dnic, 0, sizeof(dnic));

		len = rose_add(ptr);

		/* Check if dnic */
		if ((len == 4) && (ndigi > 0))
		{
			memcpy(dnic, ptr, 4);
			ptr = strtok(NULL, " \t,");
			len = rose_add(ptr);
		}

		if (len == 6)
		{
			if (*dnic)
				memcpy(rsaddr, dnic, 4);
			memcpy(rsaddr+4, ptr, 6);
			ok = TRUE;
		}
	}

	if (verbose)
	{
		text[0] = 0xf0;
		sprintf(text+1, "*** Connecting %s @ %s", call, rsaddr);
		/*
		for (i = ndigi ; i >= 0 ; i--)
		{
			char str[20];
			sprintf(str, ",%s", ax2asc(&exp.fsa_digipeater[ndigi]));
			strcat(text, str);
		}
		*/
		strcat(text, "\r");
		write(new, text, strlen(text));
	}

	/* Start the L3 connection */
	if ((rsfd = socket (AF_ROSE, SOCK_SEQPACKET, 0)) < 0)
	{
		syslog(LOG_ERR, "socket_rs\n");
		end(pl2);
		return(-1);
	}

	buflen = BUFLEN;
	setsockopt(rsfd, SOL_SOCKET, SO_RCVBUF, &buflen, sizeof(buflen));
	setsockopt(rsfd, SOL_SOCKET, SO_SNDBUF, &buflen, sizeof(buflen));

	yes = 1;
	if (setsockopt(rsfd, SOL_ROSE, ROSE_QBITINCL, &yes, sizeof(yes)) == -1) 
	{
		syslog(LOG_ERR, "cannot setsockopt(ROSE_QBITINCL) - %s\n", strerror(errno));
		end(pl2);
		free(stmp);
		return(-1);
	}

	roseconnect.srose_family = rosebind.srose_family = AF_ROSE;

	rosebind.srose_ndigis    = 0;
	convert_rose_address (cfg.fulladdr, rosebind.srose_addr.rose_addr);
        rosebind.srose_call = exp.fsa_ax25.sax25_call;

	roseconnect.srose_ndigis = 0;
	convert_rose_address (rsaddr, roseconnect.srose_addr.rose_addr);
	if (convert_call_entry(call, roseconnect.srose_call.ax25_call) == -1)
	{
		syslog(LOG_ERR, "convert_call_entry %s\n", cfg.callsign);
		close (rsfd);
		end(pl2);
		free(stmp);
      		return(-1);
	}

	free(stmp);

	addrlen = sizeof(struct sockaddr_rose);
	
	if (bind (rsfd, (struct sockaddr *) &rosebind, addrlen) == -1)
	{
		syslog(LOG_ERR, "bind_rs\n");
		close (rsfd);
		end(pl2);
      		return(-1);
	}

	/* Non blocking connection */
	yes = 1;
	ioctl(rsfd, FIONBIO, &yes);

	if (connect (rsfd, (struct sockaddr *) &roseconnect, addrlen) == -1)
	{
		if (errno != EINPROGRESS)
		{
			if (verbose)
			{
				struct rose_cause_struct rose_cause;

				switch (errno) 
				{
				case ENETUNREACH:
					ioctl(rsfd, SIOCRSGCAUSE, &rose_cause); 
					break;
				case EISCONN:
				case EINVAL:
				default:
					rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
					rose_cause.diagnostic = 0;
					break;
					break;
				}
				text[0] = 0xf0;
				sprintf(text+1, "*** Failure with %s\r*** %02X%02X - %s\r", 
					ax2asc(&roseconnect.srose_call), 
					rose_cause.cause, 
					rose_cause.diagnostic, 
					reason(rose_cause.cause, rose_cause.diagnostic));
				write(new, text, strlen(text));
			}
			close (rsfd);
			end(pl2);
			return(-1);
		}
	}
  
	yes = 0;
	ioctl(rsfd, FIONBIO, &yes);

	pl3 = malloc(sizeof(user_t));
	pl3->fd = rsfd;
	pl3->queue = 0;
	pl3->type = AF_ROSE;
	pl3->state = CPROGRESS;
	pl3->peer = pl2;
	pl3->verbose = verbose;
	strcpy(pl3->user, ax2asc(&roseconnect.srose_call));

	/* Insert the new connection into the list */
	pl3->next = head;
	head = pl3;

	/* give L2 the L3 link */
	pl2->peer = pl3;

	return(new);
}

/* This function is straight from *NOS */

static char *parse_string(char *str)
{
	char *cp = str;
	unsigned long num;

	while (*str != '\0' && *str != '\"') {
		if (*str == '\\') {
			str++;
			switch (*str++) {
			case 'n':
				*cp++ = '\n';
				break;
			case 't':
				*cp++ = '\t';
				break;
			case 'v':
				*cp++ = '\v';
				break;
			case 'b':
				*cp++ = '\b';
				break;
			case 'r':
				*cp++ = '\r';
				break;
			case 'f':
				*cp++ = '\f';
				break;
			case 'a':
				*cp++ = '\007';
				break;
			case '\\':
				*cp++ = '\\';
				break;
			case '\"':
				*cp++ = '\"';
				break;
			case 'x':
				num = strtoul(--str, &str, 16);
				*cp++ = (char) num;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
				num = strtoul(--str, &str, 8);
				*cp++ = (char) num;
				break;
			case '\0':
				return NULL;
			default:
				*cp++ = *(str - 1);
				break;
			};
		} else {
			*cp++ = *str++;
		}
	}
	if (*str == '\"')
		str++;		/* skip final quote */
	*cp = '\0';		/* terminate string */
	return str;
}

static int parse_args(char **argv, char *cmd)
{
	int ct = 0;
	int quoted;

	while (ct < 31)
	{
		quoted = 0;
		while (*cmd && isspace(*cmd))
			cmd++;
		if (*cmd == 0)
			break;
		if (*cmd == '"') {
			quoted++;
			cmd++;
		}
		argv[ct++] = cmd;
		if (quoted) {
			if ((cmd = parse_string(cmd)) == NULL)
				return 0;
		} else {
			while (*cmd && !isspace(*cmd))
				cmd++;
		}
		if (*cmd)
			*cmd++ = 0;
	}
	argv[ct] = NULL;
	return ct;
}

static int new_appli_connection(int fd, int accepted, char *appli)
{
	int new;
	int pid;
	int yes;
	int addrlen;

	if (accepted)
	{
		new = fd;
	}
	else
	{
		struct full_sockaddr_ax25 exp;

		yes = TRUE;
		ioctl(fd, FIONBIO, &yes);

		addrlen = sizeof(struct full_sockaddr_ax25);
		new = accept(fd, (struct sockaddr *)&exp, &addrlen);

		yes = FALSE;
		ioctl(fd, FIONBIO, &yes);

		if (new < 0) 
		{
			if (errno == EWOULDBLOCK)
				return(-1);	/* It's gone ??? */

			syslog(LOG_ERR, "accept error %m\n");
			return(-1); 
		}
	}

	pid = fork();

	switch (pid) 
	{
	case -1:
		syslog(LOG_ERR, "fork error %m");
		close(new);
		break;

	case 0: {
		char *argv[32];
		parse_args(argv, appli);

		dup2(new, STDIN_FILENO);
		dup2(new, STDOUT_FILENO);
		close(new);

		execvp(argv[0], argv);
		exit(1);
		}
	default:
		close(new);
		break;
	}

	return(0);
}


#ifdef __TCP__
static int new_tcp_connection(int fd, int verbose)
{
	int new;
	int yes;
	user_t *pl2;
	struct sockaddr_in exp;
	int addrlen;
	char *ptr;

	yes = TRUE;
	ioctl(fd, FIONBIO, &yes);

	addrlen = sizeof(struct sockaddr_in);
	new = accept(fd, (struct sockaddr *)&exp, &addrlen);

	yes = FALSE;
	ioctl(fd, FIONBIO, &yes);

	if (new < 0) 
	{
		if (errno == EWOULDBLOCK)
			return(-1);	/* It's gone ??? */

		syslog(LOG_ERR, "accept error %m\n");
		return(-1); 
	}

	ptr = "Callsign:";
	write(new, ptr, strlen(ptr));

	pl2 = malloc(sizeof(user_t));
	pl2->fd = new;
	pl2->queue = 0;
	pl2->type = AF_INET;
	pl2->state = WAITCALL;
	pl2->peer = NULL;
	pl2->verbose = FALSE;
	strcpy(pl2->user, "None");

	/* Insert the new connection into the list */
	pl2->next = head;
	head = pl2;

	return new;
}

static int get_tcp_callsign(user_t *pl2, char *buffer, int len)
{
	char *sender;
	char *receiver;
	char *p_name;
	char *addr;
	char application[256];
	int offs = len;
	char *av[10];
	int ac;
	char rsaddr[11];
	int ndigi;
	struct full_sockaddr_ax25 exp, rec;
	struct sockaddr_rose rosebind, roseconnect;
	int addrlen;
	char *ptr;
	char text[256];
	int ok;
	int wp;
	int yes;
	int rsfd;
	user_t *pl3;
	alias_t *a;
	int verbose = FALSE;


	/* Deletes the invalid end of the command string */
	for (offs = len-1 ; offs >= 0 ; offs--)
	{
		if ((isgraph(buffer[offs])) && (buffer[offs] != '^'))
			break;
	}
	buffer[offs+1] = '\0';

	/* extract sender */
	ptr = strchr(buffer, '^');
	if (ptr)
		*ptr++ = '\0';
	else
	{
		end(pl2);
		syslog(LOG_ERR, "ERROR: destination callsign missing\n");
		return(-1);
	}

	/* Deletes the '.' for binary information */
	sender = buffer;
	if (*sender == '.')
		++sender;

	/* Check for an alias */
	for (a = cfg.alias ; a != NULL ; a = a->next)
	{
		if (strcasecmp(a->alias, ptr) == 0)
		{
			strcpy(ptr, a->path);
			verbose = TRUE;
			break;
		}
	}


	if (convert_call_entry(sender, exp.fsa_ax25.sax25_call.ax25_call) == -1) 
	{
		end(pl2);
		syslog(LOG_ERR, "ERROR: invalid callsign - %s\n", sender);
		return(-1);
	}

	/* Build AX25 structures */
	exp.fsa_ax25.sax25_family = AF_AX25;
	exp.fsa_ax25.sax25_ndigis = 1;

	/* extract arguments of the destination and digis */
	for (ac = 0; ac < 9 ;)
	{
		while ((*ptr) && (!isgraph(*ptr)))
			++ptr;
		if (*ptr == '\0')
			break;
		av[ac] = ptr;
		while (isgraph(*ptr))
		{
			if ((*ptr == '^') || (*ptr == ','))
				break;
			++ptr;
		}
		if (*ptr)
			*ptr++ = '\0';
		if (ac == 1) 
		{
			if (	(strcasecmp(av[1], "VIA") == 0) ||
				(strcasecmp(av[1], "V") == 0) ||
				(strcasecmp(av[1], cfg.callsign) == 0) ||
				(strcasecmp(av[1], cfg.alt_callsign) == 0))
			{
				verbose = (strcasecmp(av[1], cfg.callsign) == 0);
				continue;
			}
		}
		++ac;
	}
	av[ac] = NULL;

	{
		/* Add the local callsign to the WP */
		struct sockaddr_rose wp;

		wp.srose_family = AF_ROSE;
		wp.srose_ndigis = 0;

		convert_rose_address (cfg.fulladdr, wp.srose_addr.rose_addr);
		wp.srose_call = exp.fsa_ax25.sax25_call;

		wp_update_addr(&wp);
	}

	if (ac == 1)
	{
		/* Check if it is an application */
		if (appli_connection(av[0], application))
		{
			char str[80];

			sprintf(str, "CALL_TCP=%s", sender);
			putenv(str);
			new_appli_connection(pl2->fd, TRUE, application);
			end(pl2);
			return(-1);
		}

	}

	ndigi = 0;
	receiver = av[0];

	/* Build ROSE address - Skip rec and 1st digi (node) */
	ndigi++;
	ac--;

	strcpy(rsaddr, cfg.fulladdr);

	ok = FALSE;

	if (ac > 0)
	{
		int len;
		char dnic[5];

		memset(&dnic, 0, sizeof(dnic));

		len = rose_add(av[ndigi]);

		/* Check if dnic */
		if ((len == 4) && (ac > 0))
		{
			memcpy(dnic, av[ndigi], 4);
			++ndigi;
			--ac;

			len = rose_add(av[ndigi]);
		}

		if (len == 6)
		{
			if (*dnic)
				memcpy(rsaddr, dnic, 4);
			memcpy(rsaddr+4, av[ndigi], 6);
			++ndigi;
			--ac;
			ok = TRUE;
		}
	}

	roseconnect.srose_family = rosebind.srose_family = AF_ROSE;
	roseconnect.srose_ndigis = ac;
	rosebind.srose_ndigis    = 0;

	convert_rose_address (cfg.fulladdr, rosebind.srose_addr.rose_addr);
        rosebind.srose_call = exp.fsa_ax25.sax25_call;

	convert_rose_address (rsaddr, roseconnect.srose_addr.rose_addr);
	if (convert_call_entry(receiver, roseconnect.srose_call.ax25_call) == -1)
	{
		end(pl2);
		return(-1);
	}

	if (roseconnect.srose_ndigis)
	{
		if (convert_call_entry(av[ndigi], roseconnect.srose_digi.ax25_call) == -1)
		{
			end(pl2);
			return(-1);
		}
	}

	wp = 0;

	if (!ok)
	{
		/* No X25L3 routing. Trying via WP */

		struct sockaddr_rose wpaddr;

		if (wp_search(&rec.fsa_ax25.sax25_call, &wpaddr) == 0)
		{
			roseconnect = wpaddr;
			wp = 1;
		}
		else 
		{
			wp = 2;
		}
	}

	if (verbose)
	{
		int i;
		static char *constr[] = { "Connecting", "WP routing", "Trying local" } ;

		sprintf(text, "*** %s %s @ %s", constr[wp], ax2asc(&roseconnect.srose_call), fpac2asc(&roseconnect.srose_addr));
		for (i = 0 ; i < roseconnect.srose_ndigis ; i++)
		{
			char str[20];
			sprintf(str, ",%s", ax2asc(&roseconnect.srose_digi));
			strcat(text, str);
		}
		strcat(text, "\r");
		msg(pl2, text);
	}

	/* Start the L3 connection */
	if ((rsfd = socket (AF_ROSE, SOCK_SEQPACKET, 0)) < 0)
	{
		syslog(LOG_ERR, "socket_rs\n");
		end(pl2);
		return(-1);
	}

	addrlen = sizeof(struct sockaddr_rose);
	
	if (bind (rsfd, (struct sockaddr *) &rosebind, addrlen) == -1)
	{
		syslog(LOG_ERR, "bind_rs\n");
		close (rsfd);
		end(pl2);
      		return(-1);
	}

	/* Non blocking connection */
	yes = 1;
	ioctl(rsfd, FIONBIO, &yes);

	if (connect (rsfd, (struct sockaddr *) &roseconnect, addrlen) == -1)
	{
		if (errno != EINPROGRESS)
		{
			if (pl2->verbose)
			{
				struct rose_cause_struct rose_cause;

				switch (errno) 
				{
				case ENETUNREACH:
					ioctl(rsfd, SIOCRSGCAUSE, &rose_cause); 
					break;
				case EISCONN:
					rose_cause.cause      = ROSE_NUMBER_BUSY;
					rose_cause.diagnostic = 0x48;
					break;
				case EINVAL:
				default:
					rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
					rose_cause.diagnostic = 0;
					break;
				}
				msg(pl2, "*** Disconnected from %s\r*** %02X%02X - %s\r", 
					ax2asc(&roseconnect.srose_call), 
					rose_cause.cause, 
					rose_cause.diagnostic, 
					reason(rose_cause.cause, rose_cause.diagnostic));
			}
			close (rsfd);
			end(pl2);
			return(-1);
		}
	}
  
	yes = 0;
	ioctl(rsfd, FIONBIO, &yes);

	pl3 = malloc(sizeof(user_t));
	pl3->fd = rsfd;
	pl3->queue = 0;
	pl3->type = AF_ROSE;
	pl3->state = CPROGRESS;
	pl3->peer = pl2;
	pl3->verbose = verbose;
	strcpy(pl3->user, ax2asc(&roseconnect.srose_call));

	/* Insert the new connection into the list */
	pl3->next = head;
	head = pl3;

	/* give L2 the L3 link */
	pl2->peer = pl3;

	return(pl2->fd);
}
#endif

static int rose_add(char *str)
{
	int len = 0;

	while (isdigit(*str))
	{
		++len;
		++str;
	}

	if ((*str) && (*str != '-'))
		return(0);

	return(len);
}

void dump_ax25(char *title, struct full_sockaddr_ax25 *ax25)
{
	int i;

	printf("%s\n", title);
	printf("\tFamily   = %d\n", ax25->fsa_ax25.sax25_family);
	printf("\tCallsign = %s\n", ax2asc(&ax25->fsa_ax25.sax25_call));
	printf("\tndigis   = %d\n", ax25->fsa_ax25.sax25_ndigis);

	for (i = 0; i < ax25->fsa_ax25.sax25_ndigis ; i++)
	{
		printf("\tdigi %d   = %s\n", i+1, ax2asc(&ax25->fsa_digipeater[i]));
	}
	printf("\n");
}

void dump_rose(char *title, struct sockaddr_rose *rose)
{
	int i;

	printf("%s\n", title);
	printf("\tFamily   = %d\n", rose->srose_family);
	printf("\tCallsign = %s\n", ax2asc(&rose->srose_call));
	printf("\tAddress  = %s\n", rose2asc(&rose->srose_addr));
	printf("\tndigis   = %d\n", rose->srose_ndigis);

	for (i = 0; i < rose->srose_ndigis ; i++)
	{
		printf("\tdigi %d   = %s\n", i+1, ax2asc(&rose->srose_digi));
	}
	printf("\n");
}

static char *reason(unsigned char cause, unsigned char diagnostic)
{
	static char *desc;

	switch (cause) 
	{
	case ROSE_DTE_ORIGINATED:
		desc = "Remote Station cleared connection";
		break;
	case ROSE_NUMBER_BUSY:
		if (diagnostic == 0x48)
			desc = "Remote station is connecting to you";
		else
			desc = "Remote Station is busy";
		break;
	case ROSE_INVALID_FACILITY:
		desc = "Invalid X.25 Facility Requested";
		break;
	case ROSE_NETWORK_CONGESTION:
		desc = "Network Congestion";
		break;
	case ROSE_OUT_OF_ORDER:
		desc = "Out of Order, link unavailable";
		break;
	case ROSE_ACCESS_BARRED:
		desc = "Access Barred";
		break;
	case ROSE_NOT_OBTAINABLE:
		desc = "No Route Defined for address specified";
		break;
	case ROSE_REMOTE_PROCEDURE:
		desc = "Remote Procedure Error";
		break;
	case ROSE_LOCAL_PROCEDURE:
		desc = "Local Procedure Error";
		break;
	case ROSE_SHIP_ABSENT:
		desc = "Remote Station not responding";
		break;
	default:
		desc = "Unknown";
		break;
	}

	return desc;
}
