/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.restore;

import com.codahale.metrics.DefaultSettableGauge;
import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import org.apache.cassandra.sidecar.cluster.locator.LocalTokenRangesProvider;
import org.apache.cassandra.sidecar.common.data.RestoreJobStatus;
import org.apache.cassandra.sidecar.common.server.utils.DurationSpec;
import org.apache.cassandra.sidecar.common.server.utils.SecondBoundConfiguration;
import org.apache.cassandra.sidecar.concurrent.ConcurrencyLimiter;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.concurrent.TaskExecutorPool;
import org.apache.cassandra.sidecar.config.SidecarConfiguration;
import org.apache.cassandra.sidecar.db.RestoreRange;
import org.apache.cassandra.sidecar.db.RestoreRangeDatabaseAccessor;
import org.apache.cassandra.sidecar.db.schema.SidecarSchema;
import org.apache.cassandra.sidecar.exceptions.RestoreJobException;
import org.apache.cassandra.sidecar.exceptions.RestoreJobExceptions;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.metrics.instance.InstanceRestoreMetrics;
import org.apache.cassandra.sidecar.restore.RestoreJobUtil;
import org.apache.cassandra.sidecar.restore.RestoreRangeHandler;
import org.apache.cassandra.sidecar.restore.StorageClientPool;
import org.apache.cassandra.sidecar.tasks.PeriodicTask;
import org.apache.cassandra.sidecar.tasks.ScheduleDecision;
import org.apache.cassandra.sidecar.utils.SSTableImporter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class RestoreProcessor
implements PeriodicTask {
    private static final Logger LOGGER = LoggerFactory.getLogger(RestoreProcessor.class);
    private final TaskExecutorPool pool;
    private final StorageClientPool s3ClientPool;
    private final SidecarSchema sidecarSchema;
    private final SSTableImporter importer;
    private final ConcurrencyLimiter processMaxConcurrency;
    private final WorkQueue workQueue = new WorkQueue();
    private final double requiredUsableSpacePercentage;
    private final RestoreRangeDatabaseAccessor rangeDatabaseAccessor;
    private final RestoreJobUtil restoreJobUtil;
    private final Map<RestoreRangeHandler, Long> activeTasks = new ConcurrentHashMap<RestoreRangeHandler, Long>();
    private final SecondBoundConfiguration slowTaskThreshold;
    private final SecondBoundConfiguration slowTaskReportDelay;
    private final LocalTokenRangesProvider localTokenRangesProvider;
    private final SidecarMetrics metrics;
    private volatile boolean isClosed = false;

    @Inject
    public RestoreProcessor(ExecutorPools executorPools, SidecarConfiguration config, SidecarSchema sidecarSchema, StorageClientPool s3ClientPool, SSTableImporter importer, RestoreRangeDatabaseAccessor rangeDatabaseAccessor, RestoreJobUtil restoreJobUtil, LocalTokenRangesProvider localTokenRangesProvider, SidecarMetrics metrics) {
        this.pool = executorPools.internal();
        this.s3ClientPool = s3ClientPool;
        this.sidecarSchema = sidecarSchema;
        this.processMaxConcurrency = new ConcurrencyLimiter(() -> config.restoreJobConfiguration().processMaxConcurrency());
        this.requiredUsableSpacePercentage = (double)config.serviceConfiguration().sstableUploadConfiguration().minimumSpacePercentageRequired() / 100.0;
        this.slowTaskThreshold = config.restoreJobConfiguration().slowTaskThreshold();
        this.slowTaskReportDelay = config.restoreJobConfiguration().slowTaskReportDelay();
        this.importer = importer;
        this.rangeDatabaseAccessor = rangeDatabaseAccessor;
        this.restoreJobUtil = restoreJobUtil;
        this.localTokenRangesProvider = localTokenRangesProvider;
        this.metrics = metrics;
    }

    void submit(RestoreRange range) {
        if (this.isClosed) {
            return;
        }
        this.workQueue.offer(range);
    }

    void discardAndRemove(RestoreRange range) {
        if (this.isClosed) {
            return;
        }
        this.workQueue.remove(range);
        range.discard();
        this.pool.runBlocking(() -> this.rangeDatabaseAccessor.updateStatus(range));
    }

    @Override
    public ScheduleDecision scheduleDecision() {
        boolean shouldSkip;
        boolean bl = shouldSkip = !this.sidecarSchema().isInitialized();
        if (shouldSkip) {
            LOGGER.trace("Skipping restore job processing");
            return ScheduleDecision.SKIP;
        }
        return ScheduleDecision.EXECUTE;
    }

    @Override
    public DurationSpec delay() {
        return SecondBoundConfiguration.ONE;
    }

    @Override
    public void execute(Promise<Void> promise) {
        while (this.workQueue.peek() != null && this.processMaxConcurrency.tryAcquire()) {
            RestoreRange range = this.workQueue.poll();
            if (range == null) {
                this.processMaxConcurrency.releasePermit();
                break;
            }
            if (range.hasStaged() && range.job().status != RestoreJobStatus.IMPORT_READY) {
                this.workQueue.offerStaged(range);
                this.processMaxConcurrency.releasePermit();
                break;
            }
            this.workQueue.captureImportQueueLength();
            RestoreRangeHandler task = range.toAsyncTask(this.s3ClientPool, this.pool, this.importer, this.requiredUsableSpacePercentage, this.rangeDatabaseAccessor, this.restoreJobUtil, this.localTokenRangesProvider, this.metrics);
            this.activeTasks.put(task, this.slowTaskThreshold.toSeconds());
            this.pool.executeBlocking(task, false).compose(this.taskSuccessHandler(task), this.taskFailureHandler(range)).onComplete(ignored -> {
                this.processMaxConcurrency.releasePermit();
                this.workQueue.decrementActiveSliceCount(range);
                this.workQueue.captureImportQueueLength();
                this.activeTasks.remove(task);
            });
        }
        this.checkForLongRunningTasks();
        this.workQueue.capturePendingSliceCount();
        promise.tryComplete();
    }

    @Override
    public void close() {
        this.isClosed = true;
        this.s3ClientPool.close();
        this.workQueue.close();
    }

    private void checkForLongRunningTasks() {
        for (RestoreRangeHandler t : this.activeTasks.keySet()) {
            long elapsedInNanos = t.elapsedInNanos();
            if (elapsedInNanos == -1L) continue;
            long elapsedInSeconds = TimeUnit.NANOSECONDS.toSeconds(elapsedInNanos);
            this.activeTasks.computeIfPresent(t, (task, timeToReport) -> {
                if (elapsedInSeconds > timeToReport) {
                    LOGGER.warn("Long-running restore slice task detected. elapsedSeconds={} thresholdSeconds={} sliceKey={} jobId={} status={}", new Object[]{elapsedInSeconds, this.slowTaskThreshold.toSeconds(), task.range().sliceKey(), task.range().jobId(), task.range().job().status});
                    ((Timer)task.range().owner().metrics().restore().slowRestoreTaskTime.metric).update(elapsedInNanos, TimeUnit.NANOSECONDS);
                    return timeToReport + this.slowTaskReportDelay.toSeconds();
                }
                return timeToReport;
            });
        }
    }

    private Function<RestoreRange, Future<Void>> taskSuccessHandler(RestoreRangeHandler task) {
        return range -> {
            InstanceRestoreMetrics restoreMetrics = range.owner().metrics().restore();
            if (range.hasImported()) {
                ((Timer)restoreMetrics.sliceCompletionTime.metric).update(System.nanoTime() - range.sliceCompressedSize(), TimeUnit.NANOSECONDS);
                LOGGER.info("Restore range completes successfully. sliceKey={}", (Object)range.sliceKey());
                range.complete();
            } else if (range.hasStaged()) {
                ((Timer)restoreMetrics.sliceStageTime.metric).update(task.elapsedInNanos(), TimeUnit.NANOSECONDS);
                LOGGER.info("Restore range has been staged successfully. sliceKey={}", (Object)range.sliceKey());
                this.workQueue.offerStaged((RestoreRange)range);
            } else {
                LOGGER.warn("Unexpected state of slice. It is neither staged nor imported. sliceKey={}", (Object)range.sliceKey());
                if (range.hasStaged()) {
                    this.workQueue.offerStaged((RestoreRange)range);
                } else {
                    this.workQueue.offer((RestoreRange)range);
                }
            }
            return Future.succeededFuture();
        };
    }

    private Function<Throwable, Future<Void>> taskFailureHandler(RestoreRange range) {
        return cause -> {
            if (range.isDiscarded()) {
                LOGGER.debug("RestoreRange is discarded. sliceKey={}", (Object)range.sliceKey());
            } else if (cause instanceof RestoreJobException && ((RestoreJobException)cause).retryable()) {
                LOGGER.warn("Slice failed with recoverable failure. sliceKey={}", (Object)range.sliceKey(), cause);
                if (range.hasStaged()) {
                    this.workQueue.offerStaged(range);
                } else {
                    this.workQueue.offer(range);
                }
            } else {
                LOGGER.error("Slice failed with unrecoverable failure. sliceKey={}", (Object)range.sliceKey(), cause);
                range.fail(RestoreJobExceptions.toFatal(cause));
                if (range.job().isManagedBySidecar()) {
                    this.rangeDatabaseAccessor.updateStatus(range);
                }
                this.s3ClientPool.revokeCredentials(range.jobId());
            }
            return Future.succeededFuture();
        };
    }

    @VisibleForTesting
    int activeRanges() {
        return this.workQueue.activeRangesCount();
    }

    @VisibleForTesting
    int activeTasks() {
        return this.activeTasks.size();
    }

    @VisibleForTesting
    int pendingStartRanges() {
        return this.workQueue.size();
    }

    @VisibleForTesting
    SidecarSchema sidecarSchema() {
        return this.sidecarSchema;
    }

    private class WorkQueue {
        private final Queue<RestoreRange> restoreRanges = new ConcurrentLinkedQueue<RestoreRange>();
        private final Queue<RestoreRange> stagedRestoreRanges = new ConcurrentLinkedQueue<RestoreRange>();
        private final Map<Integer, AtomicInteger> pendingRangesPerInstance = new HashMap<Integer, AtomicInteger>();
        private final Map<Integer, AtomicInteger> activeRangesPerInstance = new ConcurrentHashMap<Integer, AtomicInteger>();

        private WorkQueue() {
        }

        synchronized boolean offer(RestoreRange range) {
            this.increment(this.pendingRangesPerInstance, range);
            return this.restoreRanges.offer(range);
        }

        synchronized boolean offerStaged(RestoreRange range) {
            this.increment(this.pendingRangesPerInstance, range);
            return this.stagedRestoreRanges.offer(range);
        }

        synchronized void remove(RestoreRange range) {
            this.decrementIfPresent(this.pendingRangesPerInstance, range);
            this.restoreRanges.remove(range);
        }

        synchronized RestoreRange poll() {
            RestoreRange slice = this.pollInternal();
            if (slice == null) {
                return null;
            }
            this.decrementIfPresent(this.pendingRangesPerInstance, slice);
            this.increment(this.activeRangesPerInstance, slice);
            return slice;
        }

        synchronized void close() {
            for (RestoreRange range : this.restoreRanges) {
                range.cancel();
                LOGGER.debug("Cancelled restore ranges on closing. jobId={} sliceId={} startToken={} endToken={}", new Object[]{range.jobId(), range.sliceId(), range.startToken(), range.endToken()});
            }
            this.restoreRanges.clear();
            this.pendingRangesPerInstance.clear();
            this.activeRangesPerInstance.clear();
        }

        void decrementActiveSliceCount(RestoreRange range) {
            this.decrementIfPresent(this.activeRangesPerInstance, range);
        }

        void captureImportQueueLength() {
            this.activeRangesPerInstance.forEach((instanceId, counter) -> ((DefaultSettableGauge)this.instanceRestoreMetrics((int)instanceId.intValue()).sliceImportQueueLength.metric).setValue((Object)counter.get()));
        }

        void capturePendingSliceCount() {
            this.pendingRangesPerInstance.forEach((instanceId, counter) -> ((DefaultSettableGauge)this.instanceRestoreMetrics((int)instanceId.intValue()).pendingSliceCount.metric).setValue((Object)counter.get()));
        }

        private void increment(Map<Integer, AtomicInteger> map, RestoreRange range) {
            map.compute(range.owner().id(), (key, counter) -> {
                if (counter == null) {
                    counter = new AtomicInteger();
                }
                counter.incrementAndGet();
                return counter;
            });
        }

        private void decrementIfPresent(Map<Integer, AtomicInteger> map, RestoreRange range) {
            map.computeIfPresent(range.owner().id(), (key, counter) -> {
                if (counter.get() < 0) {
                    LOGGER.warn("Slice counter dropped below 0. sliceKey={}", (Object)range.sliceKey(), (Object)new IllegalStateException("Unexpected slice counter state"));
                    counter.set(0);
                    return counter;
                }
                counter.decrementAndGet();
                return counter;
            });
        }

        private RestoreRange peek() {
            RestoreRange range = this.restoreRanges.peek();
            if (range == null) {
                range = this.stagedRestoreRanges.peek();
            }
            return range;
        }

        private RestoreRange pollInternal() {
            RestoreRange range = this.restoreRanges.poll();
            if (range == null) {
                range = this.stagedRestoreRanges.poll();
            }
            return range;
        }

        private InstanceRestoreMetrics instanceRestoreMetrics(int instanceId) {
            return RestoreProcessor.this.metrics.instance(instanceId).restore();
        }

        @VisibleForTesting
        int size() {
            return this.restoreRanges.size();
        }

        @VisibleForTesting
        int activeRangesCount() {
            return this.activeRangesPerInstance.values().stream().mapToInt(AtomicInteger::get).sum();
        }
    }
}

