/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.examples.core.config.store;

import static org.apache.juneau.commons.utils.Utils.*;

import java.util.*;
import java.util.concurrent.*;

import org.apache.juneau.commons.utils.*;
import org.apache.juneau.config.store.*;

/**
 * Example of a {@link ConfigStore} that uses a relational database as a backend.
 *
 * <p>
 * This class provides a basic framework but requires implementing the following methods:
 * <ul class='javatree'>
 * 	<li class='jm'>{@link #getDatabaseValue(String)}
 * 	<li class='jm'>{@link #exists(String)}
 * </ul>
 *
 */
@SuppressWarnings({ "resource", "unused" })
public class SqlStore extends ConfigStore {
	/**
	 * Builder class.
	 */
	public static class Builder extends ConfigStore.Builder {

		String jdbcUrl, tableName, nameColumn, valueColumn;
		int pollInterval;

		Builder() {
			this.jdbcUrl = env(SQLSTORE_jdbcUrl, "jdbc:derby:mydb");
			this.tableName = env(SQLSTORE_tableName, "config");
			this.nameColumn = env(SQLSTORE_nameColumn, "name");
			this.valueColumn = env(SQLSTORE_valueColumn, "value");
			this.pollInterval = env(SQLSTORE_pollInterval, 600);  // Time in seconds.
		}

		@Override
		public SqlStore build() {
			return build(SqlStore.class);
		}

		@Override
		public Builder copy() {
			return null;
		}

		/**
		 * Sets the JDBC URL for the database connection.
		 *
		 * @param value The JDBC URL.
		 * @return This object.
		 */
		public Builder jdbcUrl(String value) {
			jdbcUrl = value;
			return this;
		}

		/**
		 * Sets the name of the column containing configuration entry names.
		 *
		 * @param value The column name.
		 * @return This object.
		 */
		public Builder nameColumn(String value) {
			nameColumn = value;
			return this;
		}

		/**
		 * Sets the polling interval in seconds for checking database changes.
		 *
		 * @param value The polling interval in seconds.
		 * @return This object.
		 */
		public Builder pollInterval(int value) {
			pollInterval = value;
			return this;
		}

		/**
		 * Sets the name of the database table containing configuration entries.
		 *
		 * @param value The table name.
		 * @return This object.
		 */
		public Builder tableName(String value) {
			tableName = value;
			return this;
		}

		/**
		 * Sets the name of the column containing configuration entry values.
		 *
		 * @param value The column name.
		 * @return This object.
		 */
		public Builder valueColumn(String value) {
			valueColumn = value;
			return this;
		}
	}

	static final String
		SQLSTORE_jdbcUrl = "SqlStore.jdbcUrl",
		SQLSTORE_tableName = "SqlStore.tableName",
		SQLSTORE_nameColumn = "SqlStore.nameColumn",
		SQLSTORE_valueColumn = "SqlStore.valueColumn",
		SQLSTORE_pollInterval = "SqlStore.pollInterval";

	/**
	 * Instantiates a builder for this object.
	 *
	 * @return A new builder for this object.
	 */
	public static Builder create() {
		return new Builder();
	}

	private final String jdbcUrl;
	private final String tableName, nameColumn, valueColumn;
	private final Timer watcher;
	private final ConcurrentHashMap<String,String> cache = new ConcurrentHashMap<>();

	/**
	 * Constructor.
	 *
	 * @param builder The builder for this object.
	 */
	protected SqlStore(Builder builder) {
		super(builder);
		this.jdbcUrl = builder.jdbcUrl;
		this.tableName = builder.tableName;
		this.nameColumn = builder.nameColumn;
		this.valueColumn = builder.valueColumn;

		var pollInterval = builder.pollInterval;

		var timerTask = new TimerTask() {
			@Override
			public void run() {
				SqlStore.this.poll();
			}
		};

		this.watcher = new Timer("MyTimer");
		watcher.scheduleAtFixedRate(timerTask, 0, pollInterval * 1000);
	}

	@Override /* Closeable */
	public synchronized void close() {
		if (watcher != null)
			watcher.cancel();
	}

	@Override /* ConfigStore */
	public boolean exists(String name) {
		// Implement me!
		return false;
	}

	@Override /* ConfigStore */
	public synchronized String read(String name) {
		var contents = cache.get(name);
		if (contents == null) {
			contents = getDatabaseValue(name);
			update(name, contents);
		}
		return contents;
	}

	@Override /* ConfigStore */
	public synchronized SqlStore update(String name, String newContents) {
		cache.put(name, newContents);
		super.update(name, newContents);  // Trigger any listeners.
		return this;
	}

	@Override /* ConfigStore */
	public synchronized String write(String name, String expectedContents, String newContents) {

		// This is a no-op.
		if (eq(expectedContents, newContents))
			return null;

		var currentContents = read(name);

		if (expectedContents != null && neq(currentContents, expectedContents))
			return currentContents;

		update(name, newContents);

		// Success!
		return null;
	}

	// Reads the value from the database.
	protected String getDatabaseValue(String name) {
		// Implement me!
		return null;
	}

	synchronized void poll() {

		// Loop through all our entries and find the latest values.
		cache.forEach((name, cacheContents) -> {
			var newContents = getDatabaseValue(name);

			// Change detected!
			if (! cacheContents.equals(newContents))
				update(name, newContents);
		});
	}
}